/*
 * Command-line interface to the watch_dir function provided in watch.c.
 *
 * Copyright 2014, 2021, 2023, 2025-2026 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

#include "config.h"
#include "common.h"
#include "watch.h"

#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <sys/utsname.h>

/* List of command line arguments after options. */
static
 /*@null@ */
 /*@only@ */
char **cli_args = NULL;
static size_t cli_args_count = 0;


/*
 * Output program usage information.
 */
static void usage(void)
{
	struct parameterDefinition parameterDefinitions[] = {
		{ "-i", "--dump-interval", N_("SEC"),
		 N_("interval between writing change files"),
		 { 0, 0, 0, 0} },
		{ "-0", "--null", NULL,
		 N_("delimit changes with nulls, not newlines"),
		 { 0, 0, 0, 0} },
		{ "-a", "--absolute", NULL,
		 N_("list absolute paths, not relative paths"),
		 { 0, 0, 0, 0} },
		{ "-F", "--only-files", NULL,
		 N_("only list changes to files"),
		 { 0, 0, 0, 0} },
		{ "-D", "--only-directories", NULL,
		 N_("only list changes to directories"),
		 { 0, 0, 0, 0} },
		{ "-n", "--no-file-tracking", NULL,
		 N_("don't track filenames, sizes, or times"),
		 { 0, 0, 0, 0} },
		{ "-s", "--suffix", N_("SUFFIX"),
		 N_("append SUFFIX to change file names"),
		 { 0, 0, 0, 0} },
		{ "-o", "--output-dir", N_("OUTPUTDIR"),
		 N_("write change files to OUTPUTDIR"),
		 { 0, 0, 0, 0} },
		{ "-f", "--full-scan-interval", N_("SEC"),
		 N_("do a full rescan every SEC seconds"),
		 { 0, 0, 0, 0} },
		{ "-e", "--exclude", N_("PATTERN"),
		 N_("add to the list of glob patterns to exclude"),
		 { 0, 0, 0, 0} },
		{ "-r", "--recursion-depth", N_("NUM"),
		 N_("maximum directory depth to descend"),
		 { 0, 0, 0, 0} },
		{ "-q", "--queue-run-interval", N_("SEC"),
		 N_("inotify queue processing interval"),
		 { 0, 0, 0, 0} },
		{ "-m", "--queue-run-max", N_("SEC"),
		 N_("maximum time to spend processing the queue"),
		 { 0, 0, 0, 0} },
		{ "-x", "--exchange-dir", N_("DIR"),
		 N_("use a directory to communicate with other instances"),
		 { 0, 0, 0, 0} },
		{ "-t", "--exchange-interval", N_("SEC"),
		 N_("use the exchange directory every SEC seconds"),
		 { 0, 0, 0, 0} },
		{ "-I", "--identity", N_("NAME"),
		 N_("override the name of this instance"),
		 { 0, 0, 0, 0} },
		{ "-M", "--metrics", N_("FILE"),
		 N_("write metrics to a file"),
		 { 0, 0, 0, 0} },
		{ "", NULL, NULL, NULL, { 0, 0, 0, 0} },
		{ "-h", "--help", NULL,
		 N_("show this help and exit"),
		 { 0, 0, 0, 0} },
		{ "-V", "--version", NULL,
		 N_("show version information and exit"),
		 { 0, 0, 0, 0} },
#ifdef ENABLE_DEBUGGING
		{ "-d", "--debug", NULL,
		 N_("enable debugging"),
		 { 0, 0, 0, 0} },
#endif
		{ NULL, NULL, NULL, NULL, { 0, 0, 0, 0} }
	};
	size_t terminalColumns = 77;
	const char *programDescription;
	const char *usageNote;
	const char *bugReportNote;

	(void) readTerminalSize(stdout, &terminalColumns, NULL);
	if (terminalColumns < 6)
		terminalColumns = 6;

	/*@-mustfreefresh@ */
	/*
	 * splint note: the gettext calls made by _() cause memory leak
	 * warnings, but in this case it's unavoidable, and mitigated by the
	 * fact we only translate each string once.
	 */
	printf("%s: %s %s\n", _("Usage"), common_program_name,
	       _("[OPTION...] -o OUTPUTDIR DIRECTORY... | DIRECTORY OUTPUTDIR"));

	programDescription =
	    _
	    ("Watch DIRECTORY for changes, dumping the changed paths to a unique file in the OUTPUTDIR directory every few seconds.");
	if (NULL != programDescription) {
		outputWordWrap(stdout, programDescription, terminalColumns - 1, 0);
		printf("\n");
	}

	printf("\n");
	showParameterDefinitions(stdout, terminalColumns, parameterDefinitions);
	printf("\n");

	usageNote = _("The OUTPUTDIR must not be under the DIRECTORY being watched.");
	if (NULL != usageNote) {
		outputWordWrap(stdout, usageNote, terminalColumns - 1, 0);
		printf("\n");
	}
	printf("\n");

	bugReportNote = _("Please report any bugs to:");
	if (NULL != bugReportNote) {
		outputWordWrap(stdout, bugReportNote, terminalColumns - 1, 0);
		printf(" %s\n", PACKAGE_BUGREPORT);
	}
}

/*@+mustfreefresh@ */


/*
 * Parse the command line arguments.  Returns 0 on success, -1 if the
 * program should exit immediately without an error, or 1 if the program
 * should exit with an error.
 */
/*@-mustfreefresh@ */
/* see gettext note earlier. */
static int parse_options(int argc, char **argv, struct watch_dir_params_s *params)
{
	/*@-nullassign@ */
	/* splint rationale: NULL is allowed for "flags" in long options. */
	struct option long_options[] = {
		{ "help", 0, NULL, (int) 'h' },
		{ "version", 0, NULL, (int) 'V' },
		{ "full-scan-interval", 1, NULL, (int) 'f' },
		{ "full", 1, NULL, (int) 'f' },
		{ "exclude", 1, NULL, (int) 'e' },
		{ "recursion-depth", 1, NULL, (int) 'r' },
		{ "queue-run-interval", 1, NULL, (int) 'q' },
		{ "queue", 1, NULL, (int) 'q' },
		{ "queue-run-max", 1, NULL, (int) 'm' },
		{ "max", 1, NULL, (int) 'm' },
		{ "dump-interval", 1, NULL, (int) 'i' },
		{ "interval", 1, NULL, (int) 'i' },
		{ "null", 0, NULL, (int) '0' },
		{ "absolute", 0, NULL, (int) 'a' },
		{ "only-files", 0, NULL, (int) 'F' },
		{ "only-directories", 0, NULL, (int) 'D' },
		{ "no-file-tracking", 0, NULL, (int) 'n' },
		{ "suffix", 1, NULL, (int) 's' },
		{ "depth", 1, NULL, (int) 'r' },
		{ "output-dir", 1, NULL, (int) 'o' },
		{ "output", 1, NULL, (int) 'o' },
		{ "exchange-dir", 1, NULL, (int) 'x' },
		{ "exchange-interval", 1, NULL, (int) 't' },
		{ "identity", 1, NULL, (int) 'I' },
		{ "metrics", 1, NULL, (int) 'M' },
		{ "debug", 0, NULL, (int) 'd' },
		{ NULL, 0, NULL, 0 }
	};
	/*@+nullassign@ */
	int option_index = 0;
	char *short_options = "hVf:e:r:q:m:i:0aFDns:o:x:t:I:M:d";
	char *copy;
	int c;
	unsigned long param;

	cli_args_count = 0;
	if (NULL == cli_args)
		cli_args = calloc((size_t) (argc + 1), sizeof(char *));
	if (NULL == cli_args) {
		die("%s", strerror(errno));
		return 1;
	}

	do {
		c = getopt_long(argc, argv, short_options, long_options, &option_index);	/* flawfinder: ignore */
		if (c < 0)
			continue;

		/*
		 * flawfinder rationale: as we are passing argv here, it is
		 * not practical for us to try to limit the size of all
		 * user-supplied arguments - this would likely be more risky
		 * than just trusting the local getopt_long() implementation
		 * not to overflow.
		 */

		/*
		 * Parse each command line option.
		 */
		switch (c) {
		case 'h':
			usage();
			free(cli_args);
			cli_args = NULL;
			cli_args_count = 0;
			return -1;
		case 'V':
			/* GNU standard first line format: program (package) and version only */
			printf("%s (%s) %s\n", common_program_name, PACKAGE_NAME, PACKAGE_VERSION);
			/* GNU standard second line format - "Copyright" always in English */
			printf("Copyright %s %s\n", COPYRIGHT_YEAR, COPYRIGHT_HOLDER);
			/* GNU standard license line and free software notice */
			printf("%s\n", _("License: GPLv3+ <https://www.gnu.org/licenses/gpl-3.0.html>"));
			printf("%s\n", _("This is free software: you are free to change and redistribute it."));
			printf("%s\n", _("There is NO WARRANTY, to the extent permitted by law."));
			/* Project web site link */
			printf("\n%s: <%s>\n", _("Project web site"), PACKAGE_URL);
			free(cli_args);
			cli_args = NULL;
			cli_args_count = 0;
			return -1;
		case 'd':
#if ENABLE_DEBUGGING
			debugging_enabled = true;
			break;
#else
			error("%s", _("Debugging is not enabled in this build."));
			free(cli_args);
			cli_args = NULL;
			cli_args_count = 0;
			return 1;
#endif
		case 'e':
			if (params->exclude_count >= (MAX_EXCLUDES - 1)) {
				error("%s", _("too many exclusions specified"));
				free(cli_args);
				cli_args = NULL;
				cli_args_count = 0;
				return 1;
			}
			copy = xstrdup(optarg);
			if (NULL == copy) {
				return 1;
			} else {
				params->excludes[params->exclude_count++] = copy;
			}
			break;
		case '0':
			params->null_delimiter = true;
			break;
		case 'a':
			params->absolute_paths = true;
			break;
		case 'F':
			params->only_list_files = true;
			break;
		case 'D':
			params->only_list_directories = true;
			break;
		case 'n':
			params->no_file_tracking = true;
			break;
		case 's':
			copy = xstrdup(optarg);
			if (NULL == copy) {
				return 1;
			} else {
				if (NULL != params->changefile_suffix) {
					free(params->changefile_suffix);
				}
				params->changefile_suffix = copy;
			}
			break;
		case 'o':
			/*@-unrecog@ *//* splint doesn't know about realpath() */
			copy = realpath(optarg, NULL);	/* flawfinder: ignore (see below) */
			/*@+unrecog@ */
			if (NULL == copy) {
				fprintf(stderr, "%s: %s: %s\n", common_program_name, optarg, strerror(errno));
				exit(EXIT_FAILURE);
			}
			if (NULL != params->changedpath_dir) {
				free((void *) (params->changedpath_dir));
			}
			params->changedpath_dir = copy;
			break;
		case 'x':
			/*@-unrecog@ *//* splint doesn't know about realpath() */
			copy = realpath(optarg, NULL);	/* flawfinder: ignore (see below) */
			/*@+unrecog@ */
			if (NULL == copy) {
				fprintf(stderr, "%s: %s: %s\n", common_program_name, optarg, strerror(errno));
				exit(EXIT_FAILURE);
			}
			if (NULL != params->exchange_dir) {
				free((void *) (params->exchange_dir));
			}
			params->exchange_dir = copy;
			break;
		case 'I':
			copy = xstrdup(optarg);
			if (NULL == copy) {
				return 1;
			} else {
				if (NULL != params->exchange_identity) {
					free((void *) (params->exchange_identity));
				}
				params->exchange_identity = copy;
			}
			break;
		case 'M':
			copy = xstrdup(optarg);
			if (NULL == copy) {
				return 1;
			} else {
				if (NULL != params->metrics_file) {
					free((void *) (params->metrics_file));
				}
				params->metrics_file = copy;
			}
			break;
		case 'f':
		case 'r':
		case 'q':
		case 'm':
		case 'i':
		case 't':
			errno = 0;
			param = strtoul(optarg, NULL, 10);
			if (0 != errno) {
				error("-%c: %s", c, strerror(errno));
				free(cli_args);
				cli_args = NULL;
				cli_args_count = 0;
				return 1;
			}
			switch (c) {
			case 'f':
				params->full_scan_interval = param;
				break;
			case 'r':
				params->max_dir_depth = (unsigned int) param;
				break;
			case 'q':
				params->queue_run_interval = param;
				break;
			case 'm':
				params->queue_run_max_seconds = param;
				break;
			case 'i':
				params->changedpath_dump_interval = param;
				break;
			case 't':
				params->exchange_interval = param;
				break;
			}
			break;
		default:
			/*@-formatconst@ */
			fprintf(stderr, _("Try `%s --help' for more information."), common_program_name);
			/*@+formatconst@ */
			/*
			 * splint note: formatconst is warning about the use
			 * of a non constant (translatable) format string;
			 * this is unavoidable here and the only attack
			 * vector is through the message catalogue.
			 */
			fprintf(stderr, "\n");
			free(cli_args);
			cli_args = NULL;
			cli_args_count = 0;
			return 1;
		}

	} while (c != -1);

	/*
	 * Store remaining command-line arguments.
	 */
	while ((NULL != cli_args) && (optind < argc)) {
		cli_args[cli_args_count++] = argv[optind++];
	}

	if (((NULL == params->changedpath_dir) && (cli_args_count != 2))
	    || ((NULL != params->changedpath_dir) && (cli_args_count < 1))
	    ) {
		/*
		 * If "-o" wasn't used, we have to have 2 arguments - the
		 * one directory to watch, and the output directory.
		 *
		 * If "-o" was used, we need at least 1 argument, providing
		 * at least 1 directory to watch.
		 */
		usage();
		free(cli_args);
		cli_args = NULL;
		cli_args_count = 0;
		return 1;
	}

	if ((NULL != params->exchange_dir) && (NULL == params->exchange_identity)) {
		/*
		 * If "-x" was used but "-I" was not, take the system
		 * nodename as a default identity.
		 */
		struct utsname info;

		memset(&info, 0, sizeof(info));
		(void) uname(&info);

		copy = xstrdup(info.nodename);
		if (NULL == copy) {
			return 1;
		} else {
			if (NULL != params->exchange_identity) {
				/* for splint; this should optimise out. */
				free((void *) (params->exchange_identity));
			}
			params->exchange_identity = copy;
		}
	}

	if ((NULL != params->exchange_dir) && (NULL != params->exchange_identity)
	    && !validate_exchange_identity(params->exchange_identity)) {
		fprintf(stderr, "%s: %s: %s\n", common_program_name, params->exchange_identity,
			_("not a valid exchange identity"));
		return 1;
	}

	return 0;
}

/*@+mustfreefresh@ */


/*
 * Command line entry point: parse command line arguments and start the main
 * watch_dir loop.
 */
/*@-mustfreeonly @*/
/*@-kepttrans @*/
/*@-compdestroy @*/
int main(int argc, char **argv)
{
	struct watch_dir_params_s params;
	int rc;
	size_t eidx, path_idx;

#ifdef ENABLE_NLS
	/* Initialise language translation. */
	(void) setlocale(LC_ALL, "");
	(void) bindtextdomain(PACKAGE, LOCALEDIR);
	(void) textdomain(PACKAGE);
#endif

	/* Don't use argv[0], use canonical program name. */
	common_program_name = "watchdir";

	memset(&params, 0, sizeof(params));

	/* Set the default watch parameters. */
	params.full_scan_interval = 7200;
	params.queue_run_interval = 2;
	params.queue_run_max_seconds = 5;
	params.changedpath_dump_interval = 30;
	params.exchange_interval = 1;
	params.max_dir_depth = 20;
	params.null_delimiter = false;
	params.absolute_paths = false;
	params.only_list_files = false;
	params.only_list_directories = false;

	rc = parse_options(argc, argv, &params);
	if (rc < 0)
		return EXIT_SUCCESS;
	else if (rc > 0)
		return EXIT_FAILURE;

	if (NULL == cli_args)
		return EXIT_FAILURE;

	if (NULL == params.changedpath_dir) {
		/*
		 * If params.changedpath_dir is NULL, then the first CLI
		 * argument is the only toplevel path, and the second
		 * argument is the changed paths dir.
		 */

		char *toplevel_path = NULL;
		char *changedpath_dir = NULL;

		/*@-unrecog@ *//* splint doesn't know about realpath() */
		toplevel_path = realpath(cli_args[0], NULL);	/* flawfinder: ignore */
		/*@+unrecog@ */
		if (NULL == toplevel_path) {
			fprintf(stderr, "%s: %s: %s\n", common_program_name, cli_args[0], strerror(errno));
			exit(EXIT_FAILURE);
		}

		/*@-unrecog@ */
		/* splint doesn't know about realpath() */
		changedpath_dir = realpath(cli_args[1], NULL);	/* flawfinder: ignore */
		/*@+unrecog@ */
		if (NULL == changedpath_dir) {
			fprintf(stderr, "%s: %s: %s\n", common_program_name, cli_args[1], strerror(errno));
			free(toplevel_path);
			exit(EXIT_FAILURE);
		}
		params.changedpath_dir = changedpath_dir;

		params.toplevel_paths = calloc(1, sizeof(toplevel_path_t));
		if (NULL == params.toplevel_paths) {
			fprintf(stderr, "%s: %s\n", common_program_name, strerror(errno));
			free(changedpath_dir);
			free(toplevel_path);
			exit(EXIT_FAILURE);
		}

		params.toplevel_paths[0] = toplevel_path;
		params.toplevel_path_count = 1;

	} else {
		/*
		 * params.changedpath_dir is not NULL, meaning "-o" was
		 * passed, so all remaining CLI arguments are top-level
		 * paths.
		 */
		params.toplevel_paths = calloc(cli_args_count, sizeof(toplevel_path_t));
		if (NULL == params.toplevel_paths) {
			fprintf(stderr, "%s: %s\n", common_program_name, strerror(errno));
			exit(EXIT_FAILURE);
		}

		for (path_idx = 0; path_idx < cli_args_count; path_idx++) {
			char *toplevel_path;
			/*@-unrecog@ */
			/* splint doesn't know about realpath() */
			toplevel_path = realpath(cli_args[path_idx], NULL);	/* flawfinder: ignore */
			/*@+unrecog@ */
			if (NULL == toplevel_path) {
				fprintf(stderr, "%s: %s: %s\n", common_program_name, cli_args[path_idx],
					strerror(errno));
				exit(EXIT_FAILURE);
			}
			params.toplevel_paths[path_idx] = toplevel_path;
		}
		params.toplevel_path_count = cli_args_count;
	}


	/*
	 * flawfinder rationale: we pass NULL as the destination buffer so
	 * that realpath() will allocate it for us.  The manual says the
	 * buffer it allocates will be up to PATH_MAX long, which should be
	 * sufficient to avoid overflows.
	 */

	/*
	 * Switch on absolute paths in the change files if more than one
	 * top-level directory is being watched, since otherwise the
	 * relative paths are meaningless.
	 */
	if (params.toplevel_path_count > 1)
		params.absolute_paths = true;

	rc = watch_dir(&params);

	for (path_idx = 0; path_idx < params.toplevel_path_count; path_idx++) {
		free((void *) (params.toplevel_paths[path_idx]));
		params.toplevel_paths[path_idx] = NULL;
	}

	free(params.toplevel_paths);
	params.toplevel_paths = NULL;
	free((void *) (params.changedpath_dir));
	params.changedpath_dir = NULL;

	free((void *) (params.exchange_dir));
	params.exchange_dir = NULL;
	free((void *) (params.exchange_identity));
	params.exchange_identity = NULL;

	free((void *) (params.metrics_file));
	params.metrics_file = NULL;

	for (eidx = 0; eidx < params.exclude_count; eidx++) {
		if (NULL != params.excludes[eidx])
			free(params.excludes[eidx]);
	}
	if (NULL != cli_args)
		free(cli_args);
	cli_args = NULL;

	if (NULL != params.changefile_suffix) {
		free(params.changefile_suffix);
		params.changefile_suffix = NULL;
	}

	return rc;
}

/*@+compdestroy @*/
/*@+kepttrans @*/
/*@+mustfreeonly @*/
/*
 * splint - some allocated strings are moved around to allow them to be
 * marked const in the struct, and the exclusions list is also tricky to
 * handle, so these warnings are silenced for now.
 */
