/*
 * Watch the given directory, producing output files containing lists of
 * files changed.
 *
 * Copyright 2014, 2021, 2023, 2025-2026 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

/* File and subdirectory array allocation chunk size */
#define DIRCONTENTS_ALLOC_CHUNK	4

/* Directory index array allocation chunk size */
#define DIR_INDEX_ALLOC_CHUNK 1024

/* Change queue allocation chunk size */
#define CHANGE_QUEUE_ALLOC_CHUNK 1024

/* Changed paths list allocation chunk size */
#define CHANGEDPATH_ALLOC_CHUNK 1024

#define _ATFILE_SOURCE

#include "config.h"
#include "common.h"
#include "watch.h"

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdint.h>
#include <inttypes.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <search.h>
#ifndef SPLINT
/* splint 3.1.2 chokes on syslog.h */
#include <syslog.h>
#endif
#include <limits.h>
#include <sys/inotify.h>
#include <sys/select.h>
#include <poll.h>
#include <fnmatch.h>
#include <assert.h>


/*
 * Actions to take on inotify events.
 */
typedef enum {
	IN_ACTION_CREATE,
	IN_ACTION_UPDATE,
	IN_ACTION_DELETE,
	IN_ACTION_NONE
} inotify_action_t;

struct ds_file_s;
typedef struct ds_file_s *ds_file_t;
struct ds_dir_s;
typedef struct ds_dir_s *ds_dir_t;
struct ds_topdir_s;
typedef struct ds_topdir_s *ds_topdir_t;
struct ds_watch_index_s;
typedef struct ds_watch_index_s *ds_watch_index_t;
struct ds_change_queue_s;
typedef struct ds_change_queue_s *ds_change_queue_t;

typedef /*@null@ */ void *nullable_pointer;


/*
 * Counters for writing to the metrics file.
 */
static struct {
	uint64_t directories_watching;	 /* active inotify watches */
	uint64_t change_files_output;	 /* change files completed */
	uint64_t changed_paths_output;	 /* paths written to change files */
	uint64_t exchange_files_output;	 /* exchange files completed */
	uint64_t exchange_paths_output;	 /* paths written to exchange files */
	uint64_t exchange_files_input;	 /* exchange files read */
	uint64_t exchange_paths_input;	 /* paths read from exchange files */
	uint64_t watch_index_length;	 /* sum of topdir watch index lengths */
	uint64_t change_queue_length;	 /* sum of topdir change queue lengths */
} watch_dir_metrics;

/*
 * Structure holding information about a file.
 */
struct ds_file_s {
	nullable_string leaf;		 /* leafname of this file */
	time_t mtime;			 /* file last-modification time */
	off_t size;			 /* file size */
	/*@null@ *//*@dependent@ */ ds_dir_t parent;
	/* containing directory */
	uint8_t leaf_length;		 /* length of the leafname */
	bool seen_in_rescan;		 /* set during dir rescan */
};

/*
 * Structure holding information about a directory.  The top-level directory
 * has an empty leaf.
 */
struct ds_dir_s {
	nullable_string leaf;		 /* leafname of this directory */
	int wd;				 /* inotify watch fd (if dir) */
	size_t file_count;		 /* number of files in directory */
	size_t subdir_count;		 /* number of immediate subdirs */
	/*@null@ */
	ds_file_t *files;		 /* array of files */
	/*@null@ */
	ds_dir_t *subdirs;		 /* array of subdirectories */
	nullable_pointer tree_files;	 /* binary tree of files */
	nullable_pointer tree_subdirs;	 /* binary tree of subdirs */
	/*@null@ *//*@dependent@ */
	ds_dir_t parent;		 /* pointer to parent directory */
	/*@null@ *//*@dependent@ */
	ds_topdir_t topdir;		 /* pointer to top directory */
	uint8_t leaf_length;		 /* length of the leafname */
	/* Booleans, at the end to minimise wasted space due to padding. */
	bool files_unsorted;		 /* set if files array needs sorting */
	bool subdirs_unsorted;		 /* set if subdirs array needs sorting */
	bool seen_in_rescan;		 /* set during dir rescan */
	bool removed;			 /* set when directory is removed */
};


/*
 * Structure holding information about a top-level directory.
 */
struct ds_topdir_s {
	nullable_string path;		 /* absolute path of the directory */
	size_t path_length;		 /* length of the path string */
	struct ds_dir_s dir;		 /* the directory's details */
	/*
	 * The watch_index array maps each inotify watch descriptor to a
	 * directory pointer. Each pointer is an alias of the original.
	 */
	/*@null@ */
	ds_watch_index_t watch_index;	 /* array of watch descriptors */
	size_t watch_index_length;	 /* number of entries in array */
	/*
	 * The change_queue array is populated when inotify events are
	 * received, and contains the file or directory pointers associated
	 * with each event.  Processing the change queue means checking each
	 * of these items to see whether they really have changed, in which
	 * case their path gets added to the changed_paths, and for
	 * directories, they are rescanned.
	 *
	 * These file and dir pointers are aliases of the originals, so
	 * before file or dir structures are freed, their pointers must be
	 * removed from this array as well.
	 *
	 * The tree_change_queue pointer is a binary tree (tfind() etc)
	 * containing aliases of the path string pointers from change queue
	 * array entries.  It is used to check for duplicates.
	 */
	/*@null@ */
	ds_change_queue_t change_queue;	 /* array of changes needed */
	size_t change_queue_length;	 /* number of changes in queue */
	nullable_pointer tree_change_queue;	/* tree of change queue paths */
	/*
	 * The changed_paths array contains the paths that are known to have
	 * changed, as allocated strings (copies, not aliases of the file or
	 * dir path pointers).  It is populated by change queue processing
	 * and emptied by dump_changed_paths().
	 *
	 * The tree_changed_paths pointer is a binary tree (tfind() etc)
	 * containing aliases of the string pointers from the changed_paths
	 * array.  It is used to check for duplicates.
	 */
	/*@null@ */
	char **changed_paths;		 /* array of changed paths */
	size_t changed_paths_length;	 /* number of paths in array */
	nullable_pointer tree_changed_paths;	/* binary tree of changed paths */
	/* Integers, at the end to avoid padding if they're smaller than a pointer. */
	int fd_inotify;			 /* directory watch file descriptor */
	int highest_wd;			 /* highest watch descriptor value */
	/* Booleans, at the end to minimise wasted space due to padding. */
	bool watch_index_unsorted;	 /* set if watch index array needs sorting */
};


/*
 * Structure for indexing directory structures by watch identifier.
 */
struct ds_watch_index_s {
	int wd;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


/*
 * Structure for a file check or directory scan queue entry.
 */
struct ds_change_queue_s {
	time_t when;
	/*@null@ */ char *path;
	/*@null@ *//*@dependent@ */ ds_file_t file;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


static int ds_filename_valid(const char *name);

static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add( /*@dependent@ */ ds_dir_t dir, const char *name);
static void ds_file_remove(ds_file_t file);
static int ds_file_checkchanged(ds_file_t file);

static /*@null@ */ ds_topdir_t ds_dir_toplevel(int fd_inotify, const char *top_path);
/*@dependent@ */
static /*@null@ */ ds_dir_t ds_dir_add( /*@dependent@ */ ds_dir_t dir, const char *name);
static void ds_dir_remove(ds_dir_t dir);
static bool ds_dir_scan( /*@dependent@ */ ds_dir_t dir, bool defer_subdir_scan);

static ssize_t ds_dir_file_lookup(ds_dir_t dir, ds_file_t file);
static ssize_t ds_dir_subdir_lookup(ds_dir_t dir, ds_dir_t subdir);

static void ds_watch_index_add( /*@dependent@ */ ds_dir_t dir, int wd);
static void ds_watch_index_remove(ds_topdir_t topdir, int wd);
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_topdir_t topdir, int wd, /*@null@ */ size_t *idxptr);

static void ds_change_queue_file_add( /*@dependent@ */ ds_file_t file, time_t when);
static void ds_change_queue_file_remove(ds_file_t file);
static void ds_change_queue_dir_add( /*@dependent@ */ ds_dir_t dir, time_t when);
static void ds_change_queue_dir_remove(ds_dir_t dir);

static void ds_change_queue_process(ds_topdir_t topdir, time_t work_until);

static void mark_file_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_file_t file);
static void mark_dir_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_dir_t dir);
static void dump_changed_paths(ds_topdir_t topdir, const char *changedpath_dir);

static bool extend_array( /*@null@ */ nullable_pointer * array_ptr, size_t array_length, size_t entry_size,
			 size_t entry_alloc_chunk);

static nullable_string _ds_path_buffer = NULL;
static size_t _ds_path_bufsize = 0;

static struct watch_dir_params_s *watch_dir_params;
static
 /*@null@ */
 /*@only@ */
ds_topdir_t *topdirs = NULL;
static bool watch_dir_exit_now = false;


/*
 * Dummy function which does nothing, for tdestroy() to call.
 */
static void _tdestroy_noop( /*@unused@ */  __attribute__((unused))
			   void *ptr)
{
}


/*
 * Return a pointer to a string containing the path of the given leaf within
 * the given directory, relative to the top-level directory, or absolute if
 * absolute_path is true.
 *
 * The pointer is only valid until the next call to this function and must
 * not be free()d.
 */
static /*@dependent@ */ char *_ds_path(const /*@null@ */ char *leaf, uint8_t leaf_length, /*@null@ */ ds_dir_t parent,
				       /*@null@ */ ds_topdir_t topdir, bool absolute_path)
{
	size_t path_length, position;

	if (NULL == leaf)
		leaf_length = 0;
	if (NULL == topdir)
		absolute_path = false;

	/* Calculate the length of the path. */

	path_length = leaf_length;
	if (leaf_length > 0) {
		path_length++;		    /* +1 for a preceding '/'. */
	}

	{
		ds_dir_t dir;

		dir = parent;
		while (NULL != dir) {
			if ((dir->leaf_length > 0) && (NULL != dir->leaf)) {
				path_length += 1 + dir->leaf_length;	/* +1 for the '/'. */
			}
			dir = dir->parent;
		}
	}

	if (absolute_path && (NULL != topdir))
		path_length += topdir->path_length;

	/* Ensure the buffer is big enough for the path. */
	/* The buffer needs 1 extra byte for the terminating \0. */
	/*@-branchstate@ */
	if ((1 + path_length) > _ds_path_bufsize) {
		/*@-statictrans@ */
		_ds_path_buffer = realloc(_ds_path_buffer, 1 + path_length);
		/*@+statictrans@ */
		/*@-observertrans@ */
		if (NULL == _ds_path_buffer) {
			die("%s: %s", "realloc", strerror(errno));
			_ds_path_bufsize = 0;
			return "";
		}
		/*@+observertrans@ */
		_ds_path_bufsize = 1 + path_length;
	}
	/*@+branchstate@ */

	assert(NULL != _ds_path_buffer);
	/*@-observertrans@ */
	if (NULL == _ds_path_buffer)
		return "";
	/*@+observertrans@ */

	/* Populate the buffer with the path. */

	memset(_ds_path_buffer, 0, 1 + path_length);
	position = path_length;

	if ((leaf_length > 0) && (NULL != leaf)) {
		if (position < leaf_length + 1) {
			die("%s", "string underflow - no room for the leaf");
			return &(_ds_path_buffer[path_length]);
		}
		position -= leaf_length;
		memcpy(&(_ds_path_buffer[position]), leaf, leaf_length);	/* flawfinder: ignore */
		position--;
		_ds_path_buffer[position] = '/';
	}

	{
		ds_dir_t dir;

		dir = parent;
		while (NULL != dir) {
			if ((dir->leaf_length > 0) && (NULL != dir->leaf)) {
				if (position < dir->leaf_length + 1) {
					die("%s", "string underflow - no room for path component");
					return &(_ds_path_buffer[path_length]);
				}
				position -= dir->leaf_length;
				memcpy(&(_ds_path_buffer[position]), dir->leaf, dir->leaf_length);	/* flawfinder: ignore */
				position--;
				_ds_path_buffer[position] = '/';
			}
			dir = dir->parent;
		}
	}

	if (absolute_path && (NULL != topdir) && (NULL != topdir->path)) {
		if (position < topdir->path_length) {
			die("%s", "string underflow - no room for top-level path");
			return &(_ds_path_buffer[path_length]);
		}
		position -= topdir->path_length;
		memcpy(&(_ds_path_buffer[position]), topdir->path, topdir->path_length);	/* flawfinder: ignore */
	}

	/*
	 * flawfinder rationale: each of the three memcpy() calls is to a
	 * buffer that was allocated to the specific required size and the
	 * destination position is checked to make sure it doesn't
	 * undershoot past the start of the buffer.
	 */

	if (position > 0) {
		die("%s", "path length mismatch");
		return &(_ds_path_buffer[path_length]);
	}

	/* In non-absolute mode, skip the '/' at the start of the path. */
	if ((!absolute_path) && (path_length > 0)) {
		return &(_ds_path_buffer[1]);
	}

	return _ds_path_buffer;
}


/*
 * Return the absolute path of the given file.  The string pointer is only
 * valid until the next ds_file_full_path or ds_dir_full_path (or
 * _ds_path) call and must not be free()d.
 */
static /*@dependent@ */ char *ds_file_full_path(ds_file_t file)
{
	return _ds_path(file->leaf, file->leaf_length, file->parent, NULL == file->parent ? NULL : file->parent->topdir,
			true);
}


/*
 * Return the absolute path of the given directory.  The string pointer is only
 * valid until the next ds_dir_full_path or ds_file_full_path (or
 * _ds_path) call and must not be free()d.
 */
static /*@dependent@ */ char *ds_dir_full_path(ds_dir_t dir)
{
	return _ds_path(dir->leaf, dir->leaf_length, dir->parent, dir->topdir, true);
}


/*
 * Ensure that the given array has room for another item, calling die() and
 * returning false on failure.
 *
 * The array is expanded by entry_alloc_chunk entries at a time.
 *
 * The caller is expected to increment the array length by one immediately
 * after this function returns true.
 */
bool extend_array( /*@null@ */ nullable_pointer *array_ptr, size_t array_length, size_t entry_size,
		  size_t entry_alloc_chunk)
{
	void *old_ptr;
	void *new_ptr;
	size_t new_length;

	if (NULL == array_ptr)
		return false;

	/*
	 * We only need to allocate more space when the current length is a
	 * multiple of the entry allocation chunk count (or zero).
	 */
	if (0 != (array_length % entry_alloc_chunk))
		return true;

	new_length = array_length + entry_alloc_chunk;

	old_ptr = *array_ptr;
	new_ptr = realloc(old_ptr, new_length * entry_size);
	if (NULL == new_ptr) {
		die("%s: %s", "realloc", strerror(errno));
		*array_ptr = NULL;
		return false;
	}

	*array_ptr = new_ptr;
	return true;
}


/*
 * Return how many parent directories a directory has, where a top-level
 * directory has 0.
 */
static unsigned int ds_dir_depth( /*@null@ */ ds_dir_t dir)
{
	unsigned int depth = 0;

	if (NULL == dir)
		return 0;

	while (NULL != dir->parent) {
		dir = dir->parent;
		depth++;
	}

	return depth;
}


/*
 * Add the given watch descriptor to the directory index.
 */
static void ds_watch_index_add( /*@dependent@ */ ds_dir_t dir, int wd)
{
	if (NULL == dir)
		return;
	if (wd < 0)
		return;
	if (NULL == dir->topdir)
		return;

	/*
	 * Extend the array if we need to.
	 */
	if (!extend_array
	    ((void **) (&(dir->topdir->watch_index)), dir->topdir->watch_index_length,
	     sizeof(dir->topdir->watch_index[0]), DIR_INDEX_ALLOC_CHUNK))
		return;
	assert(NULL != dir->topdir->watch_index);
	if (NULL == dir->topdir->watch_index)
		return;

	memset(&(dir->topdir->watch_index[dir->topdir->watch_index_length]), 0, sizeof(dir->topdir->watch_index[0]));

	dir->topdir->watch_index[dir->topdir->watch_index_length].wd = wd;
	dir->topdir->watch_index[dir->topdir->watch_index_length].dir = dir;

	dir->topdir->watch_index_length++;

	/*
	 * Mark the watch index array as unsorted, if the new wd is not
	 * greater than the highest wd seen so far (if it's greater than or
	 * equal to, then the array is still in wd order).
	 */
	if (wd < dir->topdir->highest_wd) {
		dir->topdir->watch_index_unsorted = true;
	} else {
		dir->topdir->highest_wd = wd;
	}
}


/*
 * Remove the given watch descriptor from the directory index.
 */
static void ds_watch_index_remove(ds_topdir_t topdir, int wd)
{
	size_t startidx, readidx, writeidx;

	if (NULL == topdir)
		return;
	if (NULL == topdir->watch_index)
		return;

	startidx = 0;
	(void) ds_watch_index_lookup(topdir, wd, &startidx);

	for (readidx = startidx, writeidx = startidx; readidx < topdir->watch_index_length; readidx++) {
		if (topdir->watch_index[readidx].wd == wd) {
			continue;
		}
		if (readidx != writeidx) {
			topdir->watch_index[writeidx] = topdir->watch_index[readidx];
		}
		writeidx++;
	}
	topdir->watch_index_length = writeidx;
}


/*
 * Comparison function for the directory index.
 */
static int ds_watch_index_compare(const void *a, const void *b)
{
	if (((ds_watch_index_t) a)->wd < ((ds_watch_index_t) b)->wd)
		return -1;
	if (((ds_watch_index_t) a)->wd > ((ds_watch_index_t) b)->wd)
		return 1;
	return 0;
}


/*
 * Return the directory structure associated with the given watch
 * descriptor, or NULL if none.  Populates *idxptr with the associated
 * watch_index array index number, if idxptr is not NULL.
 */
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_topdir_t topdir, int wd, /*@null@ */ size_t *idxptr)
{
	struct ds_watch_index_s key;
	/*@null@ */ ds_watch_index_t result;

	if (NULL == topdir)
		return NULL;
	if (NULL == topdir->watch_index)
		return NULL;

	if ((topdir->watch_index_unsorted)
	    && (topdir->watch_index_length > 0)) {
		debug("%s: %s", NULL == topdir->path ? "(null)" : topdir->path, "sorting watch index");
		qsort(topdir->watch_index, topdir->watch_index_length,
		      sizeof(topdir->watch_index[0]), ds_watch_index_compare);
		topdir->watch_index_unsorted = false;
		topdir->highest_wd = topdir->watch_index[topdir->watch_index_length - 1].wd;
	}

	key.wd = wd;
	debug("%s: %d: %s", NULL == topdir->path ? "(null)" : topdir->path, wd, "searching for watch index");
	result =
	    bsearch(&key, topdir->watch_index, topdir->watch_index_length,
		    sizeof(topdir->watch_index[0]), ds_watch_index_compare);

	if (NULL == result) {
		debug("%s: %d: %s", NULL == topdir->path ? "(null)" : topdir->path, wd, "watch index not found");
		return NULL;
	}

	debug("%s: %d: %s: %p (- %p) = %s", NULL == topdir->path ? "(null)" : topdir->path, wd, "bsearch result",
	      result, topdir->watch_index, NULL == result->dir ? "(null)" : ds_dir_full_path(result->dir));

	if (NULL != idxptr) {
		assert(result >= topdir->watch_index);
		if (result < topdir->watch_index)
			return result->dir;
		/*
		 * Subtracting the pointers gives us the array index since
		 * the pointers are of type ds_watch_index_t.
		 */
		*idxptr = (size_t) (result - topdir->watch_index);
	}

	return result->dir;
}


/*
 * Comparison function that compares two pointers to ds_file_t pointers by
 * their leafname.
 */
static int ds_fileptr_leaf_compare(const void *a, const void *b)
{
	ds_file_t ptr_a, ptr_b;
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	ptr_a = *((ds_file_t *) a);
	ptr_b = *((ds_file_t *) b);

	if ((NULL == ptr_a) || (NULL == ptr_b))
		return 0;

	x = ptr_a->leaf;
	y = ptr_b->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Return the index into dir->files[] that contains the given file, or -1 if
 * it's not present.  If the files array is unsorted, it is sorted first.
 */
static ssize_t ds_dir_file_lookup(ds_dir_t dir, ds_file_t file)
{
	/*@null@ */ ds_file_t *result;
	size_t index;

	if ((NULL == dir) || (NULL == file) || (NULL == dir->files) || (NULL == file->leaf))
		return -1;

	/* Sort the array if it needs to be sorted. */
	if (dir->files_unsorted) {
		debug("%s: %s", ds_dir_full_path(dir), "sorting files");
		qsort(dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);
		dir->files_unsorted = false;
#if ENABLE_DEBUGGING
		{
			size_t idx;
			for (idx = 0; NULL != dir->files && idx < dir->file_count; idx++) {
				debug("%s: %s[%d]: [%s]", ds_dir_full_path(dir), "files", idx, dir->files[idx]->leaf);
			}
		}
#endif
	}

	debug("%s: %s: %s", ds_dir_full_path(dir), file->leaf, "searching for file entry");
	result = bsearch(&file, dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);

	if (NULL == result) {
		debug("%s: %s: %s", ds_dir_full_path(dir), file->leaf, "file entry not found");
		return -1;
	}

	assert(result >= dir->files);
	if (result < dir->files)
		return -1;

	/*
	 * Subtracting the pointers gives us the array index since the
	 * pointers are of type ds_file_t.
	 */
	index = (size_t) (result - dir->files);

	debug("%s: %s: %s: %p (- %p) -> [%d]", ds_dir_full_path(dir), file->leaf, "bsearch result", result, dir->files,
	      index);

	assert(index < dir->file_count);
	if (index >= dir->file_count)
		return -1;

	return (ssize_t) index;
}


/*
 * Comparison function that compares two pointers to ds_dir_t pointers by
 * their leafname.
 */
static int ds_dirptr_leaf_compare(const void *a, const void *b)
{
	ds_dir_t ptr_a, ptr_b;
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	ptr_a = *((ds_dir_t *) a);
	ptr_b = *((ds_dir_t *) b);

	if ((NULL == ptr_a) || (NULL == ptr_b))
		return 0;

	x = ptr_a->leaf;
	y = ptr_b->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Return the index into dir->subdirs[] that contains the given
 * subdirectory, or -1 if it's not present.  If the subdirs array is
 * unsorted, it is sorted first.
 */
static ssize_t ds_dir_subdir_lookup(ds_dir_t dir, ds_dir_t subdir)
{
	/*@null@ */ ds_dir_t *result;
	size_t index;

	if ((NULL == dir) || (NULL == subdir) || (NULL == dir->subdirs) || (NULL == subdir->leaf))
		return -1;

	/* Sort the array if it needs to be sorted. */
	if (dir->subdirs_unsorted) {
		debug("%s: %s", ds_dir_full_path(dir), "sorting subdirs");
		qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
		dir->subdirs_unsorted = false;
#if ENABLE_DEBUGGING
		{
			size_t idx;
			for (idx = 0; NULL != dir->subdirs && idx < dir->subdir_count; idx++) {
				debug("%s: %s[%d]: [%s]", ds_dir_full_path(dir), "subdirs", idx,
				      dir->subdirs[idx]->leaf);
			}
		}
#endif
	}

	debug("%s: %s: %s", ds_dir_full_path(dir), subdir->leaf, "searching for subdir entry");
	result = bsearch(&subdir, dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);

	if (NULL == result) {
		debug("%s: %s: %s", ds_dir_full_path(dir), subdir->leaf, "subdir entry not found");
		return -1;
	}

	assert(result >= dir->subdirs);
	if (result < dir->subdirs)
		return -1;

	/*
	 * Subtracting the pointers gives us the array index since the
	 * pointers are of type ds_dir_t.
	 */
	index = (size_t) (result - dir->subdirs);

	debug("%s: %s: %s: %p (- %p) -> [%d]", ds_dir_full_path(dir), subdir->leaf, "bsearch result", result,
	      dir->subdirs, index);

	assert(index < dir->subdir_count);
	if (index >= dir->subdir_count)
		return -1;

	return (ssize_t) index;
}


/*
 * Add an entry to the change queue.
 *
 * In no_file_tracking mode, if this is a file, then immediately add the
 * file to the changed paths list without checks, and then remove the file
 * from its parent directory structure, freeing it (via ds_file_remove).
 */
static void _ds_change_queue_add( /*@null@ */ ds_topdir_t topdir, time_t when,	/*@null@ *//*@dependent@ */
				 ds_file_t file,
				 /*@null@ *//*@dependent@ */ ds_dir_t dir)
{
	char *changed_path;

	if (NULL == topdir)
		return;

	if ((NULL == file) && (NULL == dir))
		return;

	/*
	 * Shortcut straight to marking the path as changed, and forgetting
	 * the file, if this is a file and we're in no_file_tracking mode.
	 */
	if (watch_dir_params->no_file_tracking && (NULL != file)) {
		if (NULL != file->parent) {
			mark_file_path_changed(file->parent->topdir, file);
		}
		ds_file_remove(file);
		return;
	}

	/*
	 * Check the change isn't already queued - don't queue it twice.
	 */
	changed_path = NULL;
	if (NULL != file) {
		changed_path = ds_file_full_path(file);
	} else if (NULL != dir) {
		changed_path = ds_dir_full_path(dir);
	}

	/* Nothing to do if there was no path. */
	if (NULL == changed_path)
		return;

	/*@-mustfreefresh@ */
	if (NULL != changed_path && NULL != topdir->tree_change_queue) {
		nullable_string result;
		result = tfind(changed_path, &(topdir->tree_change_queue), compare_with_strcmp);
		if (NULL != result) {
			debug("%s: %s", changed_path, "path is already in the change queue");
			return;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the array if necessary.
	 */
	if (!extend_array
	    ((void **) (&(topdir->change_queue)), topdir->change_queue_length, sizeof(topdir->change_queue[0]),
	     CHANGE_QUEUE_ALLOC_CHUNK))
		return;
	assert(NULL != topdir->change_queue);
	if (NULL == topdir->change_queue)
		return;

	if (NULL != file) {
		debug("%s: %s: %s", "adding to change queue", "check file", changed_path);
	} else if (NULL != dir) {
		debug("%s: %s: %s", "adding to change queue", "scan directory", changed_path);
	}

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	/* Make a copy of the changed path. */
	changed_path = xstrdup(changed_path);
	assert(NULL != changed_path);
	if (NULL == changed_path)
		return;
	topdir->change_queue[topdir->change_queue_length].when = when;
	topdir->change_queue[topdir->change_queue_length].path = changed_path;
	topdir->change_queue[topdir->change_queue_length].file = file;
	topdir->change_queue[topdir->change_queue_length].dir = dir;

	/* Add the changed path to the binary tree. */
	/*@-nullstate@ */
	if (NULL != changed_path)
		(void) tsearch(changed_path, &(topdir->tree_change_queue), compare_with_strcmp);
	/*@+nullstate@ *//* topdir->tree_change_queue is OK to be NULL here. */

	topdir->change_queue_length++;
}


/*
 * Queue a file check.
 *
 * Note that in no_file_tracking mode, file will be freed, so never use file
 * again after calling this function (see _ds_change_queue_add).
 */
static void ds_change_queue_file_add( /*@dependent@ */ ds_file_t file, time_t when)
{
	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;

	if (0 == when) {
		/*
		 * If no specific time was given, choose a time that's a
		 * couple of seconds in the future so it's more likely it's
		 * finished being changed before we take any action.
		 *
		 * The larger the file is, the longer we give it.
		 */
		if (file->size < 102400) {  /* < 100KiB */
			when = time(NULL) + 2;
		} else if (file->size < 1048576) {	/* < 1MiB */
			when = time(NULL) + 3;
		} else if (file->size < 10485760) {	/* < 10MiB */
			when = time(NULL) + 4;
		} else if (file->size < 104857600) {	/* < 100MiB */
			when = time(NULL) + 5;
		} else {		    /* >= 100 MiB */
			when = time(NULL) + 6;
		}
	}

	_ds_change_queue_add(file->parent->topdir, when, file, NULL);
}


/*
 * Remove a file from the change queue.
 */
static void ds_change_queue_file_remove(ds_file_t file)
{
	ds_topdir_t topdir;
	size_t idx;

	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;
	if (NULL == file->parent->topdir->change_queue)
		return;

	topdir = file->parent->topdir;

	/* Remove the path from the change queue binary tree. */
	if (NULL != topdir->tree_change_queue) {
		char *path;
		path = ds_file_full_path(file);
		if (NULL != path)
			(void) tdelete(path, &(topdir->tree_change_queue), compare_with_strcmp);
	}

	/* Clear the entry from the change queue array. */
	for (idx = 0; idx < topdir->change_queue_length; idx++) {
		if (topdir->change_queue[idx].file == file) {
			topdir->change_queue[idx].file = NULL;
			free(topdir->change_queue[idx].path);
			topdir->change_queue[idx].path = NULL;
		}
	}
}


/*
 * Queue a directory scan.
 */
static void ds_change_queue_dir_add( /*@dependent@ */ ds_dir_t dir, time_t when)
{
	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;
	if (0 == when)
		when = time(NULL);
	_ds_change_queue_add(dir->topdir, when, NULL, dir);
}


/*
 * Remove a directory from the change queue.
 */
static void ds_change_queue_dir_remove(ds_dir_t dir)
{
	size_t idx;

	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;
	if (NULL == dir->topdir->change_queue)
		return;

	/* Remove the path from the change queue binary tree. */
	if (NULL != dir->topdir->tree_change_queue) {
		char *path;
		path = ds_dir_full_path(dir);
		if (NULL != path)
			(void) tdelete(path, &(dir->topdir->tree_change_queue), compare_with_strcmp);
	}

	/* Clear the entry from the change queue array. */
	for (idx = 0; idx < dir->topdir->change_queue_length; idx++) {
		if (dir->topdir->change_queue[idx].dir == dir) {
			dir->topdir->change_queue[idx].dir = NULL;
			free(dir->topdir->change_queue[idx].path);
			dir->topdir->change_queue[idx].path = NULL;
		}
	}
}


/*
 * Comparison function that compares two ds_file_t pointers by their
 * leafname.
 */
static int ds_file_leaf_compare(const void *a, const void *b)
{
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	x = ((ds_file_t) a)->leaf;
	y = ((ds_file_t) b)->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Add a file to the list of files in the given directory; if the file is
 * already in the list, return the existing file.
 *
 * The "name" string should contain the name of the file relative to the
 * directory (not the full path), e.g. "somefile".
 *
 * Returns the file, or NULL on error.
 */
static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add( /*@dependent@ */ ds_dir_t dir, const char *name)
{
	ds_file_t file;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;

	/*
	 * Use a binary tree to find this file if we've already added it to
	 * this directory - if we have, return a pointer to its structure.
	 */
	/*@-mustfreefresh@ */
	/*@-compdestroy@ */
	if (NULL != dir->tree_files) {
		struct ds_file_s key;
		ds_file_t *result;

		key.leaf = (char *) name;
		result = tfind(&key, &(dir->tree_files), ds_file_leaf_compare);
		if (NULL != result) {
			debug("%s: %s: %s", ds_dir_full_path(dir), name, "skipping duplicate file");
			return *result;
		}
	}
	/*@+compdestroy@ */
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the file array in the directory structure if we need to.
	 */
	if (!extend_array((void **) (&(dir->files)), dir->file_count, sizeof(dir->files[0]), DIRCONTENTS_ALLOC_CHUNK))
		return NULL;

	/*
	 * At this point, dir->files should have been allocated.
	 */
	assert(NULL != dir->files);
	if (NULL == dir->files)
		return NULL;

	/*
	 * Allocate a new file structure.
	 */
	file = calloc(1, sizeof(*file));
	if (NULL == file) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the file structure.
	 */
	file->leaf = xstrdup(name);
	file->leaf_length = strlen(name);   /* flawfinder: ignore */
	/* flawfinder - name is guaranteed null-terminated by the caller. */
	file->parent = dir;
	file->seen_in_rescan = false;

	/*
	 * Add the file to the directory structure.
	 */
	dir->files[dir->file_count] = file;
	dir->file_count++;

	/* Add the file to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(file, &(dir->tree_files), ds_file_leaf_compare);
	/*@+nullstate@ *//* dir->tree_files is OK to be NULL here. */

	/* Mark the files array as unsorted. */
	dir->files_unsorted = true;

	return file;
}


/*
 * Free a file information structure and its contents.
 *
 * The parent directory's "files" list is updated to remove this file from
 * it.
 */
static void ds_file_remove(ds_file_t file)
{
	if (NULL == file)
		return;
	if (NULL == file->leaf)
		return;

	debug("%s: %s", ds_file_full_path(file), "removing from file list");

	/*
	 * Remove this file from our parent directory's file listing, if we
	 * have a parent.
	 */
	if ((NULL != file->parent) && (NULL != file->leaf)) {
		size_t readidx, writeidx;
		ssize_t startidx;

		startidx = ds_dir_file_lookup(file->parent, file);

		if (startidx >= 0) {
			debug("%s: %s: %s[%d]", file->leaf, "removing from parent files index",
			      ds_dir_full_path(file->parent), startidx);
			for (readidx = (size_t) startidx + 1, writeidx = (size_t) startidx;
			     NULL != file->parent->files && readidx < file->parent->file_count; readidx++, writeidx++) {
				file->parent->files[writeidx] = file->parent->files[readidx];
			}
			file->parent->file_count = writeidx;
		}
	}

	/* Remove the file from the change queue. */
	ds_change_queue_file_remove(file);

	/* Remove the file from its directory's binary tree. */
	if (NULL != file->parent && NULL != file->parent->tree_files) {
		(void) tdelete(file, &(file->parent->tree_files), ds_file_leaf_compare);
	}

	/*
	 * Free the memory used by the pathname.
	 */
	free(file->leaf);
	file->leaf = NULL;
	file->leaf_length = 0;

	/* Free the file structure itself. */
	free(file);
}


/*
 * Check the given file's mtime and size; if either have changed, return 1.
 *
 * Returns 0 if nothing has changed, 1 if it has, or -1 if the file does not
 * exist or is not a regular file.
 *
 * If the file is successfully opened but there is an error while reading
 * it, 0 is returned as if it had not changed.
 */
static int ds_file_checkchanged(ds_file_t file)
{
	struct stat sb;
	char *path;

	if (NULL == file)
		return -1;

	path = ds_file_full_path(file);
	if (NULL == path)
		return -1;

	memset(&sb, 0, sizeof(sb));

	if (lstat(path, &sb) != 0)
		return -1;

	if (!S_ISREG((mode_t) (sb.st_mode)))
		return -1;

	if (((time_t) (sb.st_mtime) == file->mtime) && ((off_t) (sb.st_size) == file->size))
		return 0;

	debug("%s: %s", path, "file changed");

	file->mtime = (time_t) (sb.st_mtime);
	file->size = (off_t) (sb.st_size);

	return 1;
}


/*
 * Allocate and return a new top-level directory absolutely rooted at
 * "top_path".  All reported paths within the structure will be relative to
 * "top_path".
 *
 * The "fd_inotify" parameter should be the file descriptor to add directory
 * watches to for inotify, or -1 if inotify is not being used.
 */
static /*@null@ */ ds_topdir_t ds_dir_toplevel(int fd_inotify, const char *top_path)
{
	/*@null@ */ ds_topdir_t topdir;
	nullable_string real_top_path;
	size_t real_top_path_length;

	if (strlen(top_path) > PATH_MAX) {  /* flawfinder: ignore */
		die("%s: %s", top_path, strerror(ENAMETOOLONG));
		return NULL;
	}
	/* flawfinder - top_path is guaranteed null-terminated by the caller. */

	/*@-unrecog@ */
	/*@-mustfreeonly@ */
	/* splint doesn't know about realpath */
	real_top_path = realpath(top_path, NULL);	/* flawfinder: ignore */
	/*@+unrecog@ *//*@+mustfreeonly@ */
	if (NULL == real_top_path) {
		die("%s: %s", "realpath", strerror(errno));
		return NULL;
	}

	real_top_path_length = strlen(real_top_path);	/* flawfinder: ignore */

	/*
	 * flawfinder rationale: realpath()'s destination buffer is
	 * allocated by realpath() itself here, and we've checked that
	 * top_path is not too long; realpath() null-terminates the string
	 * it outputs, so calling strlen() on it will be OK.
	 */

	topdir = calloc(1, sizeof(*topdir));
	if (NULL == topdir) {
		die("%s: %s", "calloc", strerror(errno));
		free(real_top_path);
		return NULL;
	}

	topdir->path = real_top_path;
	topdir->path_length = real_top_path_length;

	topdir->dir.leaf = NULL;
	topdir->dir.leaf_length = 0;
	topdir->dir.wd = -1;
	topdir->dir.parent = NULL;
	topdir->dir.topdir = topdir;
	topdir->dir.seen_in_rescan = false;
	topdir->dir.removed = false;

	topdir->fd_inotify = fd_inotify;
	topdir->highest_wd = -1;

	return topdir;
}


/*
 * Comparison function that compares two ds_dir_t pointers by their
 * leafname.
 */
static int ds_dir_leaf_compare(const void *a, const void *b)
{
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	x = ((ds_dir_t) a)->leaf;
	y = ((ds_dir_t) b)->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Add a subdirectory to the list of subdirectories in the given parent
 * directory; if the subdirectory is already in the list, just return the
 * existing subdirectory.
 *
 * The "name" string should contain the name of the directory relative to
 * the parent directory (not the full path), e.g. "somedir".
 *
 * Returns the subdirectory, or NULL on error.
 */
/*@dependent@ */
static /*@null@ */ ds_dir_t ds_dir_add( /*@dependent@ */ ds_dir_t dir, const char *name)
{
	/*@null@ */ ds_dir_t subdir;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;

	/*
	 * Check that this subdirectory wouldn't be too deep.
	 */
	if (ds_dir_depth(dir) >= watch_dir_params->max_dir_depth) {
		debug("%s/%s: %s", ds_dir_full_path(dir), name, "too deep - not adding");
		return NULL;
	}

	/*
	 * Use a binary tree to find this subdirectory if we already know
	 * about it - if we do, return a pointer to its structure.
	 */
	/*@-mustfreefresh@ */
	if (NULL != dir->tree_subdirs) {
		struct ds_dir_s key;
		ds_dir_t *result;
		key.leaf = (char *) name;
		result = tfind(&key, &(dir->tree_subdirs), ds_dir_leaf_compare);
		if (NULL != result) {
			debug("%s: %s: %s", ds_dir_full_path(dir), name, "skipping duplicate directory");
			return *result;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the subdirectory array in the directory structure if we
	 * need to.
	 */
	if (!extend_array
	    ((void **) (&(dir->subdirs)), dir->subdir_count, sizeof(dir->subdirs[0]), DIRCONTENTS_ALLOC_CHUNK))
		return NULL;

	/*
	 * At this point, dir->subdirs should have been allocated.
	 */
	assert(NULL != dir->subdirs);
	if (NULL == dir->subdirs)
		return NULL;

	/*
	 * Allocate a new directory structure for the subdirectory.
	 */
	subdir = calloc(1, sizeof(*subdir));
	if (NULL == subdir) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the new subdirectory structure.
	 */
	subdir->leaf = xstrdup(name);
	subdir->leaf_length = strlen(name); /* flawfinder: ignore */
	/* flawfinder - name is guaranteed null-terminated by the caller. */

	subdir->wd = -1;
	subdir->parent = dir;
	subdir->topdir = dir->topdir;
	subdir->seen_in_rescan = false;
	subdir->removed = false;

	/*
	 * Add the subdirectory to the directory structure.
	 */
	dir->subdirs[dir->subdir_count] = subdir;
	dir->subdir_count++;

	/* Add the subdirectory to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(subdir, &(dir->tree_subdirs), ds_dir_leaf_compare);
	/*@+nullstate@ *//* dir->tree_subdirs is OK to be NULL here. */

	/* Mark the subdirectories array as unsorted. */
	dir->subdirs_unsorted = true;

	return subdir;
}


/*
 * Free a directory information structure.  This includes all files and
 * directories it contains.
 *
 * The parent directory's "subdirs" list is updated to remove this
 * subdirectory from it.
 *
 * If this the top directory (with a null parent), it is not freed (since in
 * that case it will be part of a ds_topdir_s structure, not allocated in
 * its own right), but all its values are set to NULL.
 */
static void ds_dir_remove(ds_dir_t dir)
{
	bool is_top_dir;
	size_t item;

	if (NULL == dir)
		return;

	debug("%s: %s", ds_dir_full_path(dir), "removing from directory list");

	is_top_dir = (NULL == dir->parent) ? true : false;

	/*
	 * Remove the watch on this directory.
	 */
	if ((dir->wd >= 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s (%d:%d)", ds_dir_full_path(dir), "removing watch", dir->topdir->fd_inotify, dir->wd);
		if (inotify_rm_watch(dir->topdir->fd_inotify, dir->wd) != 0) {
			/*
			 * We can get EINVAL if the directory was deleted,
			 * so just ignore that.
			 */
			if (errno != EINVAL) {
				error("%s: %s", "inotify_rm_watch", strerror(errno));
			}
		}
		ds_watch_index_remove(dir->topdir, dir->wd);
		watch_dir_metrics.directories_watching--;
		dir->wd = -1;
	}

	/*
	 * Remove all files from this directory.
	 */
	if (NULL != dir->files) {
		for (item = 0; item < dir->file_count; item++) {
			/*
			 * Wipe the parent field, so that ds_file_remove()
			 * doesn't try to remove the file from the directory
			 * information, since we're going to free it all
			 * below.
			 */
			dir->files[item]->parent = NULL;
			ds_file_remove(dir->files[item]);
		}
		free(dir->files);
		dir->files = NULL;
		dir->file_count = 0;
	}

	/*
	 * Remove all subdirectories from this directory (recursive).
	 */
	if (NULL != dir->subdirs) {
		/*
		 * First ensure subdirs is sorted, so the order can't change
		 * during removal.
		 */
		if (dir->subdir_count > 0 && dir->subdirs_unsorted) {
			qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
			dir->subdirs_unsorted = false;
		}
		for (item = 0; item < dir->subdir_count; item++) {
			/*
			 * Wipe the parent field, for the same reason as
			 * above: so that when we call ds_dir_remove()
			 * (ourselves) on each subdir, we don't try to
			 * remove it from the directory information, since
			 * the whole lot will be freed below.
			 */
			dir->subdirs[item]->parent = NULL;
			ds_dir_remove(dir->subdirs[item]);
			/*
			 * Because we set the parent to NULL,
			 * ds_dir_remove() treats this subdir as a top-level
			 * directory, so it won't free() the structure
			 * itself, only its contents.
			 */
			free(dir->subdirs[item]);
		}
		free(dir->subdirs);
		dir->subdirs = NULL;
		dir->subdir_count = 0;
	}

	/*
	 * Remove this subdirectory from our parent's directory listing, if
	 * we have a parent.  Note that when we call ourselves, above, we've
	 * wiped the parent in the subdirectory, to avoid pointlessly
	 * adjusting the array and binary tree that we're going to be
	 * removing anyway after this loop.
	 */
	if ((NULL != dir->parent) && (NULL != dir->leaf)) {
		size_t readidx, writeidx;
		ssize_t startidx;

		startidx = ds_dir_subdir_lookup(dir->parent, dir);

		if (startidx >= 0) {
			for (readidx = (size_t) startidx + 1, writeidx = (size_t) startidx;
			     NULL != dir->parent->subdirs && readidx < dir->parent->subdir_count;
			     readidx++, writeidx++) {
				dir->parent->subdirs[writeidx] = dir->parent->subdirs[readidx];
			}
			dir->parent->subdir_count = writeidx;
		}

		/* Remove this directory from its parent directory's binary tree. */
		if (NULL != dir->parent->tree_subdirs)
			(void) tdelete(dir, &(dir->parent->tree_subdirs), ds_dir_leaf_compare);
	}

	/* Remove the directory from the change queue. */
	ds_change_queue_dir_remove(dir);

	/*
	 * Free the memory used by the pathname.
	 */
	if (NULL != dir->leaf) {
		free(dir->leaf);
		dir->leaf = NULL;
		dir->leaf_length = 0;
	}

	/* Clear out the binary trees. */
	if (NULL != dir->tree_files) {
		tdestroy(dir->tree_files, _tdestroy_noop);
		dir->tree_files = NULL;
	}
	if (NULL != dir->tree_subdirs) {
		tdestroy(dir->tree_subdirs, _tdestroy_noop);
		dir->tree_subdirs = NULL;
	}

	/* Mark the directory as having been removed. */
	dir->removed = true;

	/* Free the directory structure itself, if it's not the top dir. */
	if (!is_top_dir)
		free(dir);
}


/*
 * Free the allocated items inside a topdir.
 */
static void ds_topdir_free_contents(ds_topdir_t topdir)
{
	if (NULL == topdir)
		return;

	/*
	 * Free the directory details.
	 */
	ds_dir_remove(&(topdir->dir));

	/*
	 * Free the watch index.
	 */
	if (NULL != topdir->watch_index) {
		free(topdir->watch_index);
		topdir->watch_index = NULL;
	}

	/*
	 * Free the change queue.
	 */
	if (NULL != topdir->change_queue) {
		size_t idx;
		for (idx = 0; idx < topdir->change_queue_length; idx++) {
			free(topdir->change_queue[idx].path);
			topdir->change_queue[idx].path = NULL;
		}
		free(topdir->change_queue);
		topdir->change_queue = NULL;
	}

	/*
	 * Free the changed paths list.
	 */
	if (NULL != topdir->changed_paths) {
		size_t idx;
		for (idx = 0; idx < topdir->changed_paths_length; idx++) {
			free(topdir->changed_paths[idx]);
		}
		free(topdir->changed_paths);
		topdir->changed_paths = NULL;
		topdir->changed_paths_length = 0;
	}

	/*
	 * Clear out the binary trees.
	 */
	if (NULL != topdir->tree_change_queue) {
		tdestroy(topdir->tree_change_queue, _tdestroy_noop);
		topdir->tree_change_queue = NULL;
	}
	if (NULL != topdir->tree_changed_paths) {
		tdestroy(topdir->tree_changed_paths, _tdestroy_noop);
		topdir->tree_changed_paths = NULL;
	}

	/* Free the top-level path. */
	free(topdir->path);
	topdir->path = NULL;
	topdir->path_length = 0;
}


/*
 * Filter for any filename.
 *
 * Ignore anything ending in .tmp or ~ by default, or if exclude_count is
 * >0, ignore anything matching any of the patterns in excludes[].
 *
 * Returns 1 if the file should be included, 0 if it should be ignored.
 */
static int ds_filename_valid(const char *leafname)
{
	if (leafname[0] == '\0')
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '\0'))
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '.')
	    && (leafname[2] == '\0'))
		return 0;

	if (watch_dir_params->exclude_count > 0) {
		/*
		 * If given an exclusion list, use it.
		 */
		size_t eidx;
		for (eidx = 0; eidx < watch_dir_params->exclude_count; eidx++) {
			if (NULL == watch_dir_params->excludes[eidx])
				continue;
			if (fnmatch(watch_dir_params->excludes[eidx], leafname, 0) == 0)
				return 0;
		}
	} else {
		/*
		 * Default is to exclude *~ and *.tmp
		 */
		size_t namelen;
		namelen = strlen(leafname); /* flawfinder: ignore */
		/* flawfinder - caller-supplied leafname must be null-terminated. */
		if ((1 <= namelen) && (leafname[namelen - 1] == '~'))
			return 0;
		if ((namelen > 4)
		    && (strcmp(&(leafname[namelen - 4]), ".tmp") == 0))
			return 0;
	}

	return 1;
}


/*
 * Filter for scanning directories - only include filenames we should be
 * processing.
 */
static int scan_directory_filter(const struct dirent *d)
{
	return ds_filename_valid(d->d_name);
}


/*
 * Scan the given directory.  Also checks files for changes.  Returns false
 * if the scan failed, in which case the directory will have been deleted
 * from the lists.
 *
 * If defer_subdir_scan is true, then subdirectories are added to the change
 * queue instead of being scanned immediately, though subdirectories are
 * still added and removed as necessary.  If it is false, this function
 * calls itself recursively for each subdirectory.
 */
static bool ds_dir_scan( /*@dependent@ */ ds_dir_t dir, bool defer_subdir_scan)
{
	struct dirent **namelist = NULL;
	char *scan_path;
	unsigned int depth;
	int namelist_length, itemidx, diridx, fileidx;
	struct stat dirsb;

	if (NULL == dir)
		return false;

	scan_path = ds_dir_full_path(dir);
	depth = ds_dir_depth(dir);

	debug("%s (%s %u): %s", scan_path, "depth", depth, "starting rescan");

	if (depth > watch_dir_params->max_dir_depth) {
		debug("%s: %s", scan_path, "too deep - removing");
		ds_dir_remove(dir);
		return false;
	}

	memset(&dirsb, 0, sizeof(dirsb));

	if (lstat(scan_path, &dirsb) != 0) {
		/*
		 * Use debug() rather than error() here, otherwise transient
		 * directories will generate noise.
		 */
		debug("%s: %s: %s", scan_path, "lstat", strerror(errno));
		ds_dir_remove(dir);
		return false;
	}

	/*
	 * Make a copy of "scan_path", since otherwise it will be
	 * overwritten by the next ds_*_full_path() call.
	 */
	scan_path = xstrdup(scan_path);
	if (NULL == scan_path)
		return false;

	/*@-compdef@ */
	/* splint thinks &namelist is not fully defined. */
	/*@-null@ */
	/* namelist starts off NULL; scandir fills it in. */
	namelist_length = scandir(scan_path, &namelist, scan_directory_filter, alphasort);
	/*@+null@ */
	/*@+compdef@ */
	if (namelist_length < 0) {
		/*
		 * Use debug() rather than error() here, as with lstat().
		 */
		debug("%s: %s: %s", scan_path, "scandir", strerror(errno));
		ds_dir_remove(dir);
		free(scan_path);
		return false;
	}

	/*
	 * Mark all subdirectories and files as having not been seen, so we
	 * can spot which ones have been removed after we've scanned the
	 * directory.
	 */
	for (diridx = 0; NULL != dir->subdirs && diridx < (int) (dir->subdir_count); diridx++) {
		dir->subdirs[diridx]->seen_in_rescan = false;
	}
	for (fileidx = 0; NULL != dir->files && fileidx < (int) (dir->file_count); fileidx++) {
		dir->files[fileidx]->seen_in_rescan = false;
	}

	/*
	 * Go through the directory scan results, adding new items to the
	 * arrays.
	 */
	for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
		char *item_leaf;
		char *item_full_path;
		struct stat sb;
		int d_type;

		/*
		 * In no_file_tracking mode, if scandir()'s d_type field is
		 * populated, skip directory entries that are definitely not
		 * subdirectories so we can avoid calling lstat() on files.
		 */
		d_type = (int) (namelist[itemidx]->d_type);
		if (watch_dir_params->no_file_tracking && d_type != DT_UNKNOWN && d_type != DT_DIR) {
			free(namelist[itemidx]);
			continue;
		}

		item_leaf = namelist[itemidx]->d_name;

		if (ds_filename_valid(item_leaf) == 0) {
			free(namelist[itemidx]);
			continue;
		}

		if (asprintf(&item_full_path, "%s/%s", scan_path, item_leaf) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return false;
		}
		if (NULL == item_full_path) {
			die("%s: %s", "asprintf", "NULL");
			return false;
		}

		memset(&sb, 0, sizeof(sb));

		if (lstat(item_full_path, &sb) != 0) {
			free(item_full_path);
			free(namelist[itemidx]);
			continue;
		}

		if (S_ISREG((mode_t) (sb.st_mode))) {
			ds_file_t file;
			file = ds_file_add(dir, item_leaf);
			if (NULL != file)
				file->seen_in_rescan = true;
		} else if (S_ISDIR((mode_t) (sb.st_mode))) {
			ds_dir_t subdir;
			if (sb.st_dev == dirsb.st_dev) {
				subdir = ds_dir_add(dir, item_leaf);
				if (NULL != subdir)
					subdir->seen_in_rescan = true;
			} else {
				debug("%s: %s", item_full_path, "skipping - different filesystem");
			}
		}

		free(item_full_path);
		free(namelist[itemidx]);
	}

	free(namelist);

	/*
	 * Sort the subdirectory array if it needs to be sorted.
	 *
	 * Below, while looping through the subdirectories array, after
	 * removing a subdirectory we backtrack diridx by one since the
	 * array just got shorter.  Removing a subdir causes our subdirs
	 * array to get sorted (via ds_dir_remove -> ds_dir_subdir_lookup)
	 * so if it wasn't already in order, that will change the order of
	 * the entries, and the loop will not function as intended.
	 */
	if (NULL != dir->subdirs && dir->subdir_count > 0 && dir->subdirs_unsorted) {
		qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
		dir->subdirs_unsorted = false;
	}

	/*
	 * Delete any subdirectories that we did not see on rescan, and scan
	 * those that we did.
	 */
	for (diridx = 0; NULL != dir->subdirs && diridx < (int) (dir->subdir_count); diridx++) {
		if (dir->subdirs[diridx]->seen_in_rescan) {
			if (defer_subdir_scan) {
				/*
				 * Scan subdirectory later, by adding it to
				 * the change queue.
				 */
				ds_change_queue_dir_add(dir->subdirs[diridx], 0);
			} else {
				/* Scan subdirectory now. */
				if (!ds_dir_scan(dir->subdirs[diridx], false)) {
					/*
					 * Scan failed, so this directory
					 * has gone, so move back one
					 * position in the array.
					 */
					diridx--;
				}
			}
		} else {
			debug("%s[%d]: %s: %s", scan_path, diridx, dir->subdirs[diridx]->leaf,
			      "subdirectory not seen in rescan - removing");
			ds_dir_remove(dir->subdirs[diridx]);
			/* Go back one, as this diridx has now gone */
			diridx--;
		}
	}

	/*
	 * Sort the files for the same reason as above.
	 */
	if (NULL != dir->files && dir->file_count > 0 && dir->files_unsorted) {
		qsort(dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);
		dir->files_unsorted = false;
	}

	/*
	 * Delete any files that we did not see on rescan, if we're not in
	 * no_file_tracking mode.
	 */
	for (fileidx = 0;
	     NULL != dir->files && (!watch_dir_params->no_file_tracking) && fileidx < (int) (dir->file_count);
	     fileidx++) {
		if (dir->files[fileidx]->seen_in_rescan)
			continue;
		debug("%s[%d]: %s: %s", scan_path, fileidx, dir->files[fileidx]->leaf,
		      "file not seen in rescan - removing");
		ds_file_remove(dir->files[fileidx]);
		/* Go back one, as this fileidx has now gone */
		fileidx--;
	}

	/*
	 * Check all files for changes, if we're not in no_file_tracking
	 * mode.
	 */
	for (fileidx = 0;
	     NULL != dir->files && (!watch_dir_params->no_file_tracking) && fileidx < (int) (dir->file_count);
	     fileidx++) {
		int changed;
		changed = ds_file_checkchanged(dir->files[fileidx]);
		if (changed < 0) {
			debug("%s[%d]: %s: %s", scan_path, fileidx, dir->files[fileidx]->leaf,
			      "file no longer present - removing");
			ds_file_remove(dir->files[fileidx]);
			/* Go back one, as this fileidx has now gone */
			fileidx--;
		}
	}

	/*
	 * Add an inotify watch to this directory if there isn't one
	 * already.
	 */
	if ((dir->wd < 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s", scan_path, "adding watch");
		dir->wd =
		    inotify_add_watch(dir->topdir->fd_inotify,
				      scan_path,
				      (uint32_t) (IN_CREATE | IN_DELETE | IN_MODIFY | IN_DELETE_SELF | IN_MOVED_FROM |
						  IN_MOVED_TO));
		if (dir->wd < 0) {
			error("%s: %s: %s", scan_path, "inotify_add_watch", strerror(errno));
		} else {
			/*
			 * Add this watch descriptor to the directory index,
			 * so that we can find this directory structure when
			 * an inotify event arrives.
			 */
			ds_watch_index_add(dir, dir->wd);
			watch_dir_metrics.directories_watching++;
		}
	}

	free(scan_path);

	return true;
}


/*
 * Process queued changes until the given time or until all queue entries
 * we're ready to process have been done.
 */
static void ds_change_queue_process(ds_topdir_t topdir, time_t work_until)
{
	size_t readidx, writeidx;

	if (NULL == topdir)
		return;

	if (topdir->change_queue_length < 1)
		return;

	if (NULL == topdir->change_queue)
		return;

	debug("%s: %s: %d", "change queue: starting run, queue length", topdir->path, topdir->change_queue_length);

	/*
	 * The "writeidx" position is moved along whenever we skip a change
	 * queue array entry, so that at the end of the loop, the change
	 * queue contains only the skipped items, and writeidx is the number
	 * of items left.
	 *
	 * If nothing was skipped, writeidx remains at 0, so the change
	 * queue length is set to zero at the end.
	 */

	for (readidx = 0, writeidx = 0; readidx < topdir->change_queue_length; readidx++) {
		ds_change_queue_t entry;
		time_t now;

		entry = &(topdir->change_queue[readidx]);
		if ((entry->file == NULL) && (entry->dir == NULL))
			continue;

		(void) time(&now);

		/*
		 * Skip if it's not yet time for this item, or if we've
		 * reached our work_until time.
		 */
		if ((entry->when > now) || (now >= work_until)) {
			/*
			 * When skipping an item, copy it into the write
			 * position if the read position is ahead of the
			 * write; and move the write index along one place
			 * regardless.
			 */
			if (readidx > writeidx) {
				memcpy(&(topdir->change_queue[writeidx]),	/* flawfinder: ignore */
				       entry, sizeof(*entry));
				/*
				 * flawfinder rationale - we're copying
				 * array entries, all of the same size.
				 */
			}
			writeidx++;
			continue;
		}

		if (NULL != entry->file) {
			int changed;
			ds_file_t file;

			file = entry->file;

			/* Clear the change queue array entry. */
			entry->file = NULL;

			/* Remove the path from the change queue binary tree. */
			if ((NULL != topdir->tree_change_queue) && (NULL != entry->path))
				(void) tdelete(entry->path, &(topdir->tree_change_queue), compare_with_strcmp);

			debug("%s: %s", NULL == entry->path ? "(null)" : entry->path, "checking for changes");

			free(entry->path);
			entry->path = NULL;

			changed = ds_file_checkchanged(file);

			if (changed < 0) {
				if ((NULL != file->parent)
				    && (!watch_dir_params->only_list_files)) {
					mark_dir_path_changed(file->parent->topdir, file->parent);
				}
				ds_file_remove(file);
			} else if (changed > 0) {
				if ((NULL != file->parent)
				    && (!watch_dir_params->only_list_directories)) {
					mark_file_path_changed(file->parent->topdir, file);
				}
			}

		} else if (NULL != entry->dir) {
			ds_dir_t dir;

			dir = entry->dir;

			/* Clear the change queue array entry. */
			entry->dir = NULL;

			/* Remove the path from the change queue binary tree. */
			if ((NULL != topdir->tree_change_queue) && (NULL != entry->path))
				(void) tdelete(entry->path, &(topdir->tree_change_queue), compare_with_strcmp);

			debug("%s: %s", NULL == entry->path ? "(null)" : entry->path, "triggering scan");

			free(entry->path);
			entry->path = NULL;

			(void) ds_dir_scan(dir, true);
		} else {
			free(entry->path);
			entry->path = NULL;
		}

		/*
		 * Note that free(entry->path) is repeated, above, because
		 * we free the path, and set it to NULL, *before* calling
		 * ds_dir_scan(), since that function can free() paths from
		 * the change queue and so we could end up with a
		 * double-free if it was done at the end here
		 * unconditionally.
		 */
	}

	topdir->change_queue_length = writeidx;

	debug("%s: %s: %d", "change queue: run ended, queue length", topdir->path, topdir->change_queue_length);
}


/*
 * Return true if the string is a valid exchange identity - that is, it
 * starts with a letter or number, is between 1 and 255 characters long, and
 * contains no characters other than letters, numbers, hyphens, underscores,
 * and dots.
 */
bool validate_exchange_identity(const char *string)
{
	unsigned int offset;

	if (NULL == string)
		return false;
	if ('\0' == string[0])
		return false;

	for (offset = 0; '\0' != string[offset]; offset++) {
		if (offset >= 255)
			return false;

		if (string[offset] >= '0' && string[offset] <= '9')
			continue;
		if (string[offset] >= 'A' && string[offset] <= 'Z')
			continue;
		if (string[offset] >= 'a' && string[offset] <= 'z')
			continue;

		/* Must start with a letter or number. */
		if (0 == offset)
			return false;

		if ('-' == string[offset])
			continue;
		if ('_' == string[offset])
			continue;
		if ('.' == string[offset])
			continue;

		return false;
	}

	return true;
}


/*
 * Filter for scanning the exchange directory for subdirectories to copy our
 * information to - only include valid exchange identifiers, and omit our
 * own.
 *
 * Returns 1 if the item should be included, 0 if not.
 */
static int tell_cluster_filter(const struct dirent *d)
{
	if (!validate_exchange_identity(d->d_name))
		return 0;
	if (NULL != watch_dir_params->exchange_identity) {
		if (0 == strcmp(d->d_name, watch_dir_params->exchange_identity))
			return 0;
	}
	return 1;
}


/*
 * Tell other watchdir instances about a change to a directory, by writing
 * the given path (followed by a \0 terminator) to an information file, and
 * linking that information file into every other watchdir's subdirectory of
 * the exchange directory.
 *
 * When changed_dir is not NULL, a new information filename is chosen if one
 * isn't already active, and the path is appended to that file (with a \0
 * terminator).
 *
 * When changed_dir is NULL, the active information file is hard-linked into
 * every other watchdir's subdirectory, the original is removed, and the
 * information filename becomes inactive, ready for a new one to be chosen.
 * If there wasn't an active file, nothing happens.
 *
 * This function needs to be called with NULL periodically, otherwise the
 * information file will never move on.  It is done this way so that a large
 * number of directory changes arriving quickly doesn't stress the
 * filesystem by generating a correspondingly large number of information
 * exchange files.
 */
static void tell_cluster_about_changed_dir(const /*@null@ */ char *changed_dir)
{
	static char *info_file = NULL;
	int dirfd_xch, dirfd_tmp;

	if (NULL == watch_dir_params->exchange_dir)
		return;
	if (NULL == watch_dir_params->exchange_identity)
		return;

	if (NULL == changed_dir) {
		/*
		 * Early return if it's time to pass the information file on
		 * to the other watchdirs but one isn't active.
		 */
		if (NULL == info_file)
			return;
	}

	if (NULL == info_file) {
		static unsigned long counter = 0;
		/*
		 * There isn't an active information file, so allocate a new
		 * one.
		 */
		if (asprintf
		    (&info_file, "%012lu-%s-%012lu", (unsigned long) time(NULL), watch_dir_params->exchange_identity,
		     counter) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			if (NULL != info_file) {
				free(info_file);
				info_file = NULL;
				return;
			}
		}
		if (NULL == info_file) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}
		counter++;
	}

	/* Open the exchange directory. */
	dirfd_xch = open(watch_dir_params->exchange_dir, O_DIRECTORY | O_RDONLY | O_PATH);
	if (dirfd_xch < 0) {
		debug("%s: %s: %s", watch_dir_params->exchange_dir, "open", strerror(errno));
		return;
	}

	/* Open the ".tmp" subdirectory of the exchange directory. */
	dirfd_tmp = openat(dirfd_xch, ".tmp", O_DIRECTORY | O_RDONLY | O_PATH);
	if (dirfd_tmp < 0) {
		debug("%s/%s: %s: %s", watch_dir_params->exchange_dir, ".tmp", "open", strerror(errno));
		(void) close(dirfd_xch);
		return;
	}

	/*
	 * If there's a changed path, write the changed path to the active
	 * information file.
	 */
	if (NULL != changed_dir) {
		int fd_info;
		FILE *fptr_info;

		fd_info = openat(dirfd_tmp, info_file, O_WRONLY | O_CREAT | O_APPEND, S_IRUSR | S_IWUSR);
		if (fd_info < 0) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, ".tmp", info_file, "open",
			      strerror(errno));
			(void) close(dirfd_tmp);
			(void) close(dirfd_xch);
			return;
		}

		fptr_info = fdopen(fd_info, "a");
		if (NULL == fptr_info) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, ".tmp", info_file, "fopen",
			      strerror(errno));
			(void) close(fd_info);
			(void) close(dirfd_tmp);
			(void) close(dirfd_xch);
			return;
		}

		if (fprintf(fptr_info, "%s%c", changed_dir, '\0') < 0) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, ".tmp", info_file, "fprintf",
			      strerror(errno));
			(void) fclose(fptr_info);
			(void) close(dirfd_tmp);
			(void) close(dirfd_xch);
			return;
		}

		if (0 != fclose(fptr_info)) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, ".tmp", info_file, "fclose",
			      strerror(errno));
			(void) close(dirfd_tmp);
			(void) close(dirfd_xch);
			return;
		}

		watch_dir_metrics.exchange_paths_output++;
	}

	/*
	 * If there isn't a changed path, and there is an active information
	 * file, hard-link the active information file into each of the
	 * subdirectories of the exchange directory belonging to other
	 * watchdir instances, and then delete the file and free its name so
	 * a new one will be chosen next time.
	 */
	if ((NULL == changed_dir) && (NULL != info_file)) {
		struct dirent **namelist = NULL;
		int namelist_length, itemidx;

		/*@-compdef@ */
		/* splint thinks &namelist is not fully defined. */
		/*@-null@ */
		/* namelist starts off NULL; scandirat fills it in. */
#if HAVE_SCANDIRAT
		namelist_length = scandirat(dirfd_xch, ".", &namelist, tell_cluster_filter, alphasort);
#else
		namelist_length = scandir(watch_dir_params->exchange_dir, &namelist, tell_cluster_filter, alphasort);
#endif
		/*@+null@ */
		/*@+compdef@ */
		if (namelist_length < 0) {
#if HAVE_SCANDIRAT
			debug("%s: %s: %s", watch_dir_params->exchange_dir, "scandirat", strerror(errno));
#else
			debug("%s: %s: %s", watch_dir_params->exchange_dir, "scandir", strerror(errno));
#endif
			(void) close(dirfd_tmp);
			(void) close(dirfd_xch);
			return;
		}

		for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
			char *item_leaf;
			int dirfd_item;

			item_leaf = namelist[itemidx]->d_name;

			dirfd_item = openat(dirfd_xch, item_leaf, O_DIRECTORY | O_RDONLY | O_PATH);
			if (dirfd_item < 0) {
				debug("%s/%s: %s: %s", watch_dir_params->exchange_dir, item_leaf, "open",
				      strerror(errno));
				continue;
			}

			if (0 != linkat(dirfd_tmp, info_file, dirfd_item, info_file, 0)) {
				debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, item_leaf, info_file, "link",
				      strerror(errno));
			}

			(void) close(dirfd_item);
		}

		/* Free the list of subdirectory names from scandirat(). */
		for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
			free(namelist[itemidx]);
		}
		free(namelist);

		/* Delete the info file from the ".tmp/" directory. */
		if (0 != unlinkat(dirfd_tmp, info_file, 0)) {
			debug("%s/%s: %s: %s", watch_dir_params->exchange_dir, info_file, "unlinkat", strerror(errno));
		}

		/* Free the name of the info file. */
		free(info_file);
		info_file = NULL;

		watch_dir_metrics.exchange_files_output++;
	}

	(void) close(dirfd_tmp);
	(void) close(dirfd_xch);
}


/*
 * Find the given path in topdirs[], populating *leaf with the leafname of
 * the given path, *dir_for_full_path with the ds_dir_t for the full path,
 * if present, and *dir_for_parent with the ds_dir_t for the parent of the
 * full path, if present.
 */
static void lookup_dir_by_full_path(const char *path, size_t path_length, char **leaf, ds_dir_t *dir_for_full_path,
				    ds_dir_t *dir_for_parent)
{
	char *path_leaf;
	ds_topdir_t topdir;
	size_t topdir_idx, topdir_match_length;
	ds_dir_t current_dir;

	if (NULL == path)
		return;
	if (NULL == dir_for_full_path)
		return;
	if (NULL == dir_for_parent)
		return;

	*dir_for_full_path = NULL;
	*dir_for_parent = NULL;

	path_leaf = memrchr(path, (int) '/', path_length);
	if (NULL == path_leaf) {
		/* Can't do anything with a root or empty path. */
		return;
	}
	path_leaf++;

	if (NULL != leaf)
		*leaf = path_leaf;

	if (NULL == topdirs)
		return;

	/* Find the longest matching topdir. */
	topdir_match_length = 0;
	topdir = NULL;
	for (topdir_idx = 0; topdir_idx < watch_dir_params->toplevel_path_count; topdir_idx++) {
		if (NULL == topdirs[topdir_idx])
			continue;
		if (NULL == topdirs[topdir_idx]->path)
			continue;
		if (topdirs[topdir_idx]->path_length <= topdir_match_length)
			continue;
		if (topdirs[topdir_idx]->path_length > path_length)
			continue;
		if ('/' != path[topdirs[topdir_idx]->path_length])
			continue;
		if (0 != memcmp(path, topdirs[topdir_idx]->path, topdirs[topdir_idx]->path_length))
			continue;
		topdir = topdirs[topdir_idx];
		topdir_match_length = topdirs[topdir_idx]->path_length;
	}

	if (NULL == topdir)
		return;

	debug("%s: %s=%s", path, "topdir", topdir->path);

	current_dir = &(topdir->dir);

	/* Move the path past the top directory (and trailing '/'). */
	path += topdir_match_length;
	path++;
	path_length -= topdir_match_length;
	path_length--;

	/*
	 * Step through subdirectories to match the full path.
	 */
	while ((path_length > 0) && (NULL != current_dir)) {
		char *separator;
		char *this_component;
		size_t this_component_length;
		bool is_last_component;
		/*@null@ */ ds_dir_t subdir;

		/*
		 * Get the length of this path component, determine whether
		 * it's the last one, and make a \0-terminated copy of it.
		 */
		is_last_component = false;
		separator = memchr(path, (int) '/', path_length);
		if (NULL == separator) {
			this_component_length = path_length;
			is_last_component = true;
		} else {
			this_component_length = separator - path;
		}
		this_component = malloc((size_t) (this_component_length + 1));
		if (NULL == this_component) {
			die("%s: %s", "malloc", strerror(errno));
			return;
		}
		memcpy(this_component, path, this_component_length);
		this_component[this_component_length] = '\0';

		/*
		 * Find this path component in the subdirectory index.
		 */
		subdir = NULL;
		/*@-mustfreefresh@ */
		if (NULL != current_dir->tree_subdirs) {
			struct ds_dir_s key;
			ds_dir_t *result;

			key.leaf = this_component;
			result = tfind(&key, &(current_dir->tree_subdirs), ds_dir_leaf_compare);
			if (NULL != result) {
				subdir = *result;
			}
		}
		/*@+mustfreefresh@ */
		/* splint thinks tfind() allocates "result", but it doesn't. */

		free(this_component);

		if (is_last_component) {
			*dir_for_parent = current_dir;
			*dir_for_full_path = subdir;
			break;
		}

		path += this_component_length + 1;
		if (path_length >= (this_component_length + 1)) {
			path_length -= (this_component_length + 1);
		} else {
			break;
		}

		current_dir = subdir;
	}
}


/*
 * Read and delete all of the information files in this watchdir's
 * subdirectory of the exchange directory.  Each information file is
 * expected to contain absolute paths, terminated with \0.  Every path will
 * be treated as though a change event had just been received about it.
 */
static void read_changed_dirs_from_cluster(void)
{
	int dirfd_xch, dirfd_local;
	struct dirent **namelist = NULL;
	int namelist_length, itemidx;

	if (NULL == watch_dir_params->exchange_dir)
		return;
	if (NULL == watch_dir_params->exchange_identity)
		return;

	/* Open the exchange directory. */
	dirfd_xch = open(watch_dir_params->exchange_dir, O_DIRECTORY | O_RDONLY | O_PATH);
	if (dirfd_xch < 0) {
		debug("%s: %s: %s", watch_dir_params->exchange_dir, "open", strerror(errno));
		return;
	}

	/* Open the subdirectory for this watchdir's exchange identity. */
	dirfd_local = openat(dirfd_xch, watch_dir_params->exchange_identity, O_DIRECTORY | O_RDONLY | O_PATH);
	if (dirfd_local < 0) {
		debug("%s/%s: %s: %s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity, "open",
		      strerror(errno));
		(void) close(dirfd_xch);
		return;
	}

	/*@-compdef@ */
	/* splint thinks &namelist is not fully defined. */
	/*@-null@ */
	/* namelist starts off NULL; scandirat fills it in. */
#if HAVE_SCANDIRAT
	namelist_length = scandirat(dirfd_local, ".", &namelist, NULL, alphasort);
#else
	{
		char *local_path = NULL;
		if (asprintf(&local_path, "%s/%s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity)
		    < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			(void) close(dirfd_local);
			(void) close(dirfd_xch);
			return;
		}
		if (NULL == local_path) {
			die("%s: %s", "asprintf", "NULL");
			(void) close(dirfd_local);
			(void) close(dirfd_xch);
			return;
		}
		namelist_length = scandir(local_path, &namelist, NULL, alphasort);
		free(local_path);
	}
#endif
	/*@+null@ */
	/*@+compdef@ */
	if (namelist_length < 0) {
		debug("%s/%s: %s: %s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity, "scandirat",
		      strerror(errno));
		(void) close(dirfd_local);
		(void) close(dirfd_xch);
		return;
	}

	for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
		char *item_leaf;
		int d_type;
		int fd;
		FILE *fptr_info;
		char *linebuf_ptr;
		size_t linebuf_size;
		unsigned int line_number;

		/* Only look at files starting with a digit. */
		item_leaf = namelist[itemidx]->d_name;
		if (item_leaf[0] == '\0')
			continue;
		if (item_leaf[0] < '0' || item_leaf[0] > '9')
			continue;

		/* Skip if the file type is known and it's not a regular file. */
		d_type = (int) (namelist[itemidx]->d_type);
		if (d_type != DT_UNKNOWN && d_type != DT_REG)
			continue;

		fd = openat(dirfd_local, item_leaf, O_RDONLY | O_NOFOLLOW);
		if (fd < 0) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity,
			      item_leaf, "open", strerror(errno));
			continue;
		}
		fptr_info = fdopen(fd, "r");
		if (NULL == fptr_info) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity,
			      item_leaf, "fdopen", strerror(errno));
			(void) close(fd);
			continue;
		}

		linebuf_ptr = NULL;
		linebuf_size = 0;
		line_number = 0;

		while (0 == feof(fptr_info)) {
			ssize_t line_length = 0;
			/*@null@ */ ds_dir_t dir_for_full_path, dir_for_parent;
			/*@dependent@ */ char *leaf;

			errno = 0;
			line_length = getdelim(&linebuf_ptr, &linebuf_size, 0, fptr_info);
			if ((line_length < 0) || (NULL == linebuf_ptr)) {
				if (0 != errno)
					error("%s/%s/%s:%u: %s", watch_dir_params->exchange_dir,
					      watch_dir_params->exchange_identity, item_leaf, line_number,
					      strerror(errno));
				break;
			}
			line_number++;

			/*
			 * The length includes the delimiter, and so it
			 * should always be > 0.
			 */
			if (line_length < 1)
				continue;

			/*
			 * Ensure the string is \0-terminated.
			 */
			line_length--;
			linebuf_ptr[line_length] = '\0';

			/*
			 * Mark the path as changed - if the full path
			 * already has an associated ds_dir_t then queue it
			 * for a rescan with ds_change_queue_dir_add(), if
			 * it doesn't but its parent directory does, add a
			 * ds_dir_t for it with ds_dir_add() and then queue
			 * that for a rescan that with
			 * ds_change_queue_dir_add().
			 */
			dir_for_full_path = NULL;
			dir_for_parent = NULL;
			leaf = NULL;
			lookup_dir_by_full_path(linebuf_ptr, (size_t) line_length, &leaf, &dir_for_full_path,
						&dir_for_parent);
			if (NULL == dir_for_parent) {
				/* path and its parent are unknown to us. */
				debug("%s: %s", linebuf_ptr, "path not known");
			} else if (NULL != dir_for_full_path) {
				/* full path known - existing dir changed. */
				debug("%s: %s", linebuf_ptr, "queueing rescan");
				ds_change_queue_dir_add(dir_for_full_path, 0);
			} else {
				/* path unknown but parent known - new dir. */
				ds_dir_t newdir;
				debug("%s: %s", linebuf_ptr, "adding new subdirectory");
				newdir = ds_dir_add(dir_for_parent, leaf);
				if (NULL != newdir) {
					ds_change_queue_dir_add(newdir, 0);
				}
			}

			watch_dir_metrics.exchange_paths_input++;
		}

		if (NULL != linebuf_ptr)
			free(linebuf_ptr);

		(void) fclose(fptr_info);
		if (0 != unlinkat(dirfd_local, item_leaf, 0)) {
			debug("%s/%s/%s: %s: %s", watch_dir_params->exchange_dir, watch_dir_params->exchange_identity,
			      item_leaf, "unlinkat", strerror(errno));
		}

		watch_dir_metrics.exchange_files_input++;
	}

	/* Free the list of names from scandirat(). */
	for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
		free(namelist[itemidx]);
	}
	free(namelist);

	(void) close(dirfd_local);
	(void) close(dirfd_xch);
}


/*
 * Process a change to a directory inside a watched directory.
 */
static void process_dir_change(struct inotify_event *event, /*@dependent@ */ ds_dir_t dir)
{
	ssize_t subdir_index;
	ds_dir_t subdir;
	inotify_action_t action;
	char *fullpath = NULL;
	struct stat sb;
	ds_dir_t newdir;

	/*
	 * Find the directory structure to which this event refers, if
	 * known.
	 */
	subdir = NULL;
	/*@-compdestroy@ */
	{
		struct ds_dir_s key;
		/* Put this in its own block so "key" has tiny scope. */
		memset(&key, 0, sizeof(key));
		key.leaf = event->name;
		subdir_index = ds_dir_subdir_lookup(dir, &key);
	}
	/*@+compdestroy@ *//* splint thinks key's other fields are used. */
	if ((subdir_index >= 0) && (NULL != dir->subdirs))
		subdir = dir->subdirs[subdir_index];

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != subdir)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != subdir)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new directory we've not seen before.
		 */

		/*
		 * Ignore the directory if it doesn't pass the filename
		 * filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		if (asprintf(&fullpath, "%s/%s", ds_dir_full_path(dir), event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the directory if it doesn't exist.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		}

		/*
		 * Ignore it if it's not a directory.
		 */
		if (!S_ISDIR((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Tell other watchdirs about it.
		 */
		if ((NULL != watch_dir_params->exchange_dir) && (NULL != watch_dir_params->exchange_identity)) {
			tell_cluster_about_changed_dir(fullpath);
		}

		/*
		 * Add the new directory and queue a scan for it.
		 */
		debug("%s: %s", fullpath, "adding new subdirectory");
		newdir = ds_dir_add(dir, event->name);
		free(fullpath);
		if (NULL != newdir)
			ds_change_queue_dir_add(newdir, 0);

		/*
		 * Mark this as a changed path.
		 */
		if (NULL != newdir && !watch_dir_params->only_list_files) {
			mark_dir_path_changed(dir->topdir, newdir);
		}

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a directory we've seen before, so queue a rescan for
		 * it.
		 */
		if (NULL != subdir) {
			/*
			 * Tell other watchdirs about it.
			 */
			if ((NULL != watch_dir_params->exchange_dir) && (NULL != watch_dir_params->exchange_identity)) {
				tell_cluster_about_changed_dir(ds_dir_full_path(subdir));
			}
			debug("%s: %s", ds_dir_full_path(subdir), "queueing rescan");
			ds_change_queue_dir_add(subdir, 0);
		}
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this directory before, delete its
		 * structure.
		 */
		if (NULL != subdir) {
			debug("%s: %s", ds_dir_full_path(subdir), "triggering removal");
			ds_dir_remove(subdir);
		}
		/*
		 * Mark the parent directory as a changed path.
		 */
		if ((NULL != dir->parent) && (!watch_dir_params->only_list_files)) {
			mark_dir_path_changed(dir->topdir, dir);
			/*
			 * Tell other watchdirs about it.
			 */
			if ((NULL != watch_dir_params->exchange_dir) && (NULL != watch_dir_params->exchange_identity)) {
				tell_cluster_about_changed_dir(ds_dir_full_path(dir));
			}
		}
		break;
	}
}


/*
 * Process a change to a file inside a watched directory.
 */
static void process_file_change(struct inotify_event *event, /*@dependent@ */ ds_dir_t dir)
{
	ssize_t file_index;
	ds_file_t file;
	inotify_action_t action;
	char *fullpath;
	struct stat sb;
	ds_file_t newfile;

	/*
	 * Find the file structure to which this event refers, if known.
	 */
	file = NULL;
	/*@-compdestroy@ */
	{
		struct ds_file_s key;
		/* Put this in its own block so "key" has tiny scope. */
		memset(&key, 0, sizeof(key));
		key.leaf = event->name;
		file_index = ds_dir_file_lookup(dir, &key);
	}
	/*@+compdestroy@ *//* splint thinks key's other fields are used. */
	if ((file_index >= 0) && (NULL != dir->files))
		file = dir->files[file_index];

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != file)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != file)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new file we've not seen before.
		 */

		/*
		 * Ignore the file if it doesn't pass the filename filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		fullpath = NULL;
		if (asprintf(&fullpath, "%s/%s", ds_dir_full_path(dir), event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the file if it doesn't exist or it isn't a regular
		 * file.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		} else if (!S_ISREG((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Add the new file and queue it to be checked.
		 */
		debug("%s: %s", fullpath, "adding new file");
		newfile = ds_file_add(dir, event->name);
		if (NULL != newfile)
			ds_change_queue_file_add(newfile, 0);

		free(fullpath);

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a file we've seen before, so queue a check for it.
		 */
		if (NULL != file)
			ds_change_queue_file_add(file, 0);
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this file before, delete its structure.
		 */
		if (NULL != file)
			debug("%s: %s", ds_file_full_path(file), "triggering removal");
		/*
		 * Mark the parent directory as a changed path.
		 */
		if (NULL != file && NULL != file->parent && !watch_dir_params->only_list_files) {
			mark_dir_path_changed(file->parent->topdir, file->parent);
		}
		if (NULL != file)
			ds_file_remove(file);
		break;
	}
}


/*
 * Process incoming inotify events.
 *
 * If the top directory is removed, it is not freed, but all its values are
 * nullified.
 */
static void process_inotify_events(ds_topdir_t topdir)
{
	unsigned char readbuf[8192];	 /* flawfinder: ignore */
	ssize_t got, pos;

	/*
	 * flawfinder rationale: readbuf is populated by read() which is
	 * passed readbuf's size.
	 */

	if (NULL == topdir)
		return;
	if (topdir->fd_inotify < 0)
		return;

	memset(readbuf, 0, sizeof(readbuf));

	/*
	 * Read as many events as we can.
	 */
	got = read(topdir->fd_inotify, readbuf, sizeof(readbuf));	/* flawfinder: ignore */
	/* flawfinder - read() called only once, bounded to buffer size. */
	if (got <= 0) {
		error("%s: %s: (%d): %s", NULL == topdir->path ? "(null)" : topdir->path, "inotify read event", got,
		      strerror(errno));
		(void) close(topdir->fd_inotify);
		topdir->fd_inotify = -1;
		return;
	}
	/* This shouldn't be possible, but check anyway. */
	if (got > (ssize_t) sizeof(readbuf))
		got = (ssize_t) sizeof(readbuf);

	/*
	 * Process each event that we've read.
	 */
	for (pos = 0; pos < got;) {
		struct inotify_event *event;
		/*@null@ */ ds_dir_t dir = NULL;

		event = (struct inotify_event *) &(readbuf[pos]);
		dir = ds_watch_index_lookup(topdir, event->wd, NULL);

#if ENABLE_DEBUGGING
		/*@-branchstate@ */
		/* splint doesn't like "p" being conditional. */
		if (debugging_enabled) {
			char flags[1024];	/* flawfinder: ignore */
			char *end;
			char *p;

			/*
			 * flawfinder - flags[] is large enough, initialised
			 * with a null terminator, and used with stpecpy()
			 * which also guards against overflow and ensures
			 * null termination.
			 */

			end = flags + sizeof(flags);
			flags[0] = '\0';
			p = flags;

			if (0 != (event->mask & IN_ACCESS))
				p = stpecpy(p, end, " IN_ACCESS");
			if (0 != (event->mask & IN_ATTRIB))
				p = stpecpy(p, end, " IN_ATTRIB");
			if (0 != (event->mask & IN_CLOSE_WRITE))
				p = stpecpy(p, end, " IN_CLOSE_WRITE");
			if (0 != (event->mask & IN_CLOSE_NOWRITE))
				p = stpecpy(p, end, " IN_CLOSE_NOWRITE");
			if (0 != (event->mask & IN_CREATE))
				p = stpecpy(p, end, " IN_CREATE");
			if (0 != (event->mask & IN_DELETE))
				p = stpecpy(p, end, " IN_DELETE");
			if (0 != (event->mask & IN_DELETE_SELF))
				p = stpecpy(p, end, " IN_DELETE_SELF");
			if (0 != (event->mask & IN_MODIFY))
				p = stpecpy(p, end, " IN_MODIFY");
			if (0 != (event->mask & IN_MOVE_SELF))
				p = stpecpy(p, end, " IN_MOVE_SELF");
			if (0 != (event->mask & IN_MOVED_FROM))
				p = stpecpy(p, end, " IN_MOVED_FROM");
			if (0 != (event->mask & IN_MOVED_TO))
				p = stpecpy(p, end, " IN_MOVED_TO");
			if (0 != (event->mask & IN_OPEN))
				p = stpecpy(p, end, " IN_OPEN");
			if (0 != (event->mask & IN_IGNORED))
				p = stpecpy(p, end, " IN_IGNORED");
			if (0 != (event->mask & IN_ISDIR))
				p = stpecpy(p, end, " IN_ISDIR");
			if (0 != (event->mask & IN_Q_OVERFLOW))
				p = stpecpy(p, end, " IN_Q_OVERFLOW");
			if (0 != (event->mask & IN_UNMOUNT))
				p = stpecpy(p, end, " IN_UNMOUNT");
			if (event->len > 0) {
				debug("%s: %d: %s: %.*s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : ds_dir_full_path(dir), event->len, event->name,
				      flags);
			} else {
				debug("%s: %d: %s: %s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : ds_dir_full_path(dir), "(none)", flags);
			}
		}
		/*@+branchstate@ */
#endif				/* ENABLE_DEBUGGING */

		pos += sizeof(*event) + event->len;

		/*
		 * There's nothing we can do if we don't know which
		 * directory it was.
		 */
		if (NULL == dir)
			continue;

		/*
		 * If the directory is itself being deleted, remove it.
		 */
		if (0 != (event->mask & IN_DELETE_SELF)) {
			/*
			 * Before removing it, mark its parent directory as
			 * a changed path.
			 */
			if ((NULL != dir->parent)
			    && (!watch_dir_params->only_list_files)) {
				mark_dir_path_changed(dir->topdir, dir->parent);
			}
			ds_dir_remove(dir);
			continue;
		}

		/*
		 * If this isn't an event about a named thing in this
		 * directory, we can't do anything.
		 */
		if (event->len < 1)
			continue;
		if ('\0' == event->name[0])
			continue;

		if (0 != (event->mask & IN_ISDIR)) {
			process_dir_change(event, dir);
		} else {
			process_file_change(event, dir);
		}
	}
}


/*
 * Add a path to the list of changed paths.
 */
static void _mark_path_changed( /*@null@ */ ds_topdir_t topdir, const char *path, bool isdir)
{
	char *savepath = NULL;

	if (NULL == topdir)
		return;

	if (NULL == path)
		return;

	if (asprintf(&savepath, "%s%s", path, isdir ? "/" : "") < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}
	assert(NULL != savepath);
	if (NULL == savepath)
		return;

	/*
	 * Check the path isn't already listed - don't list it twice.
	 */
	/*@-mustfreefresh@ */
	if (NULL != topdir->tree_changed_paths) {
		nullable_string result;
		result = tfind(savepath, &(topdir->tree_changed_paths), compare_with_strcmp);
		if (NULL != result) {
			debug("%s: %s", savepath, "path is already in the changed paths list");
			free(savepath);
			return;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the array if necessary.
	 */
	if (!extend_array((void **) (&(topdir->changed_paths)), topdir->changed_paths_length,
			  sizeof(topdir->changed_paths[0]), CHANGEDPATH_ALLOC_CHUNK))
		return;
	assert(NULL != topdir->changed_paths);
	if (NULL == topdir->changed_paths)
		return;

	debug("%s: %s", "adding to changed paths", savepath);

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	topdir->changed_paths[topdir->changed_paths_length] = savepath;
	topdir->changed_paths_length++;

	/* Add the changed path to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(savepath, &(topdir->tree_changed_paths), compare_with_strcmp);
	/*@+nullstate@ *//* topdir->tree_changed_paths is OK to be NULL here. */
}


/*
 * Add a file's path to the list of changed paths, using the absolute_paths
 * parameter to determine whether to add its absolute path or its path
 * relative to the topdir.
 */
static void mark_file_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_file_t file)
{
	const char *path_to_add;

	if (NULL == topdir)
		return;
	if (NULL == file)
		return;

	if (watch_dir_params->absolute_paths) {
		path_to_add = ds_file_full_path(file);
	} else {
		path_to_add = _ds_path(file->leaf, file->leaf_length, file->parent, topdir, false);
	}
	_mark_path_changed(topdir, path_to_add, false);
}


/*
 * Add a directory's path to the list of changed paths, using the
 * absolute_paths parameter to determine whether to add its absolute path or
 * its path relative to the topdir.
 */
static void mark_dir_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_dir_t dir)
{
	const char *path_to_add;

	if (NULL == topdir)
		return;
	if (NULL == dir)
		return;

	if (watch_dir_params->absolute_paths) {
		path_to_add = ds_dir_full_path(dir);
	} else {
		path_to_add = _ds_path(dir->leaf, dir->leaf_length, dir->parent, topdir, false);
	}
	_mark_path_changed(topdir, path_to_add, true);
}


/*
 * Write out a new file containing the current changed paths list, and clear
 * the list.
 */
static void dump_changed_paths(ds_topdir_t topdir, const char *savedir)
{
	char *savefile = NULL;
	char *temp_filename = NULL;
	struct tm *tm;
	time_t t;
	int tmpfd;
	FILE *fptr;
	size_t idx;
	char delimiter;

	if (NULL == topdir->changed_paths)
		return;
	if (topdir->changed_paths_length < 1)
		return;

	delimiter = '\n';
	if (watch_dir_params->null_delimiter)
		delimiter = '\0';

	t = time(NULL);
	tm = localtime(&t);

	if (asprintf
	    (&savefile, "%s/%04d%02d%02d-%02d%02d%02d.%d%s", savedir,
	     tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec, getpid(),
	     NULL == watch_dir_params->changefile_suffix ? "" : watch_dir_params->changefile_suffix) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}
	if (NULL == savefile) {
		die("%s: %s", "asprintf", "null");
		return;
	}

	tmpfd = ds_tmpfile(savefile, &temp_filename);
	if ((tmpfd < 0) || (NULL == temp_filename)) {
		free(savefile);
		return;
	}

	fptr = fdopen(tmpfd, "w");
	if (NULL == fptr) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		fprintf(fptr, "%s%c", topdir->changed_paths[idx], delimiter);
		watch_dir_metrics.changed_paths_output++;
	}

	if (0 != fclose(fptr)) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	if (rename(temp_filename, savefile) != 0) {
		error("%s: %s", savefile, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	free(temp_filename);
	free(savefile);

	watch_dir_metrics.change_files_output++;

	if (NULL != topdir->tree_changed_paths) {
		tdestroy(topdir->tree_changed_paths, _tdestroy_noop);
		topdir->tree_changed_paths = NULL;
	}

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		free(topdir->changed_paths[idx]);
	}

	free(topdir->changed_paths);
	topdir->changed_paths = NULL;
	topdir->changed_paths_length = 0;
}


/*
 * Write out a new file containing the current metrics.
 */
static void dump_metrics(const char *metrics_file)
{
	char *temp_filename = NULL;
	int tmpfd, dir_idx;
	FILE *fptr;

	if (NULL == metrics_file)
		return;

	tmpfd = ds_tmpfile(metrics_file, &temp_filename);
	if ((tmpfd < 0) || (NULL == temp_filename)) {
		return;
	}

	fptr = fdopen(tmpfd, "w");
	if (NULL == fptr) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	watch_dir_metrics.watch_index_length = 0;
	watch_dir_metrics.change_queue_length = 0;
	for (dir_idx = 0; dir_idx < watch_dir_params->toplevel_path_count && NULL != topdirs; dir_idx++) {
		watch_dir_metrics.watch_index_length += topdirs[dir_idx]->watch_index_length;
		watch_dir_metrics.change_queue_length += topdirs[dir_idx]->change_queue_length;
	}

	/* splint chokes on this. */
#ifndef SPLINT
#define write_metrics_value(x) fprintf(fptr, "%s %" PRId64 "\n", #x, watch_dir_metrics.x);
	write_metrics_value(directories_watching);
	write_metrics_value(change_files_output);
	write_metrics_value(changed_paths_output);
	write_metrics_value(exchange_files_output);
	write_metrics_value(exchange_paths_output);
	write_metrics_value(exchange_files_input);
	write_metrics_value(exchange_paths_input);
	write_metrics_value(watch_index_length);
	write_metrics_value(change_queue_length);
#endif

	if (0 != fclose(fptr)) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	if (rename(temp_filename, metrics_file) != 0) {
		error("%s: %s", metrics_file, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	free(temp_filename);
}


/*
 * Handler for an exit signal such as SIGTERM - set a flag to trigger an
 * exit.
 */
static void watch_dir_exitsignal( /*@unused@ */  __attribute__((unused))
				 int signum)
{
	watch_dir_exit_now = true;
}


/*
 * Main entry point.  Set everything up and enter the main loop, which does
 * the following:
 *
 *   - A periodic rescan from the top level directory down.
 *   - Processing of inotify events from all known directories.
 *   - Processing of the change queue generated from the above two.
 *   - Periodic output of a file listing updated paths.
 *   - Periodic exchange of changed directories with other watchdirs.
 *
 * Scanned directories are watched using inotify, so that changes to files
 * within it can be noticed immediately.
 *
 * A change queue is maintained, comprising a list of files and directories
 * to re-check, and the time at which to do so.  This is so that when
 * multiple files are changed, or the same file is changed several times, it
 * can be dealt with intelligently - they are pushed on to the change queue,
 * with duplicates being ignored, and then the change queue is processed in
 * chunks to avoid starvation caused by inotify events from one file
 * changing rapidly.
 */
int watch_dir(struct watch_dir_params_s *params)
{
	int *inotify_fds;		 /* fds to watch for inotify on */
	time_t next_full_scan;		 /* when to run next full scan */
	/* top-level directory contents */
	time_t next_change_queue_run;	 /* when to next run changes */
	time_t next_changedpath_dump;	 /* when to next dump changed paths */
	time_t next_exchange;		 /* when to exchange with other watchdirs */
	time_t next_metrics_dump;	 /* when to next write the metrics */
	size_t dir_idx;
	int rc;

	watch_dir_params = params;
	_ds_path_buffer = NULL;
	_ds_path_bufsize = 0;
	memset(&watch_dir_metrics, 0, sizeof(watch_dir_metrics));
	rc = EXIT_SUCCESS;

	/*
	 * Set up the signal handlers.
	 */
	/*
	 * splint sees sa.sa_mask as leaking memory, so put this code in its
	 * own scope so that compdestroy can be turned off for as short a
	 * time as possible.
	 */
	/*@-compdestroy@ */
	{
		struct sigaction sa;

		memset(&sa, 0, sizeof(sa));

		sa.sa_handler = watch_dir_exitsignal;
		(void) sigemptyset(&(sa.sa_mask));
		sa.sa_flags = 0;
		(void) sigaction(SIGTERM, &sa, NULL);

		sa.sa_handler = watch_dir_exitsignal;
		(void) sigemptyset(&(sa.sa_mask));
		sa.sa_flags = 0;
		(void) sigaction(SIGINT, &sa, NULL);
	}
	/*@+compdestroy@ */

	/*
	 * Create the inotify event queues.
	 *
	 * We use a separate event queue for each top-level directory so
	 * that when an event comes in, we can associate it with the
	 * appropriate top-level.
	 */
	inotify_fds = calloc(params->toplevel_path_count, sizeof(int));
	if (NULL == inotify_fds) {
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		inotify_fds[dir_idx] = inotify_init();
		if (inotify_fds[dir_idx] < 0) {
			error("%s: %s", "inotify", strerror(errno));
			free(inotify_fds);
			return EXIT_FAILURE;
		}
	}

	/*
	 * Create the top-level directory memory structures.
	 */
	topdirs = calloc(params->toplevel_path_count, sizeof(*topdirs));
	if (NULL == topdirs) {
		free(inotify_fds);
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		topdirs[dir_idx] = ds_dir_toplevel(inotify_fds[dir_idx], params->toplevel_paths[dir_idx]);
		if (NULL == topdirs[dir_idx]) {
			free(topdirs);
			topdirs = NULL;
			free(inotify_fds);
			return EXIT_FAILURE;
		}
	}

	if ((NULL != params->exchange_dir) && (NULL != params->exchange_identity)) {
		int dfd;
		struct stat sb;

		/*
		 * If an exchange directory and exchange identity were
		 * supplied, then check that the exchange directory exists,
		 * and then ensure that subdirectories for file creation
		 * (".tmp") and for this exchange identity are present.
		 */

		dfd = open(params->exchange_dir, O_DIRECTORY | O_RDONLY | O_PATH);
		if (dfd < 0) {
			error("%s: %s: %s", "open", params->exchange_dir, strerror(errno));
			rc = EXIT_FAILURE;
			goto end_watch_loop;
		}

		/*@-type@ */
		(void) mkdirat(dfd, ".tmp", S_IRWXU);
		(void) mkdirat(dfd, params->exchange_identity, S_IRWXU);
		/*@+type@ */
		/* splint sees __mode_t and mode_t above. */

		memset(&sb, 0, sizeof(sb));
		if ((0 != fstatat(dfd, ".tmp", &sb, AT_SYMLINK_NOFOLLOW))
		    || (((mode_t) (sb.st_mode) & S_IFMT) != S_IFDIR)) {
			error("%s/%s: %s", params->exchange_dir, ".tmp", _("cannot create required directory"));
			rc = EXIT_FAILURE;
			(void) close(dfd);
			goto end_watch_loop;
		}

		memset(&sb, 0, sizeof(sb));
		if ((0 != fstatat(dfd, params->exchange_identity, &sb, AT_SYMLINK_NOFOLLOW))
		    || (((mode_t) (sb.st_mode) & S_IFMT) != S_IFDIR)) {
			error("%s/%s: %s", params->exchange_dir, params->exchange_identity,
			      _("cannot create required directory"));
			rc = EXIT_FAILURE;
			(void) close(dfd);
			goto end_watch_loop;
		}

		(void) close(dfd);
	}

	/*
	 * Enter the main loop.
	 */

	next_change_queue_run = 0;
	next_full_scan = 0;
	next_changedpath_dump = 0;
	next_exchange = 0;
	next_metrics_dump = 0;

	while (!watch_dir_exit_now) {
		time_t now;
		fd_set readfds;
		struct timeval timeout;
		int ready, fd_max;

		/*
		 * Wait for an inotify event on any of the queues.
		 */
		memset(&readfds, 0, sizeof(readfds));
#if SPLINT
		/* splint has trouble with FD_ZERO. */
#else
		FD_ZERO(&readfds);
#endif
		fd_max = 0;
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			/*@-shiftnegative@ */
			FD_SET(fd_inotify, &readfds);
			/*@+shiftnegative@ *//* splint issues with FD_SET. */
			if (fd_inotify > fd_max)
				fd_max = fd_inotify;
		}

		timeout.tv_sec = 0;
		timeout.tv_usec = 100000;
		/*@-nullpass@ */
		ready = select(1 + fd_max, &readfds, NULL, NULL, &timeout);
		/*@+nullpass@ *//* passing NULL is OK here. */

		if (ready < 0) {
			if (errno != EINTR)
				error("%s: %s", "select", strerror(errno));
			watch_dir_exit_now = true;
			break;
		}

		/*
		 * Process new inotify events.
		 */
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && ready > 0 && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			/*@-shiftnegative@ */
			if (!FD_ISSET(fd_inotify, &readfds))
				continue;
			/*@+shiftnegative@ *//* splint issues with FD_ISSET. */

			process_inotify_events(topdirs[dir_idx]);
			if (topdirs[dir_idx]->dir.removed) {
				error("%s: %s", params->toplevel_paths[dir_idx], "top directory removed");
				watch_dir_exit_now = true;
				break;
			}
		}

		if (watch_dir_exit_now)
			break;

		if (0 == ready) {
			timeout.tv_sec = 0;
			timeout.tv_usec = 100000;
			/*@-nullpass@ */
			(void) select(0, NULL, NULL, NULL, &timeout);
			/*@+nullpass@ *//* passing NULL is OK here. */
		}

		(void) time(&now);

		/*
		 * Exchange information with other watchdir instances.
		 */
		if (now >= next_exchange && (NULL != params->exchange_dir) && (NULL != params->exchange_identity)) {
			next_exchange = now + params->exchange_interval;
			tell_cluster_about_changed_dir(NULL);
			read_changed_dirs_from_cluster();
		}

		/*
		 * Do a full scan periodically.
		 */
		if (now >= next_full_scan) {
			next_full_scan = now + params->full_scan_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_dir_add(&(topdirs[dir_idx]->dir), 0);
			}
		}

		/*
		 * Run our change queue.
		 */
		if (now >= next_change_queue_run) {
			next_change_queue_run = now + params->queue_run_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_process(topdirs[dir_idx], now + params->queue_run_max_seconds);
			}
		}

		/*
		 * Dump our list of changed paths.
		 */
		if (now >= next_changedpath_dump) {
			next_changedpath_dump = now + params->changedpath_dump_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				dump_changed_paths(topdirs[dir_idx], params->changedpath_dir);
			}
		}

		/*
		 * Dump our metrics.
		 */
		if ((now >= next_metrics_dump) && (NULL != params->metrics_file)) {
			next_metrics_dump = now + 5;
			dump_metrics(params->metrics_file);
		}
	}

      end_watch_loop:
	/*@-compdestroy@ */
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		ds_topdir_free_contents(topdirs[dir_idx]);
		(void) free(topdirs[dir_idx]);
		if (inotify_fds[dir_idx] >= 0) {
			if (0 != close(inotify_fds[dir_idx]))
				error("%s: %s", "close", strerror(errno));
		}
	}
	/*@+compdestroy@ */
	/*
	 * splint doesn't know that ds_topdir_free_contents() frees the
	 * internals of a topdir.
	 */
	(void) free(topdirs);
	(void) free(inotify_fds);
	topdirs = NULL;

	if (NULL != _ds_path_buffer) {
		(void) free(_ds_path_buffer);
		_ds_path_buffer = NULL;
		_ds_path_bufsize = 0;
	}

	if ((NULL != params->exchange_dir) && (NULL != params->exchange_identity)) {
		int dfd;

		/*
		 * If an exchange directory was supplied and we have an
		 * exchange identity, remove our exchange identity's
		 * subdirectory from the exchange directory.
		 */

		dfd = open(params->exchange_dir, O_DIRECTORY | O_RDONLY | O_PATH);
		if (dfd >= 0) {
			(void) unlinkat(dfd, params->exchange_identity, AT_REMOVEDIR);
			/*
			 * TODO: instead of unlink/rmdir, move the directory
			 * to a random name under .tmp, then delete its
			 * contents, then delete the directory.
			 * This would handle other processes trying to write
			 * to that directory while we're removing it.
			 */
			(void) close(dfd);
		}

		/* If there was an active information file, send it. */
		tell_cluster_about_changed_dir(NULL);
	}

	return rc;
}
