/*
 * Continuously synchronise the contents of a source directory to a given
 * destination, reading the details from a configuration file.
 *
 * Copyright 2014, 2021, 2023, 2025-2026 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

#include "config.h"
#include "common.h"
#include "sync.h"
#include "watch.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <wordexp.h>
#include <fnmatch.h>
#include <dirent.h>
#include <utime.h>
#include <search.h>
#include <signal.h>
#include <assert.h>

#define ACTION_WAITING "-"
#define ACTION_VALIDATION_SRC "VALIDATE-SOURCE"
#define ACTION_VALIDATION_DST "VALIDATE-DESTINATION"
#define ACTION_SYNC_FULL_WAIT "SYNC-FULL-AWAITING-LOCK"
#define ACTION_SYNC_FULL "SYNC-FULL"
#define ACTION_SYNC_PARTIAL_WAIT "SYNC-PARTIAL-AWAITING-LOCK"
#define ACTION_SYNC_PARTIAL "SYNC-PARTIAL"
#define STRING_STATUS_WAITING "-"
#define STRING_STATUS_OK "OK"
#define STRING_STATUS_FAILED "FAILED"


struct sync_status_s {
	/*@observer@ */ const char *action;
	pid_t watcher;
	pid_t pid;
	time_t next_full_sync;
	time_t next_partial_sync;
	time_t last_full_sync;
	time_t last_partial_sync;
	time_t last_failed_full_sync;
	time_t last_failed_partial_sync;
	/*@observer@ */ const char *last_full_sync_status;
	/*@observer@ */ const char *last_partial_sync_status;
	int full_sync_failures;
	int partial_sync_failures;
	char *workdir;
	char *excludes_file;
	char *rsync_error_file;
};

static int run_validation(struct sync_set_s *, /*@null@ */ const char *, const char *, struct sync_status_s *,
			  const char *);
static void run_watcher(struct sync_set_s *, bool);
static void update_timestamp_file(struct sync_set_s *cf, /*@null@ */ const char *);
static int sync_full(struct sync_set_s *, struct sync_status_s *, bool);
static int sync_partial(struct sync_set_s *, struct sync_status_s *, bool);
static void log_message( /*@null@ */ const char *, const char *, ...);
static void recursively_delete(const char *, int);


/*
 * Return a pointer to a static buffer describing the given epoch time as
 * "YYYY-MM-DD HH:MM:SS (@epoch)" in the local time zone, or "-" if the time
 * is 0.
 */
/*@dependent@ */ static char *dump_time(time_t t)
{
	static char tbuf[256];		 /* flawfinder: ignore */
	struct tm *tm;

	/*
	 * flawfinder rationale: the buffer is large enough for the string,
	 * writes are bounded using strftime()'s max buffer length
	 * parameter, and it is initialised with \0 bytes to ensure the
	 * string is terminated.
	 */

	memset(tbuf, 0, sizeof(tbuf));

	if (0 == t) {
		tbuf[0] = '-';
	} else {
		tm = localtime(&t);
		(void) strftime(tbuf, sizeof(tbuf) - 1, "%Y-%m-%d %H:%M:%S (@%s)", tm);
	}

	return tbuf;
}


/*
 * Update the status file, if we have one.
 */
static void update_status_file(struct sync_set_s *cf, struct sync_status_s *st)
{
	int tmpfd;
	char *temp_filename = NULL;
	FILE *status_fptr;

	if (sync_exit_now)
		return;
	if (NULL == cf->status_file)
		return;

	tmpfd = ds_tmpfile(cf->status_file, &temp_filename);
	if (tmpfd < 0) {
		create_parent_dirs(cf->status_file);
		tmpfd = ds_tmpfile(cf->status_file, &temp_filename);
		if (tmpfd < 0) {
			return;
		}
	}
	if (NULL == temp_filename) {
		/* An error will have already been reported, so say nothing. */
		if (tmpfd >= 0)
			(void) close(tmpfd);
		return;
	}

	status_fptr = fdopen(tmpfd, "w");
	if (NULL == status_fptr) {
		error("%s: %s(%d): %s", temp_filename, "fdopen", tmpfd, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		return;
	}

	fprintf(status_fptr, "section                  : %s\n", cf->name);
	fprintf(status_fptr, "current action           : %s\n", st->action);
	fprintf(status_fptr, "sync process             : %d\n", (int) (st->pid));
	if (0 == st->watcher) {
		fprintf(status_fptr, "watcher process          : -\n");
	} else {
		fprintf(status_fptr, "watcher process          : %d\n", (int) (st->watcher));
	}
	fprintf(status_fptr, "last full sync status    : %s\n", st->last_full_sync_status);
	fprintf(status_fptr, "last partial sync status : %s\n", st->last_partial_sync_status);
	fprintf(status_fptr, "last full sync           : %s\n", dump_time(st->last_full_sync));
	fprintf(status_fptr, "last partial sync        : %s\n", dump_time(st->last_partial_sync));
	fprintf(status_fptr, "next full sync           : %s\n", dump_time(st->next_full_sync));
	fprintf(status_fptr, "next partial sync        : %s\n", dump_time(st->next_partial_sync));
	fprintf(status_fptr, "failed full sync         : %s\n", dump_time(st->last_failed_full_sync));
	fprintf(status_fptr, "failed partial sync      : %s\n", dump_time(st->last_failed_partial_sync));
	fprintf(status_fptr, "partial sync failures    : %d\n", st->partial_sync_failures);
	fprintf(status_fptr, "full sync failures       : %d\n", st->full_sync_failures);
	fprintf(status_fptr, "working directory        : %s\n", st->workdir);

	/*
	 * Trailing blank line so you can cat everything in
	 * /var/run/continual-sync/ and it is tidy.
	 */

	fprintf(status_fptr, "\n");

	if (0 != fchmod(tmpfd, 0644)) {
		error("%s: %s: %s", cf->status_file, "fchmod()", strerror(errno));
	}

	if (0 != fclose(status_fptr)) {
		error("%s: %s: %s", cf->status_file, "fclose()", strerror(errno));
	}

	if (rename(temp_filename, cf->status_file) != 0) {
		error("%s: %s: %s", cf->status_file, "rename()", strerror(errno));
		if (0 != remove(temp_filename)) {
			error("%s: %s: %s", temp_filename, "remove()", strerror(errno));
		}
	}

	free(temp_filename);
}

/*@-mustfreefresh@ */
/*
 * splint and gettext() / _() don't get on - we turn off mustfreefresh
 * because splint thinks each gettext call causes a memory leak.
 */

/*
 * Run a continual sync as defined by the given configuration.  Assumes that
 * the "sync_exit_now" flag will be set by a signal handler when a SIGTERM
 * is received.
 */
void continual_sync(struct sync_set_s *cf, bool rsync_null_delimiter)
{
	struct sync_status_s status;
	struct stat sb;
	FILE *fptr;

	memset(&status, 0, sizeof(status));
	status.action = ACTION_WAITING;
	status.watcher = 0;
	status.pid = (pid_t) getpid();
	status.next_full_sync = 0;
	status.next_partial_sync = 0;
	status.last_full_sync = 0;
	status.last_partial_sync = 0;
	status.last_failed_full_sync = 0;
	status.last_failed_partial_sync = 0;
	status.last_full_sync_status = STRING_STATUS_WAITING;
	status.last_partial_sync_status = STRING_STATUS_WAITING;
	status.full_sync_failures = 0;
	status.partial_sync_failures = 0;
	status.workdir = NULL;
	status.rsync_error_file = NULL;
	status.excludes_file = NULL;

	/*
	 * Create a temporary working directory.
	 */
	/*@-unrecog@ */
	/* splint doesn't know about asprintf() */
	if (asprintf(&(status.workdir), "%s/%s", NULL == cf->tempdir ? "/tmp" : cf->tempdir, "syncXXXXXX")
	    < 0) {
		error("%s: %s", "asprintf", strerror(errno));
		status.workdir = NULL;
		goto continual_sync_final_cleanup;
	}
	/*@+unrecog@ */
	assert(NULL != status.workdir);
	if (NULL == status.workdir) {
		die("%s: %s", "asprintf", "NULL");
		goto continual_sync_final_cleanup;
	}
	/*@-unrecog@ */
	/* splint doesn't know about mkdtemp() */
	if (mkdtemp(status.workdir) == NULL) {
		error("%s: %s: %s", "mkdtemp", status.workdir, strerror(errno));
		goto continual_sync_final_cleanup;
	}
	/*@+unrecog@ */
	debug("%s: %s", "temporary working directory", status.workdir);

	/*
	 * Create the file that rsync will write stderr to.
	 */
	if (asprintf(&(status.rsync_error_file), "%s/%s", status.workdir, "rsync-stderr") < 0) {
		error("%s: %s", "asprintf", strerror(errno));
		status.rsync_error_file = NULL;
		goto continual_sync_final_cleanup;
	}
	assert(NULL != status.rsync_error_file);
	if (NULL == status.rsync_error_file) {
		die("%s: %s", "asprintf", "NULL");
		goto continual_sync_final_cleanup;
	}

	/*
	 * Create the file that rsync will use with --excludes-from.
	 */
	if (asprintf(&(status.excludes_file), "%s/%s", status.workdir, "excludes")
	    < 0) {
		error("%s: %s", "asprintf", strerror(errno));
		status.excludes_file = NULL;
		goto continual_sync_final_cleanup;
	}
	assert(NULL != status.excludes_file);
	if (NULL == status.excludes_file) {
		die("%s: %s", "asprintf", "NULL");
		goto continual_sync_final_cleanup;
	}

	fptr = fopen(status.excludes_file, "w");	/* flawfinder: ignore */
	/*
	 * flawfinder warns about the path being redirected but the path is
	 * under a temporary directory we just created so an attacker would
	 * need to already be running under the same user ID.
	 */
	if (NULL == fptr) {
		error("%s: %s: %s", status.excludes_file, "fopen", strerror(errno));
		goto continual_sync_final_cleanup;
	}
	if (cf->exclude_count > 0) {
		int eidx;
		for (eidx = 0; eidx < cf->exclude_count; eidx++) {
			/* Use \0 rather than \n if rsync is using null delimiters (#11). */
			fprintf(fptr, "%s%c", cf->excludes[eidx], rsync_null_delimiter ? '\0' : '\n');
		}
	} else {
		fprintf(fptr, "*.tmp\n*~\n");
	}
	if (0 != fclose(fptr)) {
		error("%s: %s: %s", status.excludes_file, "fclose", strerror(errno));
	}

	/*
	 * If no transfer list file was defined, define one under the
	 * temporary working directory.
	 */
	if (NULL == cf->transfer_list) {
		if (asprintf(&(cf->transfer_list), "%s/%s", status.workdir, "transfer") < 0) {
			error("%s: %s", "asprintf", strerror(errno));
			cf->transfer_list = NULL;
			goto continual_sync_final_cleanup;
		}
		debug("%s: %s", "automatically set transfer list", cf->transfer_list);
	}
	assert(NULL != cf->transfer_list);
	if (NULL == cf->transfer_list) {
		die("%s: %s", "asprintf", "NULL");
		goto continual_sync_final_cleanup;
	}

	/*
	 * If no change queue directory was defined, create one under the
	 * temporary working directory.
	 */
	if (NULL == cf->change_queue) {
		if (asprintf(&(cf->change_queue), "%s/%s", status.workdir, "changes") < 0) {
			error("%s: %s", "asprintf", strerror(errno));
			cf->change_queue = NULL;
			goto continual_sync_final_cleanup;
		}
		assert(NULL != cf->change_queue);
		if (NULL == cf->change_queue) {
			die("%s: %s", "asprintf", "NULL");
			goto continual_sync_final_cleanup;
		}

		if (mkdir(cf->change_queue, 0700) != 0) {
			error("%s: %s: %s", cf->change_queue, "mkdir", strerror(errno));
			free(cf->change_queue);
			cf->change_queue = NULL;
			goto continual_sync_final_cleanup;
		}
		debug("%s: %s", "automatically set change queue", cf->change_queue);
	}

	log_message(cf->log_file, "[%s] %s", cf->name, _("sync process started"));

	/*
	 * If we're using a full sync marker file, use its timestamp to set
	 * our idea of the last full sync time, and to determine the next
	 * full sync time.
	 */
	if ((NULL != cf->full_marker) && (stat(cf->full_marker, &sb) == 0)) {
		status.last_full_sync = (time_t) (sb.st_mtime);
		log_message(cf->log_file, "[%s] %s: %s", cf->name,
			    _("set last full sync time from marker file"), dump_time(status.last_full_sync));
		status.next_full_sync = (time_t) (sb.st_mtime + cf->full_interval);
		log_message(cf->log_file, "[%s] %s: %s", cf->name,
			    _("derived next full sync time from marker file"), dump_time(status.next_full_sync));
	}

	/*
	 * If we're using a partial sync marker file, use its timestamp to
	 * set our idea of the last partial sync time, and to determine the
	 * next partial sync time.
	 */
	if ((NULL != cf->partial_marker)
	    && (stat(cf->partial_marker, &sb) == 0)) {
		status.last_partial_sync = (time_t) (sb.st_mtime);
		log_message(cf->log_file, "[%s] %s: %s", cf->name,
			    _("set last partial sync time from marker file"), dump_time(status.last_partial_sync));
		status.next_partial_sync = (time_t) (sb.st_mtime + cf->partial_interval);
		log_message(cf->log_file, "[%s] %s: %s", cf->name,
			    _("derived next partial sync time from marker file"), dump_time(status.next_partial_sync));
	}

	/*
	 * Update our status file, if we have one.
	 */
	update_status_file(cf, &status);

	/*
	 * Main loop.
	 */
	while (!sync_exit_now) {
		bool check_workdir = false;

		/*
		 * If there is no watcher and there should be one, start
		 * one.
		 */
		if ((0 == status.watcher) && (cf->partial_interval > 0)) {
			pid_t child;

			/*
			 * If we have a source validation command, run that
			 * first, and skip to the end of the loop on
			 * failure.
			 */
			if (run_validation(cf, cf->source_validation, _("source"), &status, ACTION_VALIDATION_SRC) != 0) {
				status.action = ACTION_WAITING;
				update_status_file(cf, &status);
				(void) sleep(5);
			} else {
				child = (pid_t) fork();
				if (0 == child) {
					/* Child - run watcher */
					run_watcher(cf, rsync_null_delimiter);
					/*
					 * We return here instead of exiting
					 * so that the main sync program can
					 * do its usual cleanup, freeing up
					 * memory etc.
					 *
					 * NB we don't jump to
					 * continual_sync_final_cleanup
					 * because that would also delete
					 * the working directory.
					 */
					free(status.rsync_error_file);
					free(status.excludes_file);
					free(status.workdir);
					return;
				} else if (child < 0) {
					/* Error - output a warning */
					error("%s: %s", "fork", strerror(errno));
				} else {
					/* Parent */
					status.watcher = child;
					log_message(cf->log_file,
						    "[%s] %s: %d", cf->name, _("started new watcher process"),
						    status.watcher);
				}
			}
		}

		/*
		 * If it's time for a full sync, run one.
		 */
		if ((time(NULL) >= status.next_full_sync)
		    && (cf->full_interval > 0)) {

			check_workdir = true;

			if ((run_validation
			     (cf, cf->source_validation, _("source"), &status, ACTION_VALIDATION_SRC) != 0)
			    ||
			    (run_validation
			     (cf, cf->destination_validation, _("destination"), &status, ACTION_VALIDATION_DST) != 0)) {
				/*
				 * Validation failed - just retry later
				 */
				status.next_full_sync = time(NULL) + cf->full_retry;
			} else {
				int sync_rc = false;
				/*
				 * Validation succeeded - attempt to run
				 * sync
				 */
				sync_rc = sync_full(cf, &status, rsync_null_delimiter);
				if (0 == sync_rc) {
					/* sync succeeded */
					/*
					 * No need to update last_full_sync
					 * etc, as sync_full() did all that.
					 */
					status.next_full_sync = time(NULL) + cf->full_interval;
				} else {
					/* sync failed */
					status.next_full_sync = time(NULL) + cf->full_retry;
					status.last_failed_full_sync = time(NULL);
					status.full_sync_failures++;
					status.last_full_sync_status = STRING_STATUS_FAILED;
				}
			}
			/*
			 * Update our status after the attempt.
			 */
			status.action = ACTION_WAITING;
			update_status_file(cf, &status);
		}

		/*
		 * If it's time for a partial sync and we have a watcher
		 * process, run a partial sync.
		 */
		if ((0 != status.watcher)
		    && (time(NULL) >= status.next_partial_sync)) {

			check_workdir = true;

			if ((run_validation
			     (cf, cf->source_validation, _("source"), &status, ACTION_VALIDATION_SRC) != 0)
			    ||
			    (run_validation
			     (cf, cf->destination_validation, _("destination"), &status, ACTION_VALIDATION_DST) != 0)) {
				/*
				 * Validation failed - just retry later
				 */
				status.next_partial_sync = time(NULL) + cf->partial_retry;
			} else {
				int sync_rc = 0;
				/*
				 * Validation succeeded - attempt to run
				 * sync
				 */
				sync_rc = sync_partial(cf, &status, rsync_null_delimiter);
				if (0 == sync_rc) {
					/* sync succeeded OR not run */
					status.next_partial_sync = time(NULL) + cf->partial_interval;
					/*
					 * No need to update
					 * last_partial_sync etc, as
					 * sync_partial() did all that.
					 */
				} else {
					/* sync run AND failed */
					status.next_partial_sync = time(NULL) + cf->partial_retry;
					status.last_failed_partial_sync = time(NULL);
					status.partial_sync_failures++;
					status.last_partial_sync_status = STRING_STATUS_FAILED;
				}
			}
			/*
			 * Update our status after the attempt.
			 */
			status.action = ACTION_WAITING;
			update_status_file(cf, &status);
		}

		/*
		 * Clean up our child process if it has exited.
		 */
		if ((0 != status.watcher)
		    && (waitpid(status.watcher, NULL, WNOHANG) != 0)) {
			check_workdir = true;
			log_message(cf->log_file, "[%s] %s", cf->name, _("watcher process ended"));
			status.watcher = 0;
		}

		if (check_workdir) {
			/*
			 * Check whether our work directory, or the excludes
			 * file inside it, has disappeared; exit immediately
			 * if so.
			 *
			 * We only check this after a full or partial sync
			 * attempt, or when a watcher process exits, to
			 * avoid too many stat calls.
			 */
			check_workdir = false;
			if (stat(status.workdir, &sb) != 0) {
				log_message(cf->log_file, "[%s] %s",
					    cf->name, _("exiting immediately - the working directory has disappeared"));
				sync_exit_now = true;
			} else if (stat(status.excludes_file, &sb) != 0) {
				log_message(cf->log_file, "[%s] %s",
					    cf->name, _("exiting immediately - the exclusions file has disappeared"));
				sync_exit_now = true;
			}
		}

		if (!sync_exit_now)
			(void) usleep(100000);
	}

	/*
	 * Kill our watcher process, if we have one.
	 */
	if (0 != status.watcher) {
		if (0 != kill(status.watcher, SIGTERM)) {
			log_message(cf->log_file, "[%s] %s: %d: %s", cf->name, _("failed to terminate watcher process"),
				    status.watcher, strerror(errno));
		}
	}

	/*
	 * Remove our status file, if we have one.
	 */
	if (NULL != cf->status_file) {
		if (0 != remove(cf->status_file)) {
			log_message(cf->log_file, "[%s] %s: %s: %s", cf->name, _("failed to remove status file"),
				    cf->status_file, strerror(errno));
		}
	}

	log_message(cf->log_file, "[%s] %s", cf->name, _("sync process ended"));

	/*
	 * Below is where we jump to if any of the initial setup, before the
	 * main loop, failed.
	 */
      continual_sync_final_cleanup:
	/*
	 * Free the rsync_error_file string we made.
	 */
	if (NULL != status.rsync_error_file)
		free(status.rsync_error_file);

	/*
	 * Free the excludes_file string we made.
	 */
	if (NULL != status.excludes_file)
		free(status.excludes_file);

	/*
	 * Remove our temporary working directory and free its string.
	 */
	if (NULL != status.workdir) {
		recursively_delete(status.workdir, 0);
		free(status.workdir);
	}
}


/*
 * Run the given command, if there is one (returns zero if not).  Returns
 * nonzero if the command was run and it failed, and logs the error.
 *
 * Before the command starts, st->action is set to "action" and the
 * status file is updated.
 */
static int run_validation(struct sync_set_s *cf, /*@null@ */ const char *command,
			  const char *name, struct sync_status_s *st, const char *action)
{
	int ret, exit_status;

	if (NULL == command)
		return 0;

	debug("(sync) [%s] running %s validation: [%s]", cf->name, name, command);

	st->action = action;
	update_status_file(cf, st);

	ret = system(command);

	if (WIFSIGNALED(ret)) {
		log_message(cf->log_file, "[%s] %s: %s: %d",
			    cf->name, name, _("the validation command received a signal"), WTERMSIG(ret));
		sync_exit_now = true;
		return 1;
	}

	exit_status = WEXITSTATUS(ret);
	if (exit_status == 0)
		return 0;

	log_message(cf->log_file,
		    "[%s] %s: %s: %d", cf->name, name, _("the validation command returned a non-zero exit status"),
		    exit_status);

	return 1;
}


/*
 * Run the watcher on the source directory.
 */
static void run_watcher(struct sync_set_s *cf, bool rsync_null_delimiter)
{
	struct watch_dir_params_s params;
	size_t exclude_idx;

	if (NULL == cf->source)
		return;
	if (NULL == cf->change_queue)
		return;

	setproctitle("%s %s [%s]", common_program_name, _("watcher"), cf->name);

	memset(&params, 0, sizeof(params));
	params.toplevel_path_count = 1;
	params.toplevel_paths = calloc(1, sizeof(toplevel_path_t));
	if (NULL == params.toplevel_paths) {
		return;
	}
	params.toplevel_paths[0] = cf->source;
	params.changedpath_dir = cf->change_queue;
	params.full_scan_interval = cf->full_interval;
	params.queue_run_interval = 2;
	params.queue_run_max_seconds = 5;
	params.changedpath_dump_interval = cf->partial_interval;
	params.exchange_interval = 1;
	params.max_dir_depth = cf->recursion_depth;
	for (exclude_idx = 0; exclude_idx < cf->exclude_count; exclude_idx++) {
		params.excludes[exclude_idx] = cf->excludes[exclude_idx];
	}
	params.exclude_count = cf->exclude_count;
	params.null_delimiter = rsync_null_delimiter;
	params.only_list_files = false;
	params.only_list_directories = false;
	params.no_file_tracking = cf->track_files ? false : true;

	(void) watch_dir(&params);
}


/*
 * Run rsync with the given parameters, returning the exit status.
 */
static int run_rsync(const char *log_file, const char *section, const char *source, const char *destination,
		     /*@null@ */ const char *excludes_file, const char *options,
		     /*@null@ */ const char *transfer_list, bool ignore_vanished_files, bool rsync_null_delimiter,
		     const char *rsync_error_file, unsigned long timeout)
{
	char **rsync_argv;
	int rsync_argc, optidx;
	wordexp_t p;
	int rc = -1;
	pid_t rsync_pid;
	struct stat sb;
	time_t expiry_time = 0;

	memset(&p, 0, sizeof(p));
	if (wordexp(options, &p, WRDE_NOCMD) != 0) {
		error("%s: [%s]: %s", "wordexp", options, strerror(errno));
		return -1;
	}

	rsync_argc = 1;			    /* "rsync" */
	rsync_argc += p.we_wordc;	    /* options */
	if (NULL != transfer_list)
		rsync_argc += 2;	    /* --files-from */
	if (NULL != excludes_file)
		rsync_argc += 2;	    /* --exclude-from */
	if (rsync_null_delimiter)
		rsync_argc++;		    /* --from0 */
	rsync_argc += 2;		    /* source, dest */

	rsync_argv = calloc((size_t) (rsync_argc + 1), sizeof(char *));
	if (NULL == rsync_argv) {
		error("%s: %s", "calloc", strerror(errno));
		return -1;
	}

	rsync_argv[0] = "rsync";
	for (optidx = 1; optidx <= p.we_wordc; optidx++) {
		rsync_argv[optidx] = p.we_wordv[optidx - 1];
	}
	if (NULL != transfer_list) {
		rsync_argv[optidx++] = "--files-from";
		rsync_argv[optidx++] = (char *) transfer_list;
	}
	if (NULL != excludes_file) {
		rsync_argv[optidx++] = "--exclude-from";
		rsync_argv[optidx++] = (char *) excludes_file;
	}
	if (rsync_null_delimiter) {
		rsync_argv[optidx++] = "--from0";
	}
	rsync_argv[optidx++] = (char *) source;
	rsync_argv[optidx++] = (char *) destination;

	remove(rsync_error_file);

	if (timeout > 0)
		expiry_time = time(NULL) + timeout;

	rsync_pid = fork();

	if (0 == rsync_pid) {
		int fd;
		/* Child - run rsync */
		fd = open(rsync_error_file, O_CREAT | O_WRONLY, 0600);	/* flawfinder: ignore */
		/* flawfinder - see below about rsync_error_file. */
		if (fd >= 0) {
			(void) dup2(fd, 2);
			(void) close(fd);
		}
		(void) execvp("rsync", rsync_argv);
		exit(EXIT_FAILURE);
	} else if (rsync_pid < 0) {
		/* Error - output a warning */
		error("%s: %s", "fork", strerror(errno));
		rc = -1;
	} else {
		/* Parent */
		debug("(rsync) %s: %d", "process spawned", rsync_pid);
		while ((!sync_exit_now) && (0 != rsync_pid)) {
			int wait_status;
			pid_t waited_for;

			if (timeout > 0 && time(NULL) >= expiry_time) {
				debug("(rsync) %s", "timeout reached - sending SIGKILL");
				(void) kill(rsync_pid, SIGKILL);
			}

			if (timeout > 0)
				(void) alarm(1);

			waited_for = (pid_t) waitpid(rsync_pid, &wait_status, 0);

			if (waited_for < 0) {
				if ((errno == EINTR) || (errno == EAGAIN)) {
					if (timeout > 0)
						(void) alarm(0);
					continue;
				}
				log_message(log_file, "[%s] %s: %s: %s",
					    section, _("failed to wait for rsync"), "waitpid", strerror(errno));
				rc = -1;
				if (timeout > 0)
					(void) alarm(0);
				break;
			} else {
				rsync_pid = 0;
				rc = WEXITSTATUS(wait_status);
				debug("(rsync) %s: %d", "process ended, exit status", rc);
				if (timeout > 0)
					(void) alarm(0);
			}
		}
		if (0 != rsync_pid) {
			debug("(rsync) %s: %d", "killing rsync process", rsync_pid);
			(void) kill(rsync_pid, SIGTERM);
		}
	}

	free(rsync_argv);
	wordfree(&p);

	if ((stat(rsync_error_file, &sb) == 0) && (sb.st_size > 0)) {
		char *linebuf_ptr = NULL;
		size_t linebuf_size = 0;
		FILE *err_fptr;

		err_fptr = fopen(rsync_error_file, "r");	/* flawfinder: ignore */
		if (NULL == err_fptr) {
			error("%s: %s", rsync_error_file, strerror(errno));
			return rc;
		}
		/*
		 * flawfinder warns about the path being redirected but the
		 * path is under a temporary directory we just created so an
		 * attacker would need to already be running under the same
		 * user ID.
		 */

		while (0 == feof(err_fptr)) {
			ssize_t line_length = 0;

			/* Like getdelim() in collate_transfer_list(). */
			errno = 0;
			line_length = getline(&linebuf_ptr, &linebuf_size, err_fptr);
			if ((line_length < 0) || (NULL == linebuf_ptr)) {
				if (0 != errno)
					error("%s: %s", rsync_error_file, strerror(errno));
				break;
			}
			if (line_length < 1)
				continue;
			line_length--;
			linebuf_ptr[line_length] = '\0';

			log_message(log_file, "[%s] %s: %s", section, "rsync", linebuf_ptr);
		}

		if (NULL != linebuf_ptr)
			free(linebuf_ptr);

		(void) fclose(err_fptr);

		log_message(log_file, "[%s] %s: %d", section, "rsync failed with exit status", rc);
	}

	if ((24 == rc) && (ignore_vanished_files)) {
		log_message(log_file, "[%s] %s", section, "ignoring vanished files");
		rc = 0;
	}

	return rc;
}


/*
 * Update the given timestamp file if it's not NULL, creating it if it
 * doesn't exist, and setting its last-modification time to now.
 */
static void update_timestamp_file(struct sync_set_s *cf, /*@null@ */ const char *file)
{
	FILE *fptr;
	if (NULL == file)
		return;
	fptr = fopen(file, "a");	    /* flawfinder: ignore */
	/*
	 * flawfinder - negligible risk associated with possible redirection
	 * of "file" since only its timestamp is updated.
	 */
	if (NULL != fptr) {
		(void) fclose(fptr);
	} else {
		log_message(cf->log_file, "[%s] %s: %s", cf->name, file, strerror(errno));
	}
	if (utime(file, NULL) != 0) {
		log_message(cf->log_file, "[%s] %s: %s", cf->name, file, strerror(errno));
	}
}


/*
 * Run a full sync, returning nonzero on failure.
 *
 * If a sync succeeded, the full marker file's timestamp is updated, if one
 * is defined; st->last_full_sync is set to the current time;
 * st->full_sync_failures is set to 0; and st->last_full_sync_status is set
 * to point to "OK".
 *
 * The status file is updated before the sync begins.
 */
static int sync_full(struct sync_set_s *cf, struct sync_status_s *st, bool rsync_null_delimiter)
{
	int lockfd = -1;
	int rc = 0;

	if (NULL != cf->sync_lock) {
		st->action = ACTION_SYNC_FULL_WAIT;
		update_status_file(cf, st);
		lockfd = open(cf->sync_lock, O_CREAT | O_WRONLY | O_APPEND, 0600);
		if (0 > lockfd) {
			debug("(lock) %s: %s", cf->sync_lock, strerror(errno));
		} else {
			log_message(cf->log_file, "[%s] %s: %s", cf->name, _("full sync"), _("acquiring sync lock"));
			if (0 == lockf(lockfd, F_LOCK, 0)) {
				log_message(cf->log_file, "[%s] %s: %s", cf->name, _("full sync"),
					    _("sync lock acquired"));
			} else {
				error("%s: %s(%d): %s", cf->sync_lock, "lockf", lockfd, strerror(errno));
				log_message(cf->log_file, "[%s] %s: %s", cf->name, _("full sync"),
					    _("failed to acquire sync lock"));
			}
		}
	}

	st->action = ACTION_SYNC_FULL;
	update_status_file(cf, st);

	log_message(cf->log_file, "[%s] %s: %s", cf->name, _("full sync"), _("sync starting"));

	rc = run_rsync(cf->log_file, cf->name, cf->source, cf->destination,
		       st->excludes_file,
		       NULL ==
		       cf->full_rsync_opts ? "--delete -axH" :
		       cf->full_rsync_opts, NULL, cf->ignore_vanished_files, rsync_null_delimiter, st->rsync_error_file,
		       cf->full_timeout);

	log_message(cf->log_file, "[%s] %s: %s: %s", cf->name,
		    _("full sync"), _("sync ended"), rc == 0 ? _("OK") : _("FAILED"));

	if (lockfd >= 0) {
		if (0 != lockf(lockfd, F_ULOCK, 0)) {
			error("%s: %s(%d): %s", cf->sync_lock, "lockf", lockfd, strerror(errno));
		}
		close(lockfd);
	}

	if (rc == 0) {
		update_timestamp_file(cf, cf->full_marker);
		st->last_full_sync = time(NULL);
		st->full_sync_failures = 0;
		st->last_full_sync_status = STRING_STATUS_OK;
	}

	return rc;
}


/*
 * Collate a transfer list from the change queue: remove the change queue
 * entries, appending those that refer to items that still exist to the
 * transfer list.
 */
static void collate_transfer_list(struct sync_set_s *cf, bool rsync_null_delimiter)
{
	struct dirent **namelist;
	int namelist_length, idx;
	FILE *list_fptr;
	FILE *changefile_fptr;
	void *tree_root = NULL;
	char line_delimiter;

	line_delimiter = rsync_null_delimiter ? '\0' : '\n';

	if (NULL == cf->transfer_list)
		return;
	if (NULL == cf->change_queue)
		return;

	list_fptr = fopen(cf->transfer_list, "a");
	if (NULL == list_fptr) {
		error("%s: %s: %s", cf->name, cf->transfer_list, strerror(errno));
		return;
	}

	namelist_length = scandir(cf->change_queue, &namelist, NULL, alphasort);
	if (0 > namelist_length) {
		error("%s: %s: %s", "scandir", cf->change_queue, strerror(errno));
		(void) fclose(list_fptr);
		return;
	}

	for (idx = 0; idx < namelist_length; idx++) {
		char *path = NULL;
		struct stat sb;
		char *linebuf_ptr;
		size_t linebuf_size;
		unsigned int line_number;

		if ('.' == namelist[idx]->d_name[0])
			continue;

		if (asprintf(&path, "%s/%s", cf->change_queue, namelist[idx]->d_name) < 0) {
			error("%s: %s", "asprintf", strerror(errno));
			continue;
		}
		assert(NULL != path);
		if (NULL == path)
			continue;

		memset(&sb, 0, sizeof(sb));
		if (lstat(path, &sb) != 0) {
			free(path);
			continue;
		}
		if (!S_ISREG(sb.st_mode)) {
			free(path);
			continue;
		}

		changefile_fptr = fopen(path, "r");
		if (NULL == changefile_fptr) {
			debug("%s: %s", path, strerror(errno));
			remove(path);
			free(path);
			continue;
		}

		linebuf_ptr = NULL;
		linebuf_size = 0;
		line_number = 0;

		while (0 == feof(changefile_fptr)) {
			ssize_t line_length = 0;
			char *changedpath;

			errno = 0;
			line_length = getdelim(&linebuf_ptr, &linebuf_size, line_delimiter, changefile_fptr);
			if ((line_length < 0) || (NULL == linebuf_ptr)) {
				if (0 != errno)
					error("%s:%u: %s", path, line_number, strerror(errno));
				break;
			}
			line_number++;

			/*
			 * The length includes the delimiter, and so it
			 * should always be > 0.
			 */
			if (line_length < 1)
				continue;

			/*
			 * Replace the delimiter with \0 to terminate the
			 * string there.
			 */
			line_length--;
			linebuf_ptr[line_length] = '\0';

			/*
			 * Use a binary tree to keep track of entries we've
			 * seen before, so we can strip duplicates.
			 */
			if ((NULL != tree_root)
			    && (NULL != tfind(linebuf_ptr, &tree_root, compare_with_strcmp))) {
				debug("%s: %s", "skipping duplicate change item", linebuf_ptr);
				continue;
			}
			tsearch(xstrdup(linebuf_ptr), &tree_root, compare_with_strcmp);

			if (asprintf(&changedpath, "%s/%s", cf->source, linebuf_ptr) < 0) {
				error("%s: %s", "asprintf", strerror(errno));
				(void) fclose(changefile_fptr);
				break;
			}
			if (lstat(changedpath, &sb) == 0)
				fprintf(list_fptr, "%s%c", linebuf_ptr, line_delimiter);
			free(changedpath);
		}

		(void) fclose(changefile_fptr);

		if (NULL != linebuf_ptr)
			free(linebuf_ptr);

		remove(path);
		free(path);
	}

	if (NULL != tree_root)
		tdestroy(tree_root, free);

	for (idx = 0; idx < namelist_length; idx++) {
		free(namelist[idx]);
	}
	free(namelist);

	if (0 != fclose(list_fptr)) {
		error("%s: %s", cf->transfer_list, strerror(errno));
	}
}


/*
 * Run a partial sync, returning nonzero on failure.  Returns zero if there
 * is nothing to sync, or if there was a sync and it succeeded.
 *
 * If a sync was run, and it succeeded, the partial marker file's timestamp
 * is updated, if one is defined; st->last_partial_sync is set to the
 * current time; st->partial_sync_failures is set to 0; and
 * st->last_partial_sync_status is set to point to "OK".
 *
 * The status file is updated before an action is performed.
 */
static int sync_partial(struct sync_set_s *cf, struct sync_status_s *st, bool rsync_null_delimiter)
{
	struct stat sb;
	int lockfd = -1;
	int rc = 0;
	FILE *list_fptr;
	char line_delimiter;

	line_delimiter = rsync_null_delimiter ? '\0' : '\n';

	collate_transfer_list(cf, rsync_null_delimiter);

	if ((stat(cf->transfer_list, &sb) != 0) || (0 == sb.st_size)) {
		/*
		 * If there is no transfer list, there is nothing to sync.
		 */
		return 0;
	}

	if (NULL != cf->sync_lock) {
		st->action = ACTION_SYNC_PARTIAL_WAIT;
		update_status_file(cf, st);
		lockfd = open(cf->sync_lock, O_CREAT | O_WRONLY | O_APPEND, 0600);
		if (0 > lockfd) {
			debug("(lock) %s: %s", cf->sync_lock, strerror(errno));
		} else {
			log_message(cf->log_file, "[%s] %s: %s", cf->name, _("partial sync"), _("acquiring sync lock"));
			if (0 == lockf(lockfd, F_LOCK, 0)) {
				log_message(cf->log_file, "[%s] %s: %s", cf->name, _("partial sync"),
					    _("sync lock acquired"));
			} else {
				error("%s: %s(%d): %s", cf->sync_lock, "lockf", lockfd, strerror(errno));
				log_message(cf->log_file, "[%s] %s: %s", cf->name, _("partial sync"),
					    _("failed to acquire sync lock"));
			}
		}
	}

	st->action = ACTION_SYNC_PARTIAL;
	update_status_file(cf, st);

	log_message(cf->log_file, "[%s] %s: %s", cf->name, _("partial sync"), _("sync starting"));

	/*
	 * Write a copy of the transfer list to the log file (stopping at
	 * 100 lines so the log doesn't grow too much).
	 */
	list_fptr = fopen(cf->transfer_list, "r");
	if (NULL != list_fptr) {
		char *linebuf_ptr = NULL;
		size_t linebuf_size = 0;
		unsigned int line_number = 0;

		while (0 == feof(list_fptr)) {
			ssize_t line_length = 0;

			/* Same logic as in collate_transfer_list(). */
			errno = 0;
			line_length = getdelim(&linebuf_ptr, &linebuf_size, line_delimiter, list_fptr);
			if ((line_length < 0) || (NULL == linebuf_ptr)) {
				if (0 != errno)
					error("%s:%u: %s", cf->transfer_list, line_number, strerror(errno));
				break;
			}
			line_number++;
			if (line_length < 1)
				continue;
			line_length--;
			linebuf_ptr[line_length] = '\0';

			if (line_number > 100) {
				log_message(cf->log_file, "[%s]   %s", cf->name, "...");
				break;
			}

			log_message(cf->log_file, "[%s]   %s", cf->name, linebuf_ptr);
		}

		if (NULL != linebuf_ptr)
			free(linebuf_ptr);

		(void) fclose(list_fptr);
	}

	rc = run_rsync(cf->log_file, cf->name, cf->source, cf->destination,
		       st->excludes_file,
		       NULL ==
		       cf->partial_rsync_opts ? "--delete -dlptgoDH" :
		       cf->partial_rsync_opts, cf->transfer_list, cf->ignore_vanished_files, rsync_null_delimiter,
		       st->rsync_error_file, cf->partial_timeout);

	log_message(cf->log_file, "[%s] %s: %s: %s", cf->name,
		    _("partial sync"), _("sync ended"), rc == 0 ? _("OK") : _("FAILED"));

	if (lockfd >= 0) {
		if (0 != lockf(lockfd, F_ULOCK, 0)) {
			error("%s: %s(%d): %s", cf->sync_lock, "lockf", lockfd, strerror(errno));
		}
		(void) close(lockfd);
	}

	remove(cf->transfer_list);

	if (rc == 0) {
		update_timestamp_file(cf, cf->partial_marker);
		st->last_partial_sync = time(NULL);
		st->partial_sync_failures = 0;
		st->last_partial_sync_status = STRING_STATUS_OK;
	}

	return rc;
}


/*
 * Log the given message to the given file, with a timestamp.
 */
static void log_message( /*@null@ */ const char *file, const char *format, ...)
{
	FILE *fptr;
	va_list ap;
	time_t t;
	struct tm *tm;
	char tbuf[128];			 /* flawfinder: ignore */
	bool locked;

	/*
	 * flawfinder note: tbuf is only written to by strftime() which
	 * takes its size, and we enforce string termination.
	 */

	time(&t);
	tm = localtime(&t);
	tbuf[0] = '\0';
	if (0 == strftime(tbuf, sizeof(tbuf), "%Y-%m-%d %H:%M:%S", tm)) {
		tbuf[0] = '\0';
	}
	tbuf[sizeof(tbuf) - 1] = '\0';	    /* enforce termination */

#if ENABLE_DEBUGGING
	va_start(ap, format);
	if (debugging_enabled) {
		(void) fprintf(stderr, "[%s] (log) ", tbuf);
		(void) vfprintf(stderr, format, ap);	/* flawfinder: ignore */
		(void) fprintf(stderr, "\n");
	}
	va_end(ap);
#endif				/* ENABLE_DEBUGGING */

	if (NULL == file)
		return;

	fptr = fopen(file, "a");
	if (NULL == fptr) {
		create_parent_dirs(file);
		fptr = fopen(file, "a");
		if (NULL == fptr) {
			debug("(log) %s: %s", file, strerror(errno));
			return;
		}
	}

	locked = false;
	if (0 == lockf(fileno(fptr), F_LOCK, 0))
		locked = true;
	fseek(fptr, 0, SEEK_END);

	(void) fprintf(fptr, "[%s] ", tbuf);
	va_start(ap, format);
	(void) vfprintf(fptr, format, ap);  /* flawfinder: ignore */
	va_end(ap);
	(void) fprintf(fptr, "\n");

	/*
	 * flawfinder rationale: the format string for these vfprintf()
	 * calls is provided by the caller, so it's up to the caller to make
	 * sure it's safe.
	 */

	if (locked) {
		if (0 == lockf(fileno(fptr), F_ULOCK, 0))
			locked = false;
	}

	if (0 != fclose(fptr)) {
		debug("(log) %s: %s", file, strerror(errno));
	}
}


/*
 * Delete the given directory and everything in it.
 */
static void recursively_delete(const char *dir, int depth)
{
	struct dirent **namelist;
	int namelist_length, idx;
	char path[4096];		 /* flawfinder: ignore */

	/*
	 * flawfinder - path[] is zeroed before use and bounded with
	 * snprintf(); its size is sufficient for all reasonable paths.
	 */

	depth++;
	if (depth > 10)
		return;

	namelist_length = scandir(dir, &namelist, NULL, alphasort);
	if (0 > namelist_length) {
		error("%s: %s: %s", "scandir", dir, strerror(errno));
		return;
	}

	memset(path, 0, sizeof(path));

	for (idx = 0; idx < namelist_length; idx++) {
		struct stat sb;
		if (strcmp(namelist[idx]->d_name, ".") == 0)
			continue;
		if (strcmp(namelist[idx]->d_name, "..") == 0)
			continue;
		snprintf(path, sizeof(path) - 1, "%s/%s", dir, namelist[idx]->d_name);
		path[sizeof(path) - 1] = '\0';
		if (lstat(path, &sb) != 0) {
			error("%s: %s: %s", "lstat", path, strerror(errno));
			continue;
		}
		if (S_ISDIR(sb.st_mode)) {
			recursively_delete(path, depth);
		} else {
			debug("%s: %s", "removing", path);
			remove(path);
		}
	}

	for (idx = 0; idx < namelist_length; idx++) {
		free(namelist[idx]);
	}
	free(namelist);

	debug("%s: %s", "removing", dir);
	rmdir(dir);
}

/*@+mustfreefresh@ */
