/*
 * Functions for general string handling and allocation.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <ctype.h>


/*
 * Given a string length, allocate space for a new string of that length
 * (plus one byte for a null terminator), tracked in the state->strings
 * array.  The returned pointer should not be passed to free(), as freeing
 * the string is handled when the state is destroyed.
 *
 * The original string is copied into the new area.  If the original string
 * is NULL, the space is allocated and zeroed.
 *
 * Returns NULL if the allocation failed.
 */
/*@-compmempass@*/
/*@null@ */
/*@dependent@ */
char *stringCopy(struct scwState *state, /*@null@ */ const char *string, size_t length)
{
	/*@dependent@ */ char *newString;

	/* Extend the array if it's full. */
	/*@-branchstate@ */
	if ((NULL == state->strings) || (state->countStrings >= state->stringsArraySize)) {
		/*@keep@ */ char **newStringsArray;
		size_t newArraySize;

		newArraySize = state->stringsArraySize + 1024;

		/*@-keeptrans@ */
		newStringsArray = realloc(state->strings, newArraySize * sizeof(char *));
		/*@+keeptrans@ */

		/*@-usereleased *//*@-compdef@ */
		if (NULL == newStringsArray) {
			perror(PACKAGE_NAME);
			return NULL;
		}
		/*@+usereleased *//*@+compdef@ */
		state->strings = newStringsArray;
		state->stringsArraySize = newArraySize;
	}
	/*@+branchstate@ */

	/* Allocate the copy of the string. */
	newString = malloc(1 + length);
	if (NULL == newString) {
		perror(PACKAGE_NAME);
		return NULL;
	}

	/* Clear the newly allocated memory. */
	memset(newString, 0, 1 + length);

	/* Copy the original string, if it's not NULL. */
	if (NULL != string)
		memcpy(newString, string, length);	/* flawfinder: ignore */
	/*
	 * flawfinder warns that memcpy() doesn't do bounds checking, but
	 * since we have just allocated the buffer we're copying into, we
	 * know it is big enough to receive "length" bytes.
	 */

	/* Add the string to the array, so we can free it at the end. */
	state->strings[state->countStrings++] = newString;

	return newString;
}

/*@+compmempass@*/


/*
 * Append the second string to the first string, ensuring that it fits in
 * the buffer, and is null-terminated.  Returns the length of the resultant
 * string.
 */
size_t stringAppend(size_t bufferSize, char *destination, size_t destinationLength, const char *source,
		    size_t sourceLength)
{
	size_t bytesLeftInBuffer, bytesToCopy;

	if (destinationLength >= bufferSize - 1)
		return destinationLength;

	bytesLeftInBuffer = bufferSize - destinationLength;
	bytesLeftInBuffer--;		    /* leave room for the terminating \0 */

	bytesToCopy = sourceLength;
	if (bytesToCopy > bytesLeftInBuffer)
		bytesToCopy = bytesLeftInBuffer;

	if (bytesToCopy > 0)
		memmove(destination + destinationLength, source, bytesToCopy);	/* flawfinder: ignore */
	/* flawfinder - we're explicitly limiting it to fit in the buffer. */

	destinationLength += bytesToCopy;
	destination[destinationLength] = '\0';

	return destinationLength;
}


/*
 * Return true if the first string is at least as long as the second, and
 * its start matches the entirety of the second.  Both strings must be
 * null-terminated.
 */
bool stringStartsWith(const char *string, const char *match)
{
	size_t stringLength, matchLength;

	stringLength = strlen(string);	    /* flawfinder: ignore */
	matchLength = strlen(match);	    /* flawfinder: ignore */
	/* flawfinder - these are null-terminated strings. */

	if (stringLength < matchLength)
		return false;
	if (0 == strncmp(string, match, matchLength))
		return true;
	return false;
}


/*
 * Return a static buffer containing a timestamp string.
 *
 * If "when" is zero, the string "(never)" is returned.
 */
char *timeAndDateString(time_t when)
{
	struct tm *brokenDownTime;
	/*@only@ */
	static char string[256];	 /* flawfinder: ignore */

	/*
	 * flawfinder warns about overflowing the buffer.  We explicitly
	 * clear the buffer and then only use it with strftime(), which we
	 * tell about the buffer size, so there is no risk.
	 */

	memset(string, 0, sizeof(string));

	if (0 == when) {
		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */
		(void) snprintf(string, sizeof(string), "%s", _("(never)"));
		/*@+mustfreefresh@ */
		return string;
	}

	brokenDownTime = gmtime(&when);
	if (NULL == brokenDownTime) {
		fprintf(stderr, "%s: %s", PACKAGE_NAME, strerror(errno));
		return string;
	}

	(void) strftime(string, sizeof(string), "%Y-%m-%dT%H:%M:%SZ", brokenDownTime);

	return string;
}


/*
 * Return a static buffer containing a time period expressed as a string.
 */
char *timePeriodString(int period)
{
	/*@only@ */
	static char string[256];	 /* flawfinder: ignore */

	/*
	 * flawfinder warns about overflowing the buffer.  We explicitly
	 * clear the buffer and then only use it with snprintf(), which we
	 * tell about the buffer size, so there is no risk.
	 */

	memset(string, 0, sizeof(string));

	/*@-mustfreefresh@ */
	/* splint note: gettext triggers a warning we can't resolve. */

	if (period < 0) {
		(void) snprintf(string, sizeof(string), "%ds (%s)", period, _("negative amount of time"));
	} else if (0 == period) {
		(void) snprintf(string, sizeof(string), "%ds (%s)", period, _("no time"));
	} else if (period < 60) {
		(void) snprintf(string, sizeof(string), "%ds", period);
	} else if (period < 3600) {
		(void) snprintf(string, sizeof(string), "%ds (%dm %ds)", period, period / 60, period % 60);
	} else if (period < 86400) {
		(void) snprintf(string, sizeof(string), "%ds (%dh %dm %ds)", period, period / 3600,
				(period % 3600) / 60, period % 60);
	} else if (period < 604800) {
		(void) snprintf(string, sizeof(string), "%ds (%dd %dh %dm %ds)", period, period / 86400,
				(period % 86400) / 3600, (period % 3600) / 60, period % 60);
	} else {
		(void) snprintf(string, sizeof(string), "%ds (%dw %dd %dh %dm %ds)", period, period / 604800,
				(period % 604800) / 86400, (period % 86400) / 3600, (period % 3600) / 60, period % 60);
	}

	/*@+mustfreefresh@ */

	return string;
}


/*
 * Populate a buffer with a copy of a string, escaped for JSON, and return
 * the buffer.  The buffer will always be null-terminated.  If the input
 * string is NULL, the buffer will contain an empty string.
 *
 * If "lengthPointer" is not NULL, the length of the resultant string is
 * placed in it.
 *
 * This treats the input string as a stream of bytes, so is unaware of
 * multi-byte characters.  Any unexpected byte will be escaped as \uXXXX,
 * where XXXX is the byte value as 4-digit hex.
 */
/*@dependent@ */
char *jsonEscapeString( /*@dependent@ */ char *buffer, size_t bufferSize, /*@null@ */ const char *string, size_t length,
		       /*@null@ */ size_t *lengthPointer)
{
	size_t readPosition, writePosition;
	const unsigned char *unsignedString;

	unsignedString = (unsigned char *) string;

	buffer[0] = '\0';
	if (NULL != lengthPointer)
		*lengthPointer = 0;
	if (NULL == string)
		return buffer;

	for (readPosition = 0, writePosition = 0; readPosition < length && writePosition < bufferSize - 1;
	     readPosition++) {
		char readChar = string[readPosition];
		unsigned int charAsUnsignedInt = (unsigned int) (unsignedString[readPosition]);
		switch (readChar) {
		case '"':
		case '\\':
			if (writePosition > bufferSize - 3) {
				readPosition = length;
			} else {
				buffer[writePosition++] = '\\';
				buffer[writePosition++] = readChar;
			}
			break;
		case '\n':
			if (writePosition > bufferSize - 3) {
				readPosition = length;
			} else {
				buffer[writePosition++] = '\\';
				buffer[writePosition++] = 'n';
			}
			break;
		case '\t':
			if (writePosition > bufferSize - 3) {
				readPosition = length;
			} else {
				buffer[writePosition++] = '\\';
				buffer[writePosition++] = 't';
			}
			break;
		case '\r':
			if (writePosition > bufferSize - 3) {
				readPosition = length;
			} else {
				buffer[writePosition++] = '\\';
				buffer[writePosition++] = 'r';
			}
			break;
		default:
			if (readChar >= ' ' && readChar < (char) 127) {
				buffer[writePosition++] = readChar;
			} else if (writePosition > bufferSize - 6) {
				readPosition = length;
			} else {
				writePosition +=
				    snprintf(buffer + writePosition, bufferSize - writePosition, "\\u%04x",
					     charAsUnsignedInt);
			}
			break;
		}
	}

	if (writePosition >= bufferSize)
		writePosition = bufferSize - 1;
	buffer[writePosition] = '\0';

	if (NULL != lengthPointer)
		*lengthPointer = writePosition;

	return buffer;
}


/*
 * Populate a buffer with a copy of a string, URL-encoded, and return the
 * buffer.  The buffer will always be null-terminated.  If the input string
 * is NULL, the buffer will contain an empty string.
 *
 * If "lengthPointer" is not NULL, the length of the resultant string is
 * placed in it.
 *
 * The URL-encoding method is to just escape any non-alphanumeric bytes as
 * %xx, where xx is the value of the byte in hex.  It is not aware of
 * multi-byte characters.
 */
/*@dependent@ */
char *urlEncodeString( /*@dependent@ */ char *buffer, size_t bufferSize, /*@null@ */ const char *string, size_t length,
		      /*@null@ */ size_t *lengthPointer)
{
	size_t readPosition, writePosition;
	const unsigned char *unsignedString;

	unsignedString = (unsigned char *) string;

	buffer[0] = '\0';
	if (NULL != lengthPointer)
		*lengthPointer = 0;
	if (NULL == string)
		return buffer;

	for (readPosition = 0, writePosition = 0; readPosition < length && writePosition < bufferSize - 1;
	     readPosition++) {
		char readChar = string[readPosition];
		unsigned int charAsUnsignedInt = (unsigned int) (unsignedString[readPosition]);
		if (isalnum(readChar)) {
			buffer[writePosition++] = readChar;
		} else if (writePosition > bufferSize - 3) {
			readPosition = length;
		} else {
			writePosition +=
			    snprintf(buffer + writePosition, bufferSize - writePosition, "%%%02x", charAsUnsignedInt);
		}
	}

	if (writePosition >= bufferSize)
		writePosition = bufferSize - 1;
	buffer[writePosition] = '\0';

	if (NULL != lengthPointer)
		*lengthPointer = writePosition;

	return buffer;
}
