/*
 * Functions for parsing settings.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <unistd.h>
#if HAVE_POSIX_FADVISE
#include <fcntl.h>
#endif
#include <sys/utsname.h>

#ifdef ENABLE_DEBUGGING
/* Debugging output showing all settings. */
void debugOutputAllSettings(const char *description, struct scwSettings *settings)
{

	/* Use the preprocessor to avoid repetitive commands. */
#define debugShowSettingValues(item) { \
	debugShowFragmentValue(description, settings->item.rawValue, settings->item.rawLength); \
	debugShowFragmentValue(description, settings->item.expandedValue, settings->item.expandedLength); \
}
#define debugShowSettingValueArray(itemArray, countVar) { \
	size_t arrayIndex; \
	debugShowIntegerValue(description, settings->countVar); \
	for (arrayIndex = 0; arrayIndex < settings->countVar; arrayIndex++) { \
		debugShowFragmentValue(description, settings->itemArray[arrayIndex].rawValue, settings->itemArray[arrayIndex].rawLength); \
		debugShowFragmentValue(description, settings->itemArray[arrayIndex].expandedValue, settings->itemArray[arrayIndex].expandedLength); \
	} \
}

	debugShowSettingValues(itemsDir);
	debugShowSettingValues(metricsDir);
	debugShowSettingValues(checkLockFile);
	debugShowSettingValues(sendmail);
	debugShowSettingValues(transmitForm);
	debugShowSettingValues(transmitJson);
	debugShowSettingValues(userConfigFile);
	debugShowSettingValues(itemListFile);
	debugShowSettingValues(crontabFile);
	debugShowSettingValues(updateLockFile);
	debugShowSettingValues(description);
	debugShowSettingValues(command);
	debugShowSettingValues(ambiguousExitStatus);
	debugShowIntegerValue(description, settings->numAmbiguousExitStatus);
	debugShowSettingValueArray(schedule, countSchedules);
	debugShowBooleanValue(description, settings->clearedSchedules);
	debugShowSettingValues(randomDelay);
	debugShowIntegerValue(description, settings->numRandomDelay);
	debugShowSettingValues(maxRunTime);
	debugShowIntegerValue(description, settings->numMaxRunTime);
	debugShowSettingValues(prerequisite);
	debugShowSettingValues(minInterval);
	debugShowIntegerValue(description, settings->numMinInterval);
	debugShowSettingValues(successInterval);
	debugShowIntegerValue(description, settings->numSuccessInterval);
	debugShowSettingValues(concurrencyWait);
	debugShowIntegerValue(description, settings->numConcurrencyWait);
	debugShowSettingValues(silentConcurrency);
	debugShowBooleanValue(description, settings->flagSilentConcurrency);
	debugShowSettingValues(ignoreOverrun);
	debugShowBooleanValue(description, settings->flagIgnoreOverrun);
	debugShowSettingValueArray(dependsOn, countDependencies);
	debugShowBooleanValue(description, settings->clearedDependencies);
	debugShowSettingValues(dependencyWait);
	debugShowIntegerValue(description, settings->numDependencyWait);
	debugShowSettingValues(silentDependency);
	debugShowBooleanValue(description, settings->flagSilentDependency);
	debugShowSettingValueArray(conflictsWith, countConflicts);
	debugShowBooleanValue(description, settings->clearedConflicts);
	debugShowSettingValues(conflictWait);
	debugShowIntegerValue(description, settings->numConflictWait);
	debugShowSettingValues(silentConflict);
	debugShowBooleanValue(description, settings->flagSilentConflict);
	debugShowSettingValues(statusMode);
	debugShowSettingValues(statusTag);
	debugShowSettingValues(timestampUTC);
	debugShowBooleanValue(description, settings->flagTimestampUTC);
	debugShowSettingValues(httpInterval);
	debugShowIntegerValue(description, settings->numHTTPInterval);
	debugShowSettingValues(httpTimeout);
	debugShowIntegerValue(description, settings->numHTTPTimeout);
	debugShowSettingValues(sharedSecret);
	debugShowSettingValues(emailMaxBodySize);
	debugShowIntegerValue(description, settings->numEmailMaxBodySize);
	debugShowSettingValues(emailBodyText);
	debugShowSettingValues(emailAttachmentName);
	debugShowSettingValues(emailSender);
	debugShowSettingValues(emailSubject);
	debugShowSettingValueArray(outputMap, countOutputMaps);
	debugShowBooleanValue(description, settings->clearedOutputMaps);
	debugShowSettingValues(receiverStrategy);
}
#endif


/*
 * Output all settings to stdout.  As a side effect, all raw values are
 * expanded if they have not already been.
 */
void outputAllSettings(struct scwState *state, struct scwSettings *settings)
{
	/*@-mustfreefresh@ */
	/* splint note: gettext triggers a warning we can't resolve. */

	/* Use the preprocessor to avoid repetitive commands. */
#define outputFragmentValue(settingName, item, length) printf("%s = %.*s\n", settingName, (int) length, NULL == item ? "" : item)
#define outputSettingValue(settingName, item) { \
	if (NULL == settings->item.expandedValue && settings->item.rawValue != NULL) \
		expandRawValue(state, &(settings->item)); \
	outputFragmentValue(settingName, settings->item.expandedValue, settings->item.expandedLength); \
}
#define outputSettingValueArray(settingName, itemArray, countVar) { \
	size_t arrayIndex; \
	for (arrayIndex = 0; arrayIndex < settings->countVar; arrayIndex++) { \
		if (NULL == settings->itemArray[arrayIndex].expandedValue && settings->itemArray[arrayIndex].rawValue != NULL) \
			expandRawValue(state, &(settings->itemArray[arrayIndex])); \
		outputFragmentValue(settingName, settings->itemArray[arrayIndex].expandedValue, settings->itemArray[arrayIndex].expandedLength); \
	} \
}

	printf("# %s\n", _("Per-user configuration"));
	outputSettingValue("ItemsDir", itemsDir);
	outputSettingValue("MetricsDir", metricsDir);
	outputSettingValue("CheckLockFile", checkLockFile);
	outputSettingValue("Sendmail", sendmail);
	outputSettingValue("TransmitForm", transmitForm);
	outputSettingValue("TransmitJSON", transmitJson);

	printf("\n");
	printf("# %s\n", _("Global configuration"));
	outputSettingValue("UserConfigFile", userConfigFile);
	outputSettingValue("ItemListFile", itemListFile);
	outputSettingValue("CrontabFile", crontabFile);
	outputSettingValue("UpdateLockFile", updateLockFile);

	printf("\n");
	printf("# %s\n", _("Item settings"));
	outputSettingValue("Description", description);
	outputSettingValue("Command", command);
	outputSettingValue("AmbiguousExitStatus", ambiguousExitStatus);
	outputSettingValueArray("Schedule", schedule, countSchedules);
	outputSettingValue("RandomDelay", randomDelay);
	outputSettingValue("MaxRunTime", maxRunTime);
	outputSettingValue("Prerequisite", prerequisite);
	outputSettingValue("MinInterval", minInterval);
	outputSettingValue("SuccessInterval", successInterval);
	outputSettingValue("ConcurrencyWait", concurrencyWait);
	outputSettingValue("SilentConcurrency", silentConcurrency);
	outputSettingValue("IgnoreOverrun", ignoreOverrun);
	outputSettingValueArray("DependsOn", dependsOn, countDependencies);
	outputSettingValue("DependencyWait", dependencyWait);
	outputSettingValue("SilentDependency", silentDependency);
	outputSettingValueArray("ConflictsWith", conflictsWith, countConflicts);
	outputSettingValue("ConflictWait", conflictWait);
	outputSettingValue("SilentConflict", silentConflict);
	outputSettingValue("StatusMode", statusMode);
	outputSettingValue("StatusTag", statusTag);
	outputSettingValue("TimestampUTC", timestampUTC);
	outputSettingValue("HTTPInterval", httpInterval);
	outputSettingValue("HTTPTimeout", httpTimeout);
	outputSettingValue("SharedSecret", sharedSecret);
	outputSettingValue("EmailMaxBodySize", emailMaxBodySize);
	outputSettingValue("EmailBodyText", emailBodyText);
	outputSettingValue("EmailAttachmentName", emailAttachmentName);
	outputSettingValue("EmailSender", emailSender);
	outputSettingValue("EmailSubject", emailSubject);
	outputSettingValueArray("OutputMap", outputMap, countOutputMaps);
	outputSettingValue("ReceiverStrategy", receiverStrategy);

	/*@+mustfreefresh@ */
}


/*
 * Return true if the string is a valid item name.
 */
bool validateItemName(const char *string, size_t length)
{
	size_t checkIndex;

	/* Empty string is not valid. */
	if (length < 1)
		return false;

	/* Strings "." and ".." are not allowed. */
	if (1 == length && '.' == string[0])
		return false;
	if (2 == length && '.' == string[0] && '.' == string[1])
		return false;

	/* Any '/' or space characters are not allowed. */
	for (checkIndex = 0; checkIndex < length; checkIndex++) {
		if ('/' == string[checkIndex])
			return false;
		if (isspace(string[checkIndex]))
			return false;
	}

	return true;
}


/*
 * Wrapper for strncmp() to get around it being a macro that doesn't work
 * with STATIC_STRING() on CentOS 5, 6, and 7.
 */
static int do_strncmp(const char *a, const char *b, size_t n)
{
	return strncmp(a, b, n);
}

/*
 * Return true if the string looks like a valid crontab(5) schedule.
 */
static bool validateSchedule(const char *string, size_t length)
{
	struct utsname unameInfo;
	size_t sysnameLength;
	size_t checkPosition = 0;
	size_t fieldNumber = 0;
	typedef struct {
		const /*@null@ */ char *nickname;
		const /*@null@ */ char *sys;
	} nickname_t;
	typedef const /*@null@ */ char *word_t;
	static nickname_t nicknames[] = {
		{ "@reboot", NULL },
		{ "@yearly", NULL },
		{ "@annually", NULL },
		{ "@monthly", NULL },
		{ "@weekly", NULL },
		{ "@daily", NULL },
		{ "@midnight", NULL },
		{ "@hourly", NULL },
		{ "@every_minute", "FreeBSD" },
		{ "@every_second", "FreeBSD" },
		{ NULL, NULL },
	};
	static word_t monthNames[] = {
		"Jan",
		"Feb",
		"Mar",
		"Apr",
		"May",
		"Jun",
		"Jul",
		"Aug",
		"Sep",
		"Oct",
		"Nov",
		"Dec",
		NULL
	};
	static word_t dayNames[] = {
		"Mon",
		"Tue",
		"Wed",
		"Thu",
		"Fri",
		"Sat",
		"Sun",
		NULL
	};
	bool isNetBSD = false;
	bool isOpenBSD = false;

	memset(&unameInfo, 0, sizeof(unameInfo));
	(void) uname(&unameInfo);
	sysnameLength = strlen(unameInfo.sysname);	/* flawfinder: ignore */
	/*
	 * flawfinder rationale: the objection to strlen() is that it does
	 * not handle strings that are not null-terminated - but the fields
	 * that uname() returns are null-terminated, so we ignore the
	 * warning.
	 */

	/* Special case - one field starting with "@", e.g. "@daily". */
	if (length > 1 && '@' == string[0]) {
		unsigned int nicknameIndex;
		bool matchedNickname = false;

		for (nicknameIndex = 0; NULL != nicknames[nicknameIndex].nickname; nicknameIndex++) {
			size_t nameLength = strlen(nicknames[nicknameIndex].nickname);	/* flawfinder: ignore */
			size_t sysLength = 0;
			if (NULL != nicknames[nicknameIndex].sys)
				sysLength = strlen(nicknames[nicknameIndex].sys);	/* flawfinder: ignore */
			/*
			 * flawfinder: these are static strings, compiled in
			 * as null-terminated, so strlen() won't overrun.
			 */

			if (nameLength != length)
				continue;
			if (0 != strncmp(string, nicknames[nicknameIndex].nickname, length))
				continue;
			if ((0 != sysLength) && (sysnameLength != sysLength))
				continue;
			if ((0 != sysLength) && (NULL != nicknames[nicknameIndex].sys)
			    && (0 != strncmp(unameInfo.sysname, nicknames[nicknameIndex].sys, sysnameLength)))
				continue;
			matchedNickname = true;
			break;
		}

		if (matchedNickname)
			return true;

		return false;
	}

	if (sysnameLength == 6 && 0 == do_strncmp(unameInfo.sysname, STATIC_STRING("NetBSD"))) {
		isNetBSD = true;
	} else if (sysnameLength == 7 && 0 == do_strncmp(unameInfo.sysname, STATIC_STRING("OpenBSD"))) {
		isOpenBSD = true;
	}

	/*
	 * The first three fields can only contain digits, '*', ',', '-', or
	 * '/'.  The next two can also contain letters (month and weekday
	 * names).  There can only be 5 fields.
	 *
	 * On NetBSD systems, any field may be '?', or be prefixed with '?',
	 * to indicate a random choice.
	 *
	 * On OpenBSD systems, any field can be '~', or use '~' in place of
	 * '-', to indicate a random choice.  Where used in a range, the
	 * start or end of the range may be omitted, like "~30".
	 *
	 * The range of allowed numbers is 0-59 for the first field, 0-23 for
	 * the second, 1-31 for the third, 1-12 for the fourth, and 0-7 for
	 * the fifth.
	 */
	while (checkPosition < length) {
		unsigned int rangeMin = 0;
		unsigned int rangeMax = 59;
		/*@null@ */ word_t *wordList = NULL;
		size_t fieldStartPosition, itemStartPosition, itemLength;
		unsigned int currentNumber;
		bool inNumber, inRange, inWord, inDivisor, inRandomRange, requireRange, rangeSeen;

		/* Skip spaces to the next field. */
		while (checkPosition < length && isspace(string[checkPosition]))
			checkPosition++;

		fieldNumber++;

		/*
		 * Determine what range of values are allowed for this field.
		 */
		switch (fieldNumber) {
		case 1:
			/* Word 1 - the minute, 0-59. */
			rangeMin = 0;
			rangeMax = 59;
			wordList = NULL;
			break;
		case 2:
			/* Word 2 - the hour, 0-23. */
			rangeMin = 0;
			rangeMax = 23;
			wordList = NULL;
			break;
		case 3:
			/* Word 3 - the day of the month, 1-31. */
			rangeMin = 1;
			rangeMax = 31;
			wordList = NULL;
			break;
		case 4:
			/* Word 4 - month name, or 1-12. */
			rangeMin = 1;
			rangeMax = 12;
			wordList = monthNames;
			break;
		case 5:
			/* Word 5 - name of day of week, or 0-7. */
			rangeMin = 0;
			rangeMax = 7;
			wordList = dayNames;
			break;
		default:
			break;
		}

		currentNumber = 0;
		fieldStartPosition = checkPosition;
		itemStartPosition = checkPosition;
		itemLength = 0;
		inNumber = false;
		inRange = false;
		inWord = false;
		inDivisor = false;
		inRandomRange = false;
		requireRange = false;
		rangeSeen = false;

		while (checkPosition < length && !isspace(string[checkPosition])) {
			char currentChar = string[checkPosition++];
			bool checkNumber = false;
			bool checkWord = false;

			if (isdigit(currentChar)) {
				/* Build up the current number if this is a digit. */
				if (!inNumber) {
					itemStartPosition = checkPosition - 1;
					itemLength = 0;
				}
				currentNumber = 10 * currentNumber;
				currentNumber += (unsigned int) (currentChar - '0');
				inNumber = true;
				itemLength++;

			} else if (isalpha(currentChar)) {
				/* Build up the word if this is alphabetic. */
				if (inNumber) {
					/* Can't have a number in a word. */
					debug("%s %d: %s: [%c]", "field", fieldNumber, "number found mid-word",
					      currentChar);
					return false;
				}
				if (NULL == wordList) {
					/* Words not allowed in some fields. */
					debug("%s %d: %s: [%c]", "field", fieldNumber, "forbidden character",
					      currentChar);
					return false;
				}
				if (!inWord) {
					itemStartPosition = checkPosition - 1;
					itemLength = 0;
				}
				inWord = true;
				itemLength++;

			} else if ('*' == currentChar) {
				/* Must be just one '*' by itself. */
				if (inNumber || inWord || itemLength > 0) {
					debug("%s %d: %s: [%c]", "field", fieldNumber, "forbidden character",
					      currentChar);
					return false;
				}

				/*
				 * Pretend that this is a random range, so
				 * it can be followed by a divisor, and can
				 * omit the end value.
				 */
				inRange = true;
				inRandomRange = true;
				rangeSeen = true;

				itemLength++;

			} else if (',' == currentChar) {
				/* List separator - check preceding value. */
				if (inNumber) {
					if (!inDivisor)
						checkNumber = true;
					inNumber = false;
					inRange = false;
					rangeSeen = false;
					inDivisor = false;
				} else if (inWord) {
					if (!inDivisor)
						checkWord = true;
					inWord = false;
					inRange = false;
					rangeSeen = false;
					inDivisor = false;
				} else {
					/* A list separator ',' has to follow something. */
					debug("%s %d: %s", "field", fieldNumber, "list with empty value");
					return false;
				}

			} else if ('-' == currentChar) {
				/* Range separator - check preceding value. */
				if (inRange) {
					/* Can't have a range within a range. */
					debug("%s %d: %s", "field", fieldNumber, "range within a range");
					return false;
				} else if (inNumber) {
					/* Check the preceding number. */
					checkNumber = true;
					inNumber = false;
					inRange = true;
					rangeSeen = true;
					inRandomRange = false;
					inDivisor = false;
				} else if (inWord) {
					/* Check the preceding word. */
					checkWord = true;
					inWord = false;
					inRange = true;
					rangeSeen = true;
					inRandomRange = false;
					inDivisor = false;
				} else {
					/* Range separator must come after a value. */
					debug("%s %d: %s", "field", fieldNumber, "range with empty start");
					return false;
				}

			} else if ('?' == currentChar && isNetBSD) {
				/* NetBSD allows a '?' prefix. */
				if (itemStartPosition > fieldStartPosition) {
					/* Must only be at the start of the field. */
					debug("%s %d: %s", "field", fieldNumber, "? prefix not at start of field");
					return false;
				}

				/* If followed by a digit, it must be a range. */
				if ((checkPosition < length) && isdigit(string[checkPosition])) {
					requireRange = true;
				}

				itemStartPosition++;

			} else if ('~' == currentChar && isOpenBSD) {
				/* OpenBSD allows '~' like '-' with randomness. */
				if (inRange) {
					/* Can't have a range within a range. */
					debug("%s %d: %s", "field", fieldNumber, "random range within a range");
					return false;
				} else if (inNumber) {
					/* Check the preceding number. */
					checkNumber = true;
					inNumber = false;
					inRange = true;
					rangeSeen = true;
					inRandomRange = true;
					inDivisor = false;
				} else if (inWord) {
					/* Check the preceding word. */
					checkWord = true;
					inWord = false;
					inRange = true;
					rangeSeen = true;
					inRandomRange = true;
					inDivisor = false;
				} else {
					/* The '~' is allowed to lack a preceding number. */
				}

			} else if ('/' == currentChar) {
				/* Divisor - must follow a value. */
				if (inDivisor) {
					debug("%s %d: %s", "field", fieldNumber, "divisor within a divisor");
					return false;
				} else if (itemLength < 1) {
					debug("%s %d: %s", "field", fieldNumber, "divisor must follow a value");
					return false;
				} else if (!inRange) {
					debug("%s %d: %s", "field", fieldNumber, "divisor must follow a range or '*'");
					return false;
				}
				/* Must be followed by a number. */
				if (checkPosition >= length || !isdigit(string[checkPosition])) {
					debug("%s %d: %s", "field", fieldNumber,
					      "divisor must be followed by a number");
					return false;
				}

				inRange = false;
				inRandomRange = false;
				inDivisor = true;

				/* Check the preceding value if it was a number or a word. */
				if (inNumber) {
					checkNumber = true;
					inNumber = false;
				} else if (inWord) {
					checkWord = true;
					inWord = false;
				}

			} else {
				/* Unknown character. */
				debug("%s %d: %s: [%c]", "field", fieldNumber, "forbidden character", currentChar);
				return false;
			}

			if ((checkPosition >= length) || isspace(string[checkPosition])) {
				if (inNumber && !inDivisor) {
					checkNumber = true;
				} else if (inWord && !inDivisor) {
					checkWord = true;
				} else if (inRange && !inRandomRange) {
					debug("%s %d: %s", "field", fieldNumber, "no range end specified");
					return false;
				}
			}

			/*
			 * If we've flagged that the number or the word
			 * needs to be checked for validity, check it, and
			 * reset the "itemStartPosition" and "itemLength" so
			 * they begin on the next part of this field.
			 */

			if (checkNumber) {
				debug("%s %d: @%d+%d: %s=%u", "field", fieldNumber, itemStartPosition, itemLength,
				      "value to check", currentNumber);

				/* Check the number is allowed. */
				if (currentNumber < rangeMin || currentNumber > rangeMax) {
					debug("%s %d: %s: [%u: %u-%u]", "field", fieldNumber, "value is out of range",
					      currentNumber, rangeMin, rangeMax);
					return false;
				}

				/* Check whether a required range was used. */
				if (requireRange && !rangeSeen) {
					debug("%s %d: %s", "field", fieldNumber, "range required but not provided");
					return false;
				}

				currentNumber = 0;
				itemStartPosition = checkPosition;
				itemLength = 0;

			} else if (checkWord) {
				size_t wordIndex;
				bool wordAccepted;

				debug("%s %d: @%d,%d: %s=%.*s", "field", fieldNumber, itemStartPosition, itemLength,
				      "word to check", itemLength, &(string[itemStartPosition]));

				/* Check the word is allowed. */
				if (NULL == wordList) {
					debug("%s %d: %s", "field", fieldNumber,
					      "word found but no permitted word list");
					return false;
				}

				/* Words always must be 3 letters or more. */
				if (itemLength < 3) {
					debug("%s %d: %s", "field", fieldNumber, "word is too short");
					return false;
				}

				wordAccepted = false;
				for (wordIndex = 0; NULL != wordList[wordIndex]; wordIndex++) {
					/*@-unrecog@ *//* splint doesn't know about strncasecmp(). */
					if (0 != strncasecmp(wordList[wordIndex], &(string[itemStartPosition]), 3))
						continue;
					/*@+unrecog@ */
					wordAccepted = true;
					break;
				}

				if (!wordAccepted) {
					debug("%s %d: %s", "field", fieldNumber, "unknown word found");
					return false;
				}

				/* Check whether a required range was used. */
				if (requireRange && !rangeSeen) {
					debug("%s %d: %s", "field", fieldNumber, "range required but not provided");
					return false;
				}

				itemStartPosition = checkPosition;
				itemLength = 0;
			}
		}
	}

	debug("%s: %d", "words found", fieldNumber);

	if (5 == fieldNumber)
		return true;

	return false;
}


/*
 * Return true if the string is a valid status mode.
 */
static bool validateStatusMode(const char *string, size_t length)
{
	/*@-nullassign@ */
	char *modes[] = { "stdout", "stderr", "fd", NULL };
	/*@+nullassign@ */
	int modeIndex;

	for (modeIndex = 0; NULL != modes[modeIndex]; modeIndex++) {
		if (strlen(modes[modeIndex]) != length)	/* flawfinder: ignore */
			continue;
		/*
		 * flawfinder warns about strlen() in case it's given an
		 * unterminated string, but the strings in this structure
		 * are all null-terminated, so we can ignore that.
		 */
		/*@-unrecog@ */
		/* splint doesn't recognise strncasecmp(). */
		if (0 != strncasecmp(string, modes[modeIndex], length))
			continue;
		/*@+unrecog@ */
		return true;
	}

	return false;
}


/*
 * Return true if the string is a valid output map spec.
 */
static bool validateOutputMap(const char *string, size_t length)
{
	if (0 == parseOutputMap(NULL, string, length, NULL))
		return true;
	return false;
}


/*
 * Return true if the string is a valid time period, and if so, write the
 * number of seconds to *integerPointer.
 */
static bool parsePeriod(const char *string, size_t length, unsigned int *integerPointer)
{
	unsigned int totalValue, currentNumber;
	size_t position;

	totalValue = 0;
	currentNumber = 0;
	position = 0;

	while (position < length) {
		/* Skip past whitespace. */
		while (position < length && isspace(string[position]))
			position++;
		if (position >= length)
			break;

		/* If the first thing isn't a digit, fail as invalid. */
		if (string[position] < '0' || string[position] > '9')
			return false;
		while (position < length && (string[position] >= '0' && string[position] <= '9')) {
			currentNumber = (10 * currentNumber) + (unsigned int) (string[position] - '0');
			position++;
		}

		/* Skip past whitespace. */
		while (position < length && isspace(string[position]))
			position++;
		if (position >= length) {
			totalValue += currentNumber;
			currentNumber = 0;
			break;
		}

		/* Check for a suffix. */
		if ('s' == string[position] || 'S' == string[position]) {
			/* No action - seconds. */
		} else if ('m' == string[position] || 'M' == string[position]) {
			/* Minutes. */
			currentNumber = currentNumber * 60;
		} else if ('h' == string[position] || 'H' == string[position]) {
			/* Hours. */
			currentNumber = currentNumber * 3600;
		} else if ('d' == string[position] || 'D' == string[position]) {
			/* Days. */
			currentNumber = currentNumber * 86400;
		} else if ('w' == string[position] || 'W' == string[position]) {
			/* Weeks. */
			currentNumber = currentNumber * 86400 * 7;
		}

		/* Add the resultant number to the total. */
		totalValue += currentNumber;
		currentNumber = 0;

		/* Skip past any more letters. */
		while (position < length && isalpha(string[position]))
			position++;
		if (position >= length)
			break;
	}

	*integerPointer = totalValue;
	return true;
}


/*
 * Return true if the string is a valid byte count, and if so, write the
 * number of bytes to *integerPointer.
 */
static bool parseByteCount(const char *string, size_t length, unsigned int *integerPointer)
{
	unsigned int totalValue, currentNumber;
	size_t position;

	totalValue = 0;
	currentNumber = 0;
	position = 0;

	/* This is broadly similar to parsePeriod(). */

	while (position < length) {
		/* Skip past whitespace. */
		while (position < length && isspace(string[position]))
			position++;
		if (position >= length)
			break;

		/*
		 * If the first thing is a negative sign or the word
		 * "unlimited", succeed, and set the value to -1 cast to an
		 * unsigned integer.
		 */
		if (('-' == string[position])
		    || (((position + 9) <= length)
			&& (strncasecmp(&(string[position]), "unlimited", 9) == 0)
		    )
		    ) {
			*integerPointer = (unsigned int) -1;
			return true;
		}

		/* If the next thing isn't a digit, fail as invalid. */
		if (string[position] < '0' || string[position] > '9')
			return false;
		while (position < length && (string[position] >= '0' && string[position] <= '9')) {
			currentNumber = (10 * currentNumber) + (unsigned int) (string[position] - '0');
			position++;
		}

		/* Skip past whitespace. */
		while (position < length && isspace(string[position]))
			position++;
		if (position >= length) {
			totalValue += currentNumber;
			currentNumber = 0;
			break;
		}

		/* Check for a suffix. */
		if ('b' == string[position] || 'B' == string[position]) {
			/* No action - bytes. */
		} else if ('k' == string[position] || 'K' == string[position]) {
			/* Kibibytes. */
			currentNumber = currentNumber * 1024;
		} else if ('m' == string[position] || 'M' == string[position]) {
			/* Mebibytes. */
			currentNumber = currentNumber * 1024 * 1024;
		} else if ('g' == string[position] || 'G' == string[position]) {
			/* Gibibytes. */
			currentNumber = currentNumber * 1024 * 1024 * 1024;
		}
		/* Any higher suffixes would be silly in this context. */

		/* Add the resultant number to the total. */
		totalValue += currentNumber;
		currentNumber = 0;

		/* Skip past any more letters. */
		while (position < length && isalpha(string[position]))
			position++;
		if (position >= length)
			break;
	}

	*integerPointer = totalValue;
	return true;
}


/*
 * Return true if the string is a valid exit status, and if so, write the
 * number of bytes to *integerPointer.
 */
static bool parseExitStatus(const char *string, size_t length, unsigned int *integerPointer)
{
	unsigned int totalValue, currentNumber;
	size_t position;

	totalValue = 0;
	currentNumber = 0;
	position = 0;

	/* This is broadly similar to parsePeriod(). */

	while (position < length) {
		/* Skip past whitespace. */
		while (position < length && isspace(string[position]))
			position++;
		if (position >= length)
			break;

		/* If the first thing isn't a digit, fail as invalid. */
		if (string[position] < '0' || string[position] > '9')
			return false;
		while (position < length && (string[position] >= '0' && string[position] <= '9')) {
			currentNumber = (10 * currentNumber) + (unsigned int) (string[position] - '0');
			position++;
		}

		/* Add the resultant number to the total. */
		totalValue += currentNumber;
		currentNumber = 0;

		/* Ignore everything else. */
		break;
	}

	if (totalValue > 255)
		return false;

	*integerPointer = totalValue;
	return true;
}


/*
 * Return true if the string is a valid boolean, and if so, write its value
 * to *booleanPointer.
 */
static bool parseBoolean(const char *string, size_t length, bool *booleanPointer)
{
	struct {
		/*@null@ */ const char *label;
		bool value;
	} booleanLabels[] = {
		{ "true", true },
		{ "yes", true },
		{ "on", true },
		{ "1", true },
		{ "false", false },
		{ "no", false },
		{ "off", false },
		{ "0", false },
		{ NULL, false }
	};
	int labelIndex;

	for (labelIndex = 0; NULL != booleanLabels[labelIndex].label; labelIndex++) {
		if (strlen(booleanLabels[labelIndex].label) != length)	/* flawfinder: ignore */
			continue;
		/*
		 * flawfinder warns about strlen() in case it's given an
		 * unterminated string, but the strings in this structure
		 * are all null-terminated, so we can ignore that.
		 */
		/*@-unrecog@ */
		/* splint doesn't recognise strncasecmp(). */
		if (0 != strncasecmp(string, booleanLabels[labelIndex].label, length))
			continue;
		/*@+unrecog@ */
		if (NULL != booleanPointer) {
			*booleanPointer = booleanLabels[labelIndex].value;
		}
		return true;
	}

	return false;
}


/*
 * Return true if the string is a valid receiver strategy, and if so, write its value
 * to *enumPointer.
 */
static bool parseReceiverStrategy(const char *string, size_t length, /*@null@ */ void *enumPointer)
{
	struct {
		/*@null@ */ const char *label;
		scwReceiverStrategy value;
	} validLabels[] = {
		{ "pipe", SCW_RECEIVER_PIPE },
		{ "socket", SCW_RECEIVER_UNIXSOCKET },
		{ "relay", SCW_RECEIVER_RELAY },
		{ "auto", SCW_RECEIVER_AUTO },
		{ NULL, SCW_RECEIVER_AUTO }
	};
	int labelIndex;

	for (labelIndex = 0; NULL != validLabels[labelIndex].label; labelIndex++) {
		if (strlen(validLabels[labelIndex].label) != length)	/* flawfinder: ignore */
			continue;
		/*
		 * flawfinder warns about strlen() in case it's given an
		 * unterminated string, but the strings in this structure
		 * are all null-terminated, so we can ignore that.
		 */
		/*@-unrecog@ */
		/* splint doesn't recognise strncasecmp(). */
		if (0 != strncasecmp(string, validLabels[labelIndex].label, length))
			continue;
		/*@+unrecog@ */
		if (NULL != enumPointer) {
			*((scwReceiverStrategy *) enumPointer) = validLabels[labelIndex].value;
		}
		return true;
	}

	return false;
}


/* Types of values that settings can take. */
typedef enum {
	SCW_VALUE_STRING,		 /* just a string to copy */
	SCW_VALUE_ITEM,			 /* the name of an item */
	SCW_VALUE_SCHEDULE,		 /* a schedule entry */
	SCW_VALUE_STATUSMODE,		 /* status mode */
	SCW_VALUE_OUTPUTMAP,		 /* output map */
	SCW_VALUE_PERIOD,		 /* time period */
	SCW_VALUE_BYTECOUNT,		 /* byte count */
	SCW_VALUE_FLAG,			 /* a boolean */
	SCW_VALUE_RECEIVERSTRATEGY,	 /* a receiver strategy */
	SCW_VALUE_EXITSTATUS,		 /* an exit status */
	SCW_VALUE_END_OF_LIST
} scwValueType;


/*
 * Parse a single setting=value string into a settings structure.  Returns a
 * non-zero exit code (one of SCW_EXIT_*) if there was a fatal error, and if
 * the source filename was provided, output an error message.
 */
int parseSetting( /*@dependent@ */ const char *string, size_t length, /*@dependent@ */ struct scwSettings *settings,
		 scwSettingSource source,
		 /*@null@ */ const char *sourceFilename, size_t sourceLine)
{
	size_t startIndex, endIndex;
	size_t nameStartIndex, nameLength;
	size_t valueStartIndex, valueLength;
	struct {
		const char *name;	 /* name of the setting */
		scwValueType valueType;	 /* type of value to expect */
		bool globalConfigOnly;	 /* global config (or command line) only */
		bool canSetInItem;	 /* can be set in item settings */
		/*@null@ */
		/*@dependent@ */
		struct scwSettingValue *targetStruct;	/* value structure to update */
		/*@null@ */
		/*@dependent@ */
		void *auxTarget;	 /* other type of value to update */
		/*@null@ */
		/*@dependent@ */
		bool *clearedTarget;	 /* the "cleared" flag for an array type */
	} settingType[] = {
		/* Global-only settings. */
		{ "UserConfigFile", SCW_VALUE_STRING, true, false, &(settings->userConfigFile), NULL, NULL },
		{ "ItemListFile", SCW_VALUE_STRING, true, false, &(settings->itemListFile), NULL, NULL },
		{ "CrontabFile", SCW_VALUE_STRING, true, false, &(settings->crontabFile), NULL, NULL },
		{ "UpdateLockFile", SCW_VALUE_STRING, true, false, &(settings->updateLockFile), NULL, NULL },
		/* Global and per-user settings. */
		{ "ItemsDir", SCW_VALUE_STRING, false, false, &(settings->itemsDir), NULL, NULL },
		{ "MetricsDir", SCW_VALUE_STRING, false, false, &(settings->metricsDir), NULL, NULL },
		{ "CheckLockFile", SCW_VALUE_STRING, false, false, &(settings->checkLockFile), NULL, NULL },
		{ "Sendmail", SCW_VALUE_STRING, false, false, &(settings->sendmail), NULL, NULL },
		{ "TransmitForm", SCW_VALUE_STRING, false, false, &(settings->transmitForm), NULL, NULL },
		{ "TransmitJSON", SCW_VALUE_STRING, false, false, &(settings->transmitJson), NULL, NULL },
		/* Item settings. */
		{ "Description", SCW_VALUE_STRING, false, true, &(settings->description), NULL, NULL },
		{ "Command", SCW_VALUE_STRING, false, true, &(settings->command), NULL, NULL },
		{ "AmbiguousExitStatus", SCW_VALUE_EXITSTATUS, false, true, &(settings->ambiguousExitStatus),
		 &(settings->numAmbiguousExitStatus),
		 NULL },
		{ "Schedule", SCW_VALUE_SCHEDULE, false, true, &(settings->schedule[0]), &(settings->countSchedules),
		 &(settings->clearedSchedules) },
		{ "RandomDelay", SCW_VALUE_PERIOD, false, true, &(settings->randomDelay), &(settings->numRandomDelay),
		 NULL },
		{ "MaxRunTime", SCW_VALUE_PERIOD, false, true, &(settings->maxRunTime), &(settings->numMaxRunTime),
		 NULL },
		{ "Prerequisite", SCW_VALUE_STRING, false, true, &(settings->prerequisite), NULL, NULL },
		{ "MinInterval", SCW_VALUE_PERIOD, false, true, &(settings->minInterval),
		 &(settings->numMinInterval), NULL },
		{ "SuccessInterval", SCW_VALUE_PERIOD, false, true, &(settings->successInterval),
		 &(settings->numSuccessInterval), NULL },
		{ "ConcurrencyWait", SCW_VALUE_PERIOD, false, true, &(settings->concurrencyWait),
		 &(settings->numConcurrencyWait), NULL },
		{ "SilentConcurrency", SCW_VALUE_FLAG, false, true, &(settings->silentConcurrency),
		 &(settings->flagSilentConcurrency), NULL },
		{ "IgnoreOverrun", SCW_VALUE_FLAG, false, true, &(settings->ignoreOverrun),
		 &(settings->flagIgnoreOverrun), NULL },
		{ "DependsOn", SCW_VALUE_ITEM, false, true, &(settings->dependsOn[0]), &(settings->countDependencies),
		 &(settings->clearedDependencies) },
		{ "DependencyWait", SCW_VALUE_PERIOD, false, true, &(settings->dependencyWait),
		 &(settings->numDependencyWait), NULL },
		{ "SilentDependency", SCW_VALUE_FLAG, false, true, &(settings->silentDependency),
		 &(settings->flagSilentDependency), NULL },
		{ "ConflictsWith", SCW_VALUE_ITEM, false, true, &(settings->conflictsWith[0]),
		 &(settings->countConflicts), &(settings->clearedConflicts) },
		{ "ConflictWait", SCW_VALUE_PERIOD, false, true, &(settings->conflictWait),
		 &(settings->numConflictWait), NULL },
		{ "SilentConflict", SCW_VALUE_FLAG, false, true, &(settings->silentConflict),
		 &(settings->flagSilentConflict), NULL },
		{ "StatusMode", SCW_VALUE_STATUSMODE, false, true, &(settings->statusMode), NULL, NULL },
		{ "StatusTag", SCW_VALUE_STRING, false, true, &(settings->statusTag), NULL, NULL },
		{ "TimestampUTC", SCW_VALUE_FLAG, false, true, &(settings->timestampUTC),
		 &(settings->flagTimestampUTC), NULL },
		{ "HTTPInterval", SCW_VALUE_PERIOD, false, true, &(settings->httpInterval),
		 &(settings->numHTTPInterval), NULL },
		{ "HTTPTimeout", SCW_VALUE_PERIOD, false, true, &(settings->httpTimeout),
		 &(settings->numHTTPTimeout), NULL },
		{ "SharedSecret", SCW_VALUE_STRING, false, true, &(settings->sharedSecret), NULL, NULL },
		{ "EmailMaxBodySize", SCW_VALUE_BYTECOUNT, false, true, &(settings->emailMaxBodySize),
		 &(settings->numEmailMaxBodySize), NULL },
		{ "EmailBodyText", SCW_VALUE_STRING, false, true, &(settings->emailBodyText),
		 &(settings->emailBodyText), NULL },
		{ "EmailAttachmentName", SCW_VALUE_STRING, false, true, &(settings->emailAttachmentName),
		 &(settings->emailAttachmentName), NULL },
		{ "EmailSender", SCW_VALUE_STRING, false, true, &(settings->emailSender),
		 &(settings->emailSender), NULL },
		{ "EmailSubject", SCW_VALUE_STRING, false, true, &(settings->emailSubject),
		 &(settings->emailSubject), NULL },
		{ "OutputMap", SCW_VALUE_OUTPUTMAP, false, true, &(settings->outputMap[0]),
		 &(settings->countOutputMaps), &(settings->clearedOutputMaps) },
		{ "ReceiverStrategy", SCW_VALUE_RECEIVERSTRATEGY, false, true, &(settings->receiverStrategy),
		 &(settings->enumReceiverStrategy), NULL },
		{ "", SCW_VALUE_END_OF_LIST, false, false, NULL, NULL, NULL },
	};
	int typeIndex;

	debug("%s: [%.*s]", "parsing requested", length, string);

	/*
	 * Find the first '#' (comment) and shorten the length so the string
	 * stops there, so we can ignore comments.
	 */
	startIndex = 0;
	while (startIndex < length && string[startIndex] != '#')
		startIndex++;
	if (startIndex < length && '#' == string[startIndex])
		length = startIndex;

	/*
	 * Shorten the length until all trailing whitespace has been
	 * excluded.
	 */
	while ((length > 0) && (isspace(string[length - 1])))
		length--;

	debug("%s: [%.*s]", "after stripping comments and trailing spaces", length, string);

	/*
	 * Look for the name of the setting to be changed.
	 */

	startIndex = 0;

	/* Skip leading whitespace. */
	while (startIndex < length && isspace(string[startIndex]))
		startIndex++;

	/* Early return if we reached the end of the string. */
	if (startIndex >= length) {
		debug("%s", "no non-whitespace found");
		/*
		 * It's only an error if it was on the command line,
		 * otherwise we just ignore it and return 0, since config
		 * files can have blank or commented-out lines.
		 */
		if (SCW_SOURCE_COMMANDLINE == source)
			return SCW_EXIT_BAD_CONFIG;
		return 0;
	}

	/* Find the end of the first word. */
	endIndex = startIndex;
	while (endIndex < length && isalpha(string[endIndex]))
		endIndex++;

	/* This gives us the name of the setting to be changed. */
	nameStartIndex = startIndex;
	nameLength = endIndex - startIndex;

	debug("%s: [%.*s]", "setting name", nameLength, string + nameStartIndex);

	/* Find the '=' between the setting and the value. */

	startIndex = endIndex;

	/* Skip trailing whitespace. */
	while (startIndex < length && isspace(string[startIndex]))
		startIndex++;

	/* If no '=' was found, return a config error. */
	if (startIndex >= length || string[startIndex] != '=') {
		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */
		if (NULL != sourceFilename)
			fprintf(stderr, "%s: %s:%lu: %s\n", PACKAGE_NAME, sourceFilename, (unsigned long) sourceLine,
				_("missing '=' after setting name"));
		/*@+mustfreefresh@ */
		return SCW_EXIT_BAD_CONFIG;
	}

	/* Move past the '='. */
	startIndex++;

	/* Skip whitespace before the value. */
	while (startIndex < length && isspace(string[startIndex]))
		startIndex++;

	valueStartIndex = startIndex;
	valueLength = length - startIndex;

	/* We now have both the setting name, and the value. */

	debug("%s: [%.*s]", "intended value", valueLength, string + valueStartIndex);

	/*
	 * Look up the setting name in our list of settings types, and
	 * perform the appropriate parsing and storage actions.
	 */
	for (typeIndex = 0; NULL != settingType[typeIndex].targetStruct; typeIndex++) {
		/*@dependent@ */
		struct scwSettingValue *valueStruct = settingType[typeIndex].targetStruct;
		bool targetIsArray = false;
		size_t settingNameLength;
		unsigned int *integerPointer;
		bool *booleanPointer;
		size_t *countPointer;
		bool *clearedPointer;

		integerPointer = settingType[typeIndex].auxTarget;
		booleanPointer = settingType[typeIndex].auxTarget;
		countPointer = settingType[typeIndex].auxTarget;
		clearedPointer = settingType[typeIndex].clearedTarget;

		settingNameLength = strlen(settingType[typeIndex].name);	/* flawfinder: ignore */
		/*
		 * flawfinder warns about strlen() in case it's given an
		 * unterminated string, but the strings in this structure
		 * are all null-terminated, so we can ignore that.
		 */

		if (settingNameLength != nameLength)
			continue;

		/*@-unrecog@ */
		/* splint doesn't recognise strncasecmp(). */
		if (0 != strncasecmp(string + nameStartIndex, settingType[typeIndex].name, nameLength))
			continue;
		/*@+unrecog@ */

		/*
		 * Check that this setting is allowed to come from this
		 * source.
		 */
		if (settingType[typeIndex].globalConfigOnly) {
			if ((source != SCW_SOURCE_GLOBAL_CONFIG) && (source != SCW_SOURCE_COMMANDLINE)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						_("this is a global setting which cannot be set here"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
		} else if (!settingType[typeIndex].canSetInItem) {
			if ((SCW_SOURCE_ITEM_SETTINGS == source) || (SCW_SOURCE_ITEM_SCRIPT == source)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						_("this setting cannot be changed within an item"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
		}

		/* Determine whether this setting takes multiple values. */
		if ((NULL != settingType[typeIndex].auxTarget) && ((SCW_VALUE_SCHEDULE ==
								    settingType[typeIndex].valueType)
								   || (SCW_VALUE_ITEM ==
								       settingType[typeIndex].valueType)
								   || (SCW_VALUE_OUTPUTMAP ==
								       settingType[typeIndex].valueType)
		    )) {
			targetIsArray = true;
		}

		/*
		 * When adding to an array of values, adjust the target
		 * pointer so it points to the <count>th array element, and
		 * make sure the array hasn't been filled up, unless the
		 * value is blank, which will clear the array.
		 */
		if (targetIsArray && valueLength > 0 && NULL != settingType[typeIndex].auxTarget) {
			if (*countPointer >= SCW_MAX_VALUES) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						_("this setting has been given too many values"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}

			/* Move along the array by <current size> elements. */
			valueStruct += *countPointer;
			/*
			 * Possible alternative method if we don't trust
			 * pointer arithmetic to move in increments of
			 * (sizeof *pointer) bytes:
			 *
			 * valueStruct = (struct scwSettingValue *) ( ((void *)valueStruct) + (*countPointer) * sizeof(struct scwSettingValue));
			 */
		}

		/*
		 * Process the value according to its type - validate its
		 * format, and write converted integer or boolean values to
		 * the auxiliary target.  We store the string value after
		 * this section.
		 */
		switch (settingType[typeIndex].valueType) {

		case SCW_VALUE_STRING:
			/* No special action for strings. */
			break;

		case SCW_VALUE_ITEM:
			/* Item names must contain no spaces or slashes. */
			if (valueLength > 0 && !validateItemName(string + valueStartIndex, valueLength)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid item name"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_SCHEDULE:
			/* Schedules must be valid crontab schedules. */
			if (valueLength > 0 && !validateSchedule(string + valueStartIndex, valueLength)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex, _("not a valid schedule"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_STATUSMODE:
			/* Status mode must be stdout, stderr, or fd. */
			if (!validateStatusMode(string + valueStartIndex, valueLength)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid status mode"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_OUTPUTMAP:
			/* Output map has a specific format. */
			if (valueLength > 0 && !validateOutputMap(string + valueStartIndex, valueLength)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid output map"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_PERIOD:
			/* Parse and store a time period. */
			if (0 == valueLength) {
				if (NULL != integerPointer)
					*integerPointer = 0;
			} else if (valueLength > 0 && NULL != integerPointer
				   && !parsePeriod(string + valueStartIndex, valueLength, integerPointer)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid time period"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_BYTECOUNT:
			/* Parse and store a byte count. */
			if (0 == valueLength) {
				if (NULL != integerPointer)
					*integerPointer = 0;
			} else if (valueLength > 0 && NULL != integerPointer
				   && !parseByteCount(string + valueStartIndex, valueLength, integerPointer)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid byte count"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_FLAG:
			/* Check the value is a boolean, and store it. */
			if (NULL != booleanPointer
			    && !parseBoolean(string + valueStartIndex, valueLength, booleanPointer)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid boolean value"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_RECEIVERSTRATEGY:
			/* Validate the value, and store it. */
			if (!parseReceiverStrategy
			    (string + valueStartIndex, valueLength, settingType[typeIndex].auxTarget)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid receiver strategy"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_EXITSTATUS:
			/* Parse and store an exit status. */
			if (0 == valueLength) {
				if (NULL != integerPointer)
					*integerPointer = 0;
			} else if (valueLength > 0 && NULL != integerPointer
				   && !parseExitStatus(string + valueStartIndex, valueLength, integerPointer)) {
				/*@-mustfreefresh@ */
				/* splint note: gettext triggers a warning we can't resolve. */
				if (NULL != sourceFilename)
					fprintf(stderr, "%s: %s:%lu: %.*s: %.*s: %s\n", PACKAGE_NAME, sourceFilename,
						(unsigned long) sourceLine, (int) nameLength, string + nameStartIndex,
						(int) valueLength, string + valueStartIndex,
						_("not a valid exit status"));
				/*@+mustfreefresh@ */
				return SCW_EXIT_BAD_CONFIG;
			}
			break;

		case SCW_VALUE_END_OF_LIST:
			/* should never be reached */
			break;
		}

		/*
		 * Store the string value.  If this is an array entry, also
		 * increment the counter for the number of entries in that
		 * array - unless the value is empty, in which case we
		 * instead set the counter to zero and set the flag to say
		 * the array was cleared.
		 */
		if (targetIsArray && 0 == valueLength && NULL != countPointer) {
			*countPointer = 0;
			if (NULL != clearedPointer)
				*clearedPointer = true;
		}
		if (valueLength > 0) {
			valueStruct->rawValue = string + valueStartIndex;
			valueStruct->rawLength = valueLength;
			debug("%s: %p = [%.*s]@%p", "storing value", valueStruct, valueStruct->rawLength,
			      valueStruct->rawValue, valueStruct->rawValue);
		} else {
			valueStruct->rawValue = NULL;
			valueStruct->rawLength = 0;
		}
		valueStruct->expandedValue = NULL;
		valueStruct->expandedLength = 0;
		if (targetIsArray && valueLength > 0 && NULL != countPointer) {
			*countPointer = 1 + *countPointer;
		}

		return 0;		    /* exit loop and return early - parsed successfully. */
	}

	/*@-mustfreefresh@ */
	/* splint note: gettext triggers a warning we can't resolve. */
	if (NULL != sourceFilename)
		fprintf(stderr, "%s: %s:%lu: %.*s: %s\n", PACKAGE_NAME, sourceFilename, (unsigned long) sourceLine,
			(int) nameLength, string + nameStartIndex, _("unknown setting"));
	/*@+mustfreefresh@ */
	return SCW_EXIT_BAD_CONFIG;
}


/*
 * Populate the expandedValue of a setting by expanding placeholders in its
 * rawValue.  The expandedValue is always null-terminated.
 */
void expandRawValue(struct scwState *state, struct scwSettingValue *value)
{
#define PLACEHOLDER(p, a, b) { p, strlen(p), a, b }	/* flawfinder: ignore */
	/*
	 * flawfinder worries about strlen() over-read on
	 * non-null-terminated strings, but we're calling it on static
	 * strings guaranteed to be null terminated, so it's OK here.
	 */
	struct {
		const char *placeholder;
		size_t placeholderLength;
		/*@null@ */
		/*@dependent@ */
		const char *replacement;
		size_t replacementLength;
	} placeholders[] = {
		/*
		 * Note that all placeholders must start with '{', so a search of
		 * this array is only required when finding that character.
		 */
		PLACEHOLDER("{USER}", state->username, state->usernameLength),
		PLACEHOLDER("{ITEM}", state->item, state->itemLength),
		PLACEHOLDER("{HOSTNAME}", state->hostname, state->hostnameLength),
		PLACEHOLDER("{DATE}", state->currentDate, state->currentDateLength),
		PLACEHOLDER("{COMMAND}", state->itemCommand, state->itemCommandLength),
		{ "", 0, NULL, 0 }
	};
	size_t readPosition, requiredLength, writePosition;
	bool anyPlaceholdersFound;

	if (NULL == value->rawValue)
		return;

	debug("%s: %p = [%.*s]@%p", "checking", value, value->rawLength, value->rawValue, value->rawValue);

	/*
	 * Determine whether placeholders are present, and how much space is
	 * needed when replacing them with their values.
	 */

	requiredLength = 0;
	readPosition = 0;
	anyPlaceholdersFound = false;
	while (readPosition < value->rawLength) {
		size_t remainingLength, placeholderIndex;
		bool foundPlaceholder;

		if ('{' != value->rawValue[readPosition]) {
			readPosition++;
			requiredLength++;
			continue;
		}

		remainingLength = value->rawLength - readPosition;
		foundPlaceholder = false;
		for (placeholderIndex = 0; placeholders[placeholderIndex].placeholderLength > 0 && !foundPlaceholder;
		     placeholderIndex++) {
			if (placeholders[placeholderIndex].placeholderLength > remainingLength)
				continue;
			if (0 !=
			    strncasecmp(placeholders[placeholderIndex].placeholder, value->rawValue + readPosition,
					placeholders[placeholderIndex].placeholderLength))
				continue;
			foundPlaceholder = true;
			break;
		}

		if (foundPlaceholder) {
			readPosition += placeholders[placeholderIndex].placeholderLength;
			requiredLength += placeholders[placeholderIndex].replacementLength;
			anyPlaceholdersFound = true;
		} else {
			readPosition++;
			requiredLength++;
		}
	}

	debug("%s: %p: [%.*s]: %s %d -> %d", "placeholders found", value, value->rawLength, value->rawValue,
	      "length", value->rawLength, requiredLength);

	/* Allocate space for the expanded value. */
	value->expandedValue = stringCopy(state, NULL, requiredLength);
	if (NULL == value->expandedValue) {
		/* Memory allocation failure - fatal error, immediate exit. */
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, strerror(errno));
		/*@-exitarg@ */
		/*
		 * splint notes that this is not a standard exit code; we
		 * ignore that, since we've documented our non-standard exit
		 * codes in the manual.
		 */
		exit(SCW_EXIT_ERROR);
		/*@+exitarg@ */
	}
	value->expandedLength = requiredLength;

	if (!anyPlaceholdersFound) {
		/*
		 * If there were no placeholders, then the expanded value is
		 * the same as the raw value.  We have allocated a copy
		 * anyway, so that the expanded value is null-terminated.
		 */
		memcpy(value->expandedValue, value->rawValue, value->rawLength);	/* flawfinder: ignore */
		/* flawfinder - the buffer is large enough; no overflow. */
		return;
	}

	/*
	 * Construct the expanded version of the string, walking through
	 * each byte in a similar way to the loop above.
	 */

	readPosition = 0;
	writePosition = 0;
	while (readPosition < value->rawLength && writePosition < value->expandedLength) {
		size_t remainingLength, placeholderIndex;
		bool foundPlaceholder;

		if ('{' != value->rawValue[readPosition]) {
			value->expandedValue[writePosition] = value->rawValue[readPosition];
			readPosition++;
			writePosition++;
			continue;
		}

		remainingLength = value->rawLength - readPosition;
		foundPlaceholder = false;
		for (placeholderIndex = 0; placeholders[placeholderIndex].placeholderLength > 0 && !foundPlaceholder;
		     placeholderIndex++) {
			if (placeholders[placeholderIndex].placeholderLength > remainingLength)
				continue;
			if (0 !=
			    strncasecmp(placeholders[placeholderIndex].placeholder, value->rawValue + readPosition,
					placeholders[placeholderIndex].placeholderLength))
				continue;
			foundPlaceholder = true;
			break;
		}

		if (foundPlaceholder) {
			size_t replacementReadPosition;
			readPosition += placeholders[placeholderIndex].placeholderLength;
			for (replacementReadPosition = 0;
			     NULL != placeholders[placeholderIndex].replacement
			     && replacementReadPosition < placeholders[placeholderIndex].replacementLength
			     && writePosition < value->expandedLength; replacementReadPosition++, writePosition++) {
				value->expandedValue[writePosition] =
				    placeholders[placeholderIndex].replacement[replacementReadPosition];
			}
		} else {
			value->expandedValue[writePosition] = value->rawValue[readPosition];
			readPosition++;
			writePosition++;
		}
	}

	debug("%s: %p: [%.*s]", "placeholders expanded", value, value->expandedLength, value->expandedValue);
}


/*
 * Copy the source value into the target value, but in the target, re-expand
 * the expandedValue using the given item instead of the current one.
 *
 * The source value may be NULL, in which case the target value is
 * re-expanded without its raw value first being copied.
 */
void expandValueForOtherItem(struct scwState *state, /*@null@ */ struct scwSettingValue *sourceValue,
			     /*@dependent@ *//*@null@ */ const char *targetItemName, size_t targetItemLength,
			     struct scwSettingValue *targetValue)
{
	const char *currentItemName;
	size_t currentItemLength;

	currentItemName = state->item;
	currentItemLength = state->itemLength;

	state->item = targetItemName;
	state->itemLength = targetItemLength;
	if (NULL != sourceValue) {
		memmove(targetValue, sourceValue, sizeof(*targetValue));
	}
	targetValue->expandedValue = NULL;
	targetValue->expandedLength = 0;

	expandRawValue(state, targetValue);

	state->item = currentItemName;
	state->itemLength = currentItemLength;
}


/*
 * Populate the expandedValue of every setting with a string value, by
 * calling expandRawValue() on each one that hasn't already been expanded.
 */
void expandAllRawValues(struct scwState *state, struct scwSettings *settings)
{
#define expandSettingValue(x) { \
	if (NULL != settings->x.rawValue && NULL == settings->x.expandedValue) \
		expandRawValue(state, &(settings->x)); \
}
#define expandArrayOfSettingValues(x, c) { \
	size_t arrayIndex; \
	for (arrayIndex = 0; arrayIndex < settings->c; arrayIndex++) { \
		if (NULL != settings->x[arrayIndex].rawValue && NULL == settings->x[arrayIndex].expandedValue) \
			expandRawValue(state, &(settings->x[arrayIndex])); \
	} \
}
	expandSettingValue(userConfigFile);
	expandSettingValue(itemListFile);
	expandSettingValue(crontabFile);
	expandSettingValue(updateLockFile);
	expandSettingValue(itemsDir);
	expandSettingValue(metricsDir);
	expandSettingValue(checkLockFile);
	expandSettingValue(sendmail);
	expandSettingValue(transmitForm);
	expandSettingValue(transmitJson);
	expandSettingValue(description);
	expandSettingValue(command);
	/* integer, in numAmbiguousExitStatus: expandSettingValue(ambiguousExitStatus); */
	expandArrayOfSettingValues(schedule, countSchedules);
	/* integer, in numRandomDelay: expandSettingValue(randomDelay) */
	/* integer, in numMaxRunTime: expandSettingValue(maxRunTime); */
	expandSettingValue(prerequisite);
	/* integer, in numMinInterval: expandSettingValue(minInterval); */
	/* integer, in numSuccessInterval: expandSettingValue(successInterval); */
	/* integer, in numConcurrencyWait: expandSettingValue(concurrencyWait); */
	/* boolean, in flagSilentConcurrency: expandSettingValue(silentConcurrency); */
	/* boolean, in flagIgnoreOverrun: expandSettingValue(ignoreOverrun); */
	expandArrayOfSettingValues(dependsOn, countDependencies);
	/* integer, in numDependencyWait: expandSettingValue(dependencyWait); */
	/* boolean, in flagSilentDependency: expandSettingValue(silentDependency); */
	expandArrayOfSettingValues(conflictsWith, countConflicts);
	/* integer, in numConflictWait: expandSettingValue(conflictWait); */
	/* boolean, in flagSilentConflict: expandSettingValue(silentConflict); */
	expandSettingValue(statusMode);
	expandSettingValue(statusTag);
	/* boolean, in flagTimestampUTC: expandSettingValue(timestampUTC); */
	/* integer, in numHTTPInterval: expandSettingValue(httpInterval); */
	/* integer, in numHTTPTimeout: expandSettingValue(httpTimeout); */
	/* no placeholders allowed: expandSettingValue(sharedSecret); */
	/* integer, in numEmailMaxBodySize: expandSettingValue(emailMaxBodySize); */
	expandSettingValue(emailBodyText);
	expandSettingValue(emailAttachmentName);
	expandSettingValue(emailSender);
	expandSettingValue(emailSubject);
	expandArrayOfSettingValues(outputMap, countOutputMaps);
}


/*
 * Load settings from a file into a settings structure.  Returns a non-zero
 * exit code (one of SCW_EXIT_*) if there was a fatal error.  It is not an
 * error if the file cannot be opened.
 */
int loadSettings(struct scwState *state, /*@null@ */ const char *filename,
		 /*@dependent@ */ struct scwSettings *settings, scwSettingSource source)
{
	char string[SCW_MAX_LINELENGTH]; /* flawfinder: ignore */
	int retcode;
	FILE *stream;
	size_t lineNumber;

	/*
	 * flawfinder rationale: zeroed before use, and only written to by
	 * fgets() which is bounded by the size passed to it.
	 */

	/* Do nothing if the filename is null. */
	if (NULL == filename)
		return 0;

	debug("%s: %s", "opening", filename);

	stream = fileOpenStreamForRead(filename);
	if (NULL == stream) {
		debug("%s: %s", filename, strerror(errno));
		return 0;
	}

#if HAVE_POSIX_FADVISE
	/* Advise the OS that we will only be reading sequentially. */
	(void) posix_fadvise(fileno(stream), 0, 0, POSIX_FADV_SEQUENTIAL);
#endif

	memset(string, 0, sizeof(string));

	retcode = 0;
	lineNumber = 0;

	while (NULL != fgets(string, (int) (sizeof(string)), stream)) {
		char *storedCopy;
		size_t length = strlen(string);	/* flawfinder: ignore */
		size_t position = 0;

		/*
		 * flawfinder - fgets() always reads one less than the
		 * buffer size and adds a terminating null byte, and we
		 * already zeroed the buffer in case it doesn't do that, so
		 * strlen() will always be passed a null-terminated string;
		 * this mean that flawfinder's warning about possible
		 * over-read if the string isn't null-terminated can be
		 * ignored here.
		 */

		lineNumber++;

		/* Skip leading whitespace. */
		while (position < length && isspace(string[position]))
			position++;

		/* If we reached the end, move to the next line. */
		if (position >= length)
			continue;

		/*
		 * If this is a script file, look for #, then any spaces,
		 * then our name, then spaces.  If we don't find a #, then
		 * stop reading the file, because we've found the first
		 * non-comment line, after which we accept no settings.
		 */
		if (SCW_SOURCE_ITEM_SCRIPT == source) {
			if (string[position] != '#')
				break;
			position++;

			/* Skip whitespace after the comment. */
			while (position < length && isspace(string[position]))
				position++;

			/* If we reached the end, move to the next line. */
			if (position >= length)
				continue;

			/*
			 * If there's not enough room left in the line for
			 * "scw <anything>", move to the next line.
			 */
			if ((length - position) < 5)
				continue;

			/*
			 * If the next characters aren't "scw" and any type
			 * of space, move to the next line.
			 */
			if (0 != strncmp(string + position, "scw", 3))
				continue;
			position += 3;
			if (!isspace(string[position]))
				continue;

			/* Skip whitespace after "scw". */
			while (position < length && isspace(string[position]))
				position++;

			/* If we reached the end, move to the next line. */
			if (position >= length)
				continue;
		}

		/* Store a copy of the string, as the buffer is transient. */
		storedCopy = stringCopy(state, string + position, length - position);
		if (NULL == storedCopy) {
			retcode = SCW_EXIT_ERROR;
			break;
		}

		/* Parse the setting. */
		retcode = parseSetting(storedCopy, length - position, settings, source, filename, lineNumber);
		if (0 != retcode)
			break;
	}

	(void) fclose(stream);

	return retcode;
}


/*
 * Load settings for the currently selected item into the item settings
 * structure.  If there is a script for this item in the items directory,
 * read settings from that as well as from the item settings file (if it
 * exists).  Shell and Perl scripts are tried in that order.  If a script is
 * found and the item's Command is not set, the Command is set to the
 * script's path.
 *
 * It is not an error if there is no settings file or script for the item. 
 * The value of state->itemHasSettingsFile will be set to false if there was
 * no file, or true if a settings file or script file was found.
 *
 * Returns a non-zero exit code (one of SCW_EXIT_*) if there was
 * a fatal error.
 */
int loadCurrentItemSettings(struct scwState *state)
{
	struct scwSettingValue *itemsDir;
	size_t pathLength, suffixOffset;
	char *itemConfigPath;
	int retcode;

	state->itemHasSettingsFile = false;

	if (NULL == state->item)
		return 0;

	/* Use the user settings for itemsDir, fall back to global. */
	itemsDir = &(state->userSettings.itemsDir);
	if (NULL == itemsDir->rawValue)
		itemsDir = &(state->globalSettings.itemsDir);
	if (NULL == itemsDir->rawValue)
		return 0;

	/*
	 * Expand the raw value.
	 *
	 * We expand it every time, even though it wastes memory with
	 * needless string copies, otherwise in all-users mode we look in
	 * the wrong ItemsDir when changing users because {USER} was already
	 * expanded to the previous username.
	 */
	expandRawValue(state, itemsDir);

	/* Fail if the expansion failed. */
	if (NULL == itemsDir->expandedValue)
		return SCW_EXIT_ERROR;

	/*
	 * The length of the full path of the item or its scripts, which is
	 * ItemsDir/Item.{cf,sh,pl} - so the length of the directory path,
	 * the length of the item, 1 for the /, and 3 for any of ".cf" /
	 * ".sh" / ".pl".
	 */
	pathLength = itemsDir->expandedLength + 1 + state->itemLength + 3;
	suffixOffset = itemsDir->expandedLength + 1 + state->itemLength;

	itemConfigPath = stringCopy(state, NULL, pathLength);
	if (NULL == itemConfigPath)
		return SCW_EXIT_ERROR;

	/* Construct the path. */
	memcpy(itemConfigPath, itemsDir->expandedValue, itemsDir->expandedLength);	/* flawfinder: ignore */
	itemConfigPath[itemsDir->expandedLength] = '/';
	memcpy(itemConfigPath + itemsDir->expandedLength + 1, state->item, state->itemLength);	/* flawfinder: ignore */

	/* Load the ".cf" file, if it exists. */
	memcpy(itemConfigPath + suffixOffset, ".cf", 3);	/* flawfinder: ignore */
	if (0 == access(itemConfigPath, F_OK)) {	/* flawfinder: ignore */
		retcode = loadSettings(state, itemConfigPath, &(state->itemSettings), SCW_SOURCE_ITEM_SETTINGS);
		if (0 != retcode)
			return retcode;
		state->itemHasSettingsFile = true;
	}
	debug("%.*s / %.*s: %s: %s=%s", state->usernameLength, state->username, state->itemLength, state->item,
	      itemConfigPath, "itemHasSettingsFile", state->itemHasSettingsFile ? "true" : "false");

	/*
	 * flawfinder warns about time-of-check/time-of-use race conditions
	 * with access(), but such a condition isn't exploitable here since
	 * further checks are performed when actually trying to load the
	 * files.  This applies to the two access() calls below as well.
	 *
	 * flawfinder also warns about memcpy(), as it does not check that
	 * the destination buffer is large enough.  In the memcpy() calls in
	 * this function, we are copying into a buffer that has been
	 * explicitly allocated to be the right size.
	 */

	/* Load the ".sh" file, if it exists. */
	memcpy(itemConfigPath + suffixOffset, ".sh", 3);	/* flawfinder: ignore */
	if (0 == access(itemConfigPath, F_OK)) {	/* flawfinder: ignore */
		retcode = loadSettings(state, itemConfigPath, &(state->itemSettings), SCW_SOURCE_ITEM_SCRIPT);
		if (0 != retcode)
			return retcode;
		if (NULL == state->itemSettings.command.rawValue) {
			state->itemSettings.command.rawValue = itemConfigPath;
			state->itemSettings.command.rawLength = pathLength;
		}
		state->itemHasSettingsFile = true;
		debug("%.*s / %.*s: %s: %s=%s", state->usernameLength, state->username, state->itemLength, state->item,
		      itemConfigPath, "itemHasSettingsFile", state->itemHasSettingsFile ? "true" : "false");
	} else {
		/* Load the ".pl" file, if it exists. */
		memcpy(itemConfigPath + suffixOffset, ".pl", 3);	/* flawfinder: ignore */
		if (0 == access(itemConfigPath, F_OK)) {	/* flawfinder: ignore */
			retcode = loadSettings(state, itemConfigPath, &(state->itemSettings), SCW_SOURCE_ITEM_SCRIPT);
			if (0 != retcode)
				return retcode;
			if (NULL == state->itemSettings.command.rawValue) {
				state->itemSettings.command.rawValue = itemConfigPath;
				state->itemSettings.command.rawLength = pathLength;
			}
			state->itemHasSettingsFile = true;
		}
	}

	return 0;
}


/*
 * Clear the settings in the given structure and then apply the state's
 * global config, user config, the item, and command line settings to it, in
 * that order.  Array items will be appended to or reset as per each
 * settings group's "cleared" flags, so for instance if the item's settings
 * have explicitly cleared the output map before adding new ones, that will
 * be honoured, rather than just always appending item output maps to user
 * and global config output maps.
 *
 * Returns a non-zero exit code (one of SCW_EXIT_*) if there was a fatal
 * error.
 */
int combineSettings(struct scwState *state, struct scwSettings *settings)
{
	/* Start by cloning the global settings. */
	memcpy(settings, &(state->globalSettings), sizeof(*settings));	/* flawfinder: ignore */

	/*
	 * flawfinder warns about memcpy() not checking the boundaries of
	 * the destination, but here we're copying an object into another
	 * object of the same size, so it is OK.
	 */

	/* Use the preprocessor to avoid repetition. */

	/* Copy a simple value with no associated integer or boolean. */
#define copySimpleValueIfSet(source, setting) { \
	if (NULL != state->source.setting.rawValue) { \
		settings->setting.rawValue = state->source.setting.rawValue; \
		settings->setting.rawLength = state->source.setting.rawLength; \
		settings->setting.expandedValue = state->source.setting.expandedValue; \
		settings->setting.expandedLength = state->source.setting.expandedLength; \
	} \
}

	/* Copy a value with an associated integer, boolean, or enumerated type. */
#define copyParsedValueIfSet(source, setting, parsedValue) { \
	if (NULL != state->source.setting.rawValue) { \
		settings->parsedValue = state->source.parsedValue; \
		settings->setting.rawValue = state->source.setting.rawValue; \
		settings->setting.rawLength = state->source.setting.rawLength; \
		settings->setting.expandedValue = state->source.setting.expandedValue; \
		settings->setting.expandedLength = state->source.setting.expandedLength; \
	} \
}

	/* Append array values, updating the count, clearing the array first if applicable. */
#define addArrayValuesIfSet(source, array, counter, clearedFlag) { \
	if (state->source.counter > 0 || state->source.clearedFlag) { \
		size_t sourcePosition, targetIndex; \
\
		if (state->source.clearedFlag) { \
			debug("%s: %s: %s", #source, #array, "clearing"); \
			settings->counter = 0; \
		} \
\
		targetIndex = settings->counter; \
		for (sourcePosition = 0; sourcePosition < state->source.counter; sourcePosition++, targetIndex++) { \
			if (targetIndex >= SCW_MAX_VALUES) { \
				/*@-mustfreefresh@ */ \
				/* splint note: gettext triggers a warning we can't resolve. */ \
				fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, #array,  \
						_("this setting has been given too many values")); \
				/*@+mustfreefresh@ */ \
				return SCW_EXIT_BAD_CONFIG; \
			} \
			debug("%s: %s: %s: %d", #source, #array, "adding", targetIndex); \
			memcpy(&(settings->array[targetIndex]), &(state->source.array[sourcePosition]), sizeof(struct scwSettingValue));	/* flawfinder: ignore */ \
		} \
		settings->counter = targetIndex; \
	} \
}

	/*
	 * flawfinder warns about memcpy() not bounds checking; here we have
	 * checked that the size of data being copied matches the source and
	 * destination buffer sizes.
	 */

	/* Apply the user configuration settings. */
	copySimpleValueIfSet(userSettings, itemsDir);
	copySimpleValueIfSet(userSettings, metricsDir);
	copySimpleValueIfSet(userSettings, checkLockFile);
	copySimpleValueIfSet(userSettings, sendmail);
	copySimpleValueIfSet(userSettings, transmitForm);
	copySimpleValueIfSet(userSettings, transmitJson);
	copySimpleValueIfSet(userSettings, description);
	copySimpleValueIfSet(userSettings, command);
	copyParsedValueIfSet(userSettings, ambiguousExitStatus, numAmbiguousExitStatus);
	addArrayValuesIfSet(userSettings, schedule, countSchedules, clearedSchedules);
	copyParsedValueIfSet(userSettings, randomDelay, numRandomDelay);
	copyParsedValueIfSet(userSettings, maxRunTime, numMaxRunTime);
	copySimpleValueIfSet(userSettings, prerequisite);
	copyParsedValueIfSet(userSettings, minInterval, numMinInterval);
	copyParsedValueIfSet(userSettings, successInterval, numSuccessInterval);
	copyParsedValueIfSet(userSettings, concurrencyWait, numConcurrencyWait);
	copyParsedValueIfSet(userSettings, silentConcurrency, flagSilentConcurrency);
	copyParsedValueIfSet(userSettings, ignoreOverrun, flagIgnoreOverrun);
	addArrayValuesIfSet(userSettings, dependsOn, countDependencies, clearedDependencies);
	copyParsedValueIfSet(userSettings, dependencyWait, numDependencyWait);
	copyParsedValueIfSet(userSettings, silentDependency, flagSilentDependency);
	addArrayValuesIfSet(userSettings, conflictsWith, countConflicts, clearedConflicts);
	copyParsedValueIfSet(userSettings, conflictWait, numConflictWait);
	copyParsedValueIfSet(userSettings, silentConflict, flagSilentConflict);
	copySimpleValueIfSet(userSettings, statusMode);
	copySimpleValueIfSet(userSettings, statusTag);
	copyParsedValueIfSet(userSettings, timestampUTC, flagTimestampUTC);
	copyParsedValueIfSet(userSettings, httpInterval, numHTTPInterval);
	copyParsedValueIfSet(userSettings, httpTimeout, numHTTPTimeout);
	copySimpleValueIfSet(userSettings, sharedSecret);
	copyParsedValueIfSet(userSettings, emailMaxBodySize, numEmailMaxBodySize);
	copySimpleValueIfSet(userSettings, emailBodyText);
	copySimpleValueIfSet(userSettings, emailAttachmentName);
	copySimpleValueIfSet(userSettings, emailSender);
	copySimpleValueIfSet(userSettings, emailSubject);
	addArrayValuesIfSet(userSettings, outputMap, countOutputMaps, clearedOutputMaps);
	copyParsedValueIfSet(userSettings, receiverStrategy, enumReceiverStrategy);

	/* Apply the item settings. */
	/* Skip the config-only settings as they aren't allowed for items. */
	copySimpleValueIfSet(itemSettings, description);
	copySimpleValueIfSet(itemSettings, command);
	copyParsedValueIfSet(itemSettings, ambiguousExitStatus, numAmbiguousExitStatus);
	addArrayValuesIfSet(itemSettings, schedule, countSchedules, clearedSchedules);
	copyParsedValueIfSet(itemSettings, randomDelay, numRandomDelay);
	copyParsedValueIfSet(itemSettings, maxRunTime, numMaxRunTime);
	copySimpleValueIfSet(itemSettings, prerequisite);
	copyParsedValueIfSet(itemSettings, minInterval, numMinInterval);
	copyParsedValueIfSet(itemSettings, successInterval, numSuccessInterval);
	copyParsedValueIfSet(itemSettings, concurrencyWait, numConcurrencyWait);
	copyParsedValueIfSet(itemSettings, silentConcurrency, flagSilentConcurrency);
	copyParsedValueIfSet(itemSettings, ignoreOverrun, flagIgnoreOverrun);
	addArrayValuesIfSet(itemSettings, dependsOn, countDependencies, clearedDependencies);
	copyParsedValueIfSet(itemSettings, dependencyWait, numDependencyWait);
	copyParsedValueIfSet(itemSettings, silentDependency, flagSilentDependency);
	addArrayValuesIfSet(itemSettings, conflictsWith, countConflicts, clearedConflicts);
	copyParsedValueIfSet(itemSettings, conflictWait, numConflictWait);
	copyParsedValueIfSet(itemSettings, silentConflict, flagSilentConflict);
	copySimpleValueIfSet(itemSettings, statusMode);
	copySimpleValueIfSet(itemSettings, statusTag);
	copyParsedValueIfSet(itemSettings, timestampUTC, flagTimestampUTC);
	copyParsedValueIfSet(itemSettings, httpInterval, numHTTPInterval);
	copyParsedValueIfSet(itemSettings, httpTimeout, numHTTPTimeout);
	copySimpleValueIfSet(itemSettings, sharedSecret);
	copyParsedValueIfSet(itemSettings, emailMaxBodySize, numEmailMaxBodySize);
	copySimpleValueIfSet(itemSettings, emailBodyText);
	copySimpleValueIfSet(itemSettings, emailAttachmentName);
	copySimpleValueIfSet(itemSettings, emailSender);
	copySimpleValueIfSet(itemSettings, emailSubject);
	addArrayValuesIfSet(itemSettings, outputMap, countOutputMaps, clearedOutputMaps);
	copyParsedValueIfSet(itemSettings, receiverStrategy, enumReceiverStrategy);

	/* Apply the command-line settings. */
	copySimpleValueIfSet(commandLineSettings, userConfigFile);
	copySimpleValueIfSet(commandLineSettings, itemListFile);
	copySimpleValueIfSet(commandLineSettings, crontabFile);
	copySimpleValueIfSet(commandLineSettings, updateLockFile);
	copySimpleValueIfSet(commandLineSettings, itemsDir);
	copySimpleValueIfSet(commandLineSettings, metricsDir);
	copySimpleValueIfSet(commandLineSettings, checkLockFile);
	copySimpleValueIfSet(commandLineSettings, sendmail);
	copySimpleValueIfSet(commandLineSettings, transmitForm);
	copySimpleValueIfSet(commandLineSettings, transmitJson);
	copySimpleValueIfSet(commandLineSettings, description);
	copySimpleValueIfSet(commandLineSettings, command);
	copyParsedValueIfSet(commandLineSettings, ambiguousExitStatus, numAmbiguousExitStatus);
	addArrayValuesIfSet(commandLineSettings, schedule, countSchedules, clearedSchedules);
	copyParsedValueIfSet(commandLineSettings, randomDelay, numRandomDelay);
	copyParsedValueIfSet(commandLineSettings, maxRunTime, numMaxRunTime);
	copySimpleValueIfSet(commandLineSettings, prerequisite);
	copyParsedValueIfSet(commandLineSettings, minInterval, numMinInterval);
	copyParsedValueIfSet(commandLineSettings, successInterval, numSuccessInterval);
	copyParsedValueIfSet(commandLineSettings, concurrencyWait, numConcurrencyWait);
	copyParsedValueIfSet(commandLineSettings, silentConcurrency, flagSilentConcurrency);
	copyParsedValueIfSet(commandLineSettings, ignoreOverrun, flagIgnoreOverrun);
	addArrayValuesIfSet(commandLineSettings, dependsOn, countDependencies, clearedDependencies);
	copyParsedValueIfSet(commandLineSettings, dependencyWait, numDependencyWait);
	copyParsedValueIfSet(commandLineSettings, silentDependency, flagSilentDependency);
	addArrayValuesIfSet(commandLineSettings, conflictsWith, countConflicts, clearedConflicts);
	copyParsedValueIfSet(commandLineSettings, conflictWait, numConflictWait);
	copyParsedValueIfSet(commandLineSettings, silentConflict, flagSilentConflict);
	copySimpleValueIfSet(commandLineSettings, statusMode);
	copySimpleValueIfSet(commandLineSettings, statusTag);
	copyParsedValueIfSet(commandLineSettings, timestampUTC, flagTimestampUTC);
	copyParsedValueIfSet(commandLineSettings, httpInterval, numHTTPInterval);
	copyParsedValueIfSet(commandLineSettings, httpTimeout, numHTTPTimeout);
	copySimpleValueIfSet(commandLineSettings, sharedSecret);
	copyParsedValueIfSet(commandLineSettings, emailMaxBodySize, numEmailMaxBodySize);
	copySimpleValueIfSet(commandLineSettings, emailBodyText);
	copySimpleValueIfSet(commandLineSettings, emailAttachmentName);
	copySimpleValueIfSet(commandLineSettings, emailSender);
	copySimpleValueIfSet(commandLineSettings, emailSubject);
	addArrayValuesIfSet(commandLineSettings, outputMap, countOutputMaps, clearedOutputMaps);
	copyParsedValueIfSet(commandLineSettings, receiverStrategy, enumReceiverStrategy);

	/* Set the state itemCommand from the combined settings. */
	state->itemCommand = settings->command.rawValue;
	state->itemCommandLength = settings->command.rawLength;

	return 0;
}
