/*
 * Functions for the "enable" and "disable" actions.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <string.h>


/* Set the selected item to be either enabled or disabled. */
int setItemEnabledState(struct scwState *state, bool enabled)
{
	struct scwSettings combinedSettings;
	const char *markerName = "disabled";
	size_t markerLength;
	int retcode;

	markerLength = strlen(markerName);  /* flawfinder: ignore */
	/*
	 * flawfinder warns about strlen() on unterminated strings, but this
	 * is a static string that is definitely null-terminated, so is OK.
	 */

	memset(&combinedSettings, 0, sizeof(combinedSettings));
	retcode = combineSettings(state, &combinedSettings);
	if (retcode != 0)
		return retcode;

#ifdef ENABLE_DEBUGGING
	debugOutputAllSettings("combinedSettings", &combinedSettings);
#endif

	/* Expand the MetricsDir value. */
	if (NULL == combinedSettings.metricsDir.expandedValue)
		expandRawValue(state, &(combinedSettings.metricsDir));
	if (NULL == combinedSettings.metricsDir.expandedValue)
		return SCW_EXIT_ERROR;

	/* Ensure that there is a metrics directory. */
	retcode =
	    directoryParentCreate(combinedSettings.metricsDir.expandedValue,
				  combinedSettings.metricsDir.expandedLength);
	if (retcode != 0)
		return retcode;
	retcode =
	    directoryCreate(combinedSettings.metricsDir.expandedValue, combinedSettings.metricsDir.expandedLength);
	if (retcode != 0)
		return retcode;

	/* Create or delete the "disabled" marker. */
	if (enabled) {
		retcode =
		    fileDeleteAt(combinedSettings.metricsDir.expandedValue, combinedSettings.metricsDir.expandedLength,
				 markerName, markerLength);
	} else {
		retcode =
		    fileCreateAt(combinedSettings.metricsDir.expandedValue, combinedSettings.metricsDir.expandedLength,
				 markerName, markerLength, false);
	}

	return retcode;
}
