/*
 * Functions for the "run" action.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/file.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/utsname.h>


/*
 * Return true if the item with the given MetricsDir value is currently
 * running.
 *
 * Note that the CheckLockFile must be locked before this function is
 * called.
 */
bool itemIsRunning(struct scwSettingValue *itemMetricsDir)
{
	int itemLockDescriptor;
	bool isRunning;

	if (NULL == itemMetricsDir->expandedValue)
		return false;

	isRunning = false;

	itemLockDescriptor =
	    fileOpenForAppendAt(itemMetricsDir->expandedValue, itemMetricsDir->expandedLength, STATIC_STRING(".lock"));
	if (itemLockDescriptor >= 0) {
		isRunning = true;
		if (0 == flock(itemLockDescriptor, LOCK_EX | LOCK_NB)) {
			isRunning = false;
			(void) flock(itemLockDescriptor, LOCK_UN);
		}
		(void) close(itemLockDescriptor);
	}

	return isRunning;
}


/*
 * Run the Prerequisite command, if one is defined.  If it exits zero, or no
 * prerequisite command was defined, returns true and updates the
 * "prerequisites-met" metrics file; otherwise, returns false.
 *
 * Note that only the expandedValue is checked, and it is assumed to be
 * null-terminated.
 *
 * In the child process, the lock descriptors are closed if they are >= 0.
 */
static bool prerequisiteCheck(struct scwSettings *settings, int lockDescriptor1, int lockDescriptor2)
{
	if (NULL != settings->prerequisite.expandedValue && settings->prerequisite.expandedLength > 0) {
		pid_t pid;

		/*
		 * Fork, run the Prerequisite command in the child process
		 * with its standard streams connected to /dev/null and
		 * other file descriptors closed, and in the parent, wait
		 * for exit, returning early with false if the exit status
		 * was non-zero.
		 */

		pid = (pid_t) fork();
		if (pid < 0) {

			/* Fork failure. */
			perror(PACKAGE_NAME);
			return false;

		} else if (pid == 0) {

			/* Child process. */
			int nullDescriptor;

			/* Close the lock descriptors. */
			if (lockDescriptor1 >= 0)
				(void) close(lockDescriptor1);
			if (lockDescriptor2 >= 0)
				(void) close(lockDescriptor2);

			/* Replace stdin/out/err with /dev/null. */
			nullDescriptor = open("/dev/null", O_RDWR	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
					      | O_NOFOLLOW
#endif
					      , 0600);
			/*
			 * flawfinder warns of symlinks, redirections, etc
			 * when using open(); this is a fixed path, without
			 * O_CREAT or O_TRUNC, and we use O_NOFOLLOW where
			 * available, so the risk is negligible.
			 */
			if (nullDescriptor < 0) {
				perror(PACKAGE_NAME);
			} else if ((dup2(nullDescriptor, STDIN_FILENO) < 0) || (dup2(nullDescriptor, STDOUT_FILENO) < 0)
				   || (dup2(nullDescriptor, STDERR_FILENO) < 0)) {
				perror(PACKAGE_NAME);
			} else {
				(void) close(nullDescriptor);
				debugWriteOutput("", "", -1, "");	/* close debug stream. */
				(void) dup2(2, 3);
				(void) execl("/bin/sh", "sh", "-c", settings->prerequisite.expandedValue, NULL);	/* flawfinder: ignore */
				/*
				 * flawfinder warns that execl() is
				 * difficult to use safely.  There's no way
				 * for us to make it safer to run a
				 * user-supplied command here.
				 */
			}
			exit(EXIT_FAILURE);

		} else {

			/* Parent process. */
			while (true) {
				pid_t waited;
				int status;

				status = 0;
				/*@-type@ */
				/* splint disagreement about __pid_t vs pid_t. */
				waited = waitpid(pid, &status, 0);
				/*@+type@ */
				if (waited < 0 && errno == EINTR) {
					perror(PACKAGE_NAME);
					continue;
				} else if (waited < 0 && errno != EINTR) {
					perror(PACKAGE_NAME);
					return false;
				}

				debug("%s=%d, %s=%d", "waited", (int) pid, "status", status);

				/*@-predboolint@ */
				/* splint sees WIF...() as non-boolean. */
				if (WIFEXITED(status)) {
					int exitStatus = WEXITSTATUS(status);
					debug("%s: %d", "prerequisite exit status", exitStatus);
					if (0 != exitStatus)
						return false;
					break;
				} else if (WIFSIGNALED(status)) {
					debug("%s", "prerequisite terminated by signal");
					return false;
				}
				/*@+predboolint@ */

				debug("%s: %d", "unexpected parent state, returning false - status value", status);
				return false;
			}
		}
		debug("%s", "prerequisite command succeeded");
	} else {
		debug("%s", "no prerequisite command");
	}

	/* No command, or successful exit. */

	/* Update the "prerequisites-met" metrics file's timestamp. */
	if (NULL != settings->metricsDir.expandedValue) {
		(void) fileCreateAt(settings->metricsDir.expandedValue, settings->metricsDir.expandedLength,
				    STATIC_STRING("prerequisites-met"), true);
	}

	return true;
}


/*
 * Check the current item isn't already running and that its DependsOn and
 * ConflictsWith requirements are met, waiting up to the appropriate
 * timeouts.  Populates the lock file descriptors (which will be -1 if not
 * open), and sets markAsFailed and markAsOverran, and retcode as
 * appropriate.
 *
 * Returns true if the conditions were met, false if not.  This should
 * be called repeatedly until either it returns true, or *retcode is
 * non-zero.  It is up to the caller to unlock and close anything left
 * open, and to update the relevant metrics files.
 *
 * If a lock file could not be opened but continueWithoutCheckLock /
 * continueWithoutItemLock (as appropriate) are true, then a warning is
 * emitted but true is returned.  In this case the lock file descriptors
 * will both be -1.
 *
 * This implements section (B) described in the runItem() function.
 */
static bool checkItemCoordination(struct scwState *state, struct scwSettings *settings, time_t checksStarted,
				  int *checkLockDescriptor, int *itemLockDescriptor, bool *markAsFailed,
				  bool *markAsOverran, int *retcode)
{
	static struct scwSettingValue dependencyMetricsDir[SCW_MAX_VALUES];
	static struct scwSettingValue conflictMetricsDir[SCW_MAX_VALUES];
	static const char *prevItemPtr = NULL;
	const char *metricsDir;
	size_t metricsDirLength;
	struct stat statBuf;
	time_t thisItemLastEnded;
	double elapsed;
	unsigned int timeSpentWaiting;
	size_t checkIndex;

	/*
	 * Maintain static arrays for the metricsDir of each DependsOn and
	 * ConflictsWith item, which we redo whenever we're called with a
	 * different state->item from before.  Otherwise we would be
	 * allocating new strings every time this function is called, which
	 * would waste a lot of memory if delays are long.
	 *
	 * We just compare item name pointers rather than the real strings
	 * as really we just want to know if this is our first call or not,
	 * since normally there would only be one "run" action for one item
	 * in a single program invocation.
	 */
	if (state->item != prevItemPtr) {
		prevItemPtr = state->item;
		for (checkIndex = 0; checkIndex < settings->countDependencies; checkIndex++) {
			expandValueForOtherItem(state, &(settings->metricsDir),
						settings->dependsOn[checkIndex].expandedValue,
						settings->dependsOn[checkIndex].expandedLength,
						&(dependencyMetricsDir[checkIndex]));
		}
		for (checkIndex = 0; checkIndex < settings->countConflicts; checkIndex++) {
			expandValueForOtherItem(state, &(settings->metricsDir),
						settings->conflictsWith[checkIndex].expandedValue,
						settings->conflictsWith[checkIndex].expandedLength,
						&(conflictMetricsDir[checkIndex]));
		}
	}

	/* Check that key settings are not NULL. */
	if (NULL == settings->metricsDir.expandedValue) {
		/* No message to stderr - this will already have been noted. */
		*retcode = SCW_EXIT_ERROR;
		return false;
	}
	if (NULL == settings->checkLockFile.expandedValue) {
		/* No message to stderr - this will already have been noted. */
		*retcode = SCW_EXIT_ERROR;
		return false;
	}

	metricsDir = settings->metricsDir.expandedValue;
	metricsDirLength = settings->metricsDir.expandedLength;

	/*
	 * Check when this item last ended, so we have a point of comparison
	 * for DependsOn.  An dependency named in DependsOn is met if that
	 * item has succeeded since this one's previous run.
	 */

	thisItemLastEnded = 0;
	memset(&statBuf, 0, sizeof(statBuf));
	if (0 == fileStatAt(metricsDir, metricsDirLength, STATIC_STRING("ended"), &statBuf))
		thisItemLastEnded = (time_t) (statBuf.st_mtime);

	/* How long we've spent waiting for conditions to be met. */
	elapsed = difftime(time(NULL), checksStarted);
	if (elapsed > 0.0) {
		timeSpentWaiting = (unsigned int) elapsed;
	} else {
		timeSpentWaiting = 0;
	}

	/* Set initial values for the lock descriptors. */
	*checkLockDescriptor = -1;
	*itemLockDescriptor = -1;

	/* (B1) Acquire lock on CheckLockFile. */
	if (*checkLockDescriptor < 0) {
		*checkLockDescriptor = fileOpenForAppend(settings->checkLockFile.expandedValue);
		if (*checkLockDescriptor < 0) {
			/* fileOpenForAppend() does report the error. */
			/* We also report a more specific error here. */
			/*@-mustfreefresh@ */
			/* gettext triggers splint warnings we can't resolve. */
			if (!state->continueWithoutCheckLock) {
				fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "CheckLockFile",
					_("file inaccessible"), _("item cannot run"));
				*retcode = SCW_EXIT_ERROR;
				return false;
			}
			fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "CheckLockFile",
				_("file inaccessible"), _("proceeding without any checks"));
			return true;
			/*@+mustfreefresh@ */
		}
	}
	if (0 != flock(*checkLockDescriptor, LOCK_EX)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, settings->checkLockFile.expandedValue, strerror(errno));
		*retcode = SCW_EXIT_ERROR;
		return false;
	}

	/* (B2) Perform concurrency check, keep item lock open. */
	if (*itemLockDescriptor < 0) {
		*itemLockDescriptor = fileOpenForAppendAt(metricsDir, metricsDirLength, STATIC_STRING(".lock"));
		if (*itemLockDescriptor < 0) {
			/* fileOpenForAppendAt() doesn't report the error. */
			fprintf(stderr, "%s: %s/%s: %s\n", PACKAGE_NAME, metricsDir, ".lock", strerror(errno));
			/* We also report a more specific error here. */
			/*@-mustfreefresh@ */
			/* gettext triggers splint warnings we can't resolve. */
			if (!state->continueWithoutItemLock) {
				fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "MetricsDir",
					_("cannot open item lock file"), _("item cannot run"));
				*retcode = SCW_EXIT_ERROR;
				return false;
			}
			fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "MetricsDir",
				_("cannot open item lock file"), _("proceeding without any checks"));
			return true;
			/*@+mustfreefresh@ */
		}
	}
	if (0 != flock(*itemLockDescriptor, LOCK_EX | LOCK_NB)) {
		/* We couldn't lock this item, so it's already running. */
		debug("%s", "concurrency lock failed - item is already running");
		if ((settings->numConcurrencyWait < 1) || (timeSpentWaiting >= settings->numConcurrencyWait)) {
			/* No more time to wait - give up. */
			if (settings->flagSilentConcurrency) {
				if (settings->flagIgnoreOverrun) {
					debug("%s",
					      "concurrency timeout, but IgnoreOverrun is set, so not marking as overran");
				} else {
					debug("%s", "concurrency timeout, marking as overran");
					*markAsOverran = true;
				}
			} else {
				debug("%s", "concurrency timeout, marking as failed");
				*markAsFailed = true;
			}
			*retcode = SCW_EXIT_RUN_CONCURRENCY;
			return false;
		}
		/* Conditions not met, not timed out yet; return false. */
		return false;
	}

	/*
	 * At this point we've got both CheckLockFile and the item
	 * concurrency lock file locked.
	 */

	/* (B3) Perform the DependsOn checks. */
	for (checkIndex = 0; checkIndex < settings->countDependencies; checkIndex++) {
		time_t dependencyLastSucceeded = 0;

		if (NULL == dependencyMetricsDir[checkIndex].expandedValue)
			continue;

		if (0 ==
		    fileStatAt(dependencyMetricsDir[checkIndex].expandedValue,
			       dependencyMetricsDir[checkIndex].expandedLength, STATIC_STRING("succeeded"), &statBuf))
			dependencyLastSucceeded = (time_t) statBuf.st_mtime;

		if (dependencyLastSucceeded > thisItemLastEnded)
			continue;

		/* Dependency has not run - test failed. */
		debug("%.*s: %lu <= %lu - %s", settings->dependsOn[checkIndex].expandedLength,
		      settings->dependsOn[checkIndex].expandedValue, dependencyLastSucceeded, thisItemLastEnded,
		      "dependency has not run");

		if ((settings->numDependencyWait < 1) || (timeSpentWaiting >= settings->numDependencyWait)) {
			/* No more time to wait - give up. */
			if (settings->flagSilentDependency) {
				debug("%s", "dependency timeout, staying silent");
			} else {
				debug("%s", "dependency timeout, marking as failed");
				*markAsFailed = true;
			}
			*retcode = SCW_EXIT_RUN_DEPENDENCY;
			return false;
		}

		/* Conditions not met, not timed out yet; return false. */
		return false;
	}

	/* (B4) Perform the ConflictsWith checks. */
	for (checkIndex = 0; checkIndex < settings->countConflicts; checkIndex++) {
		if (!itemIsRunning(&(conflictMetricsDir[checkIndex])))
			continue;

		/* A conflicting item is currently running. */
		debug("%.*s: %s", settings->conflictsWith[checkIndex].expandedLength,
		      settings->conflictsWith[checkIndex].expandedValue, "conflicting item is running");

		if ((settings->numConflictWait < 1) || (timeSpentWaiting >= settings->numConflictWait)) {
			/* No more time to wait - give up. */
			if (settings->flagSilentConflict) {
				debug("%s", "conflict timeout, staying silent");
			} else {
				debug("%s", "conflict timeout, marking as failed");
				*markAsFailed = true;
			}
			*retcode = SCW_EXIT_RUN_CONFLICT;
			return false;
		}

		/* Conditions not met, not timed out yet; return false. */
		return false;
	}

	/* No remaining conditions to test: conditions met. */
	return true;
}


/*
 * Record the output of the running command.  If the MaxRunTime is reached,
 * we terminate the process.
 *
 * Returns zero on success, or an SCW_EXIT_* value.  The file descriptors
 * passed in are closed before returning.
 */
static int recordCommandOutput(struct scwState *state, struct scwCommandProcess *process)
{
	int retcode = 0;

	switch (process->strategy) {
	case SCW_RECEIVER_PIPE:
		retcode = captureCommandOutputViaPipe(state, process);
		break;
	case SCW_RECEIVER_UNIXSOCKET:
	case SCW_RECEIVER_RELAY:
		retcode = captureCommandOutputViaUnixSocket(state, process);
		break;
	case SCW_RECEIVER_AUTO:
		/* should not be reachable. */
		debug("%s", "strategy not auto-selected");
		retcode = captureCommandOutputViaPipe(state, process);
		break;
	}

	return retcode;
}


/*
 * Create a randomly named temporary directory in ${TMPDIR:-${TMP:-/tmp}}
 * and return a pointer to its name, or NULL on error.
 *
 * The returned pointer should not be passed to free() as it will have been
 * allocated with stringCopy(), and so will be freed by clearState().
 */
/*@null@ */
/*@dependent@ */
static char *createTemporaryDirectory(struct scwState *state)
{
	char newDirectory[1024];	 /* flawfinder: ignore */
	char *tmpDir;

	/* flawfinder - the buffer is used with boundaries, and zeroed. */

	tmpDir = (char *) getenv("TMPDIR"); /* flawfinder: ignore */
	if ((NULL == tmpDir) || ('\0' == tmpDir[0]))
		tmpDir = (char *) getenv("TMP");	/* flawfinder: ignore */
	if ((NULL == tmpDir) || ('\0' == tmpDir[0]))
		tmpDir = "/tmp";

	/*
	 * flawfinder rationale: null and zero-size values of $TMPDIR and
	 * $TMP are rejected, and the destination buffer is bounded.
	 */

	memset(newDirectory, 0, sizeof(newDirectory));
	(void) snprintf(newDirectory, sizeof(newDirectory), "%s/scwXXXXXX", tmpDir);
	/*@-unrecog@ *//* splint doesn't recognise mkdtemp(). */
	if (NULL == mkdtemp(newDirectory))
		return NULL;
	/*@+unrecog@ */

	return stringCopy(state, newDirectory, strlen(newDirectory));	/* flawfinder: ignore */
	/* flawfinder - string is definitely null-terminated. */
}


#if 0
/*
 * This turned out not to be needed (#36).
 * See https://codeberg.org/ivarch/scw/issues/36
 */
/*
 * Increase the buffer size of the given Unix socket as far as possible, for
 * the given socketOption (SO_SNDBUF or SO_RCVBUF).
 */
static void increaseUnixSocketBuffer(int socketDescriptor, int socketOption)
{
	int optionValue, attemptedValue;
	socklen_t optionLength;

	optionValue = -1;
	optionLength = sizeof(optionValue);
	if (getsockopt(socketDescriptor, SOL_SOCKET, socketOption, &optionValue, &optionLength) < 0) {
		perror("getsockopt");
		return;
	}
	debug("%s (%d) = %d", "initial buffer size", socketDescriptor, optionValue);

	for (attemptedValue = 2048; attemptedValue <= 1048576; attemptedValue *= 2) {
		optionValue = attemptedValue;
		optionLength = sizeof(optionValue);
		if (setsockopt(socketDescriptor, SOL_SOCKET, socketOption, &optionValue, optionLength) < 0) {
			debug("%s (%d, %d): %s", "buffer size set failed", socketDescriptor, attemptedValue,
			      strerror(errno));
			break;
		}
	}

	optionValue = -1;
	optionLength = sizeof(optionValue);
	if (getsockopt(socketDescriptor, SOL_SOCKET, socketOption, &optionValue, &optionLength) < 0) {
		debug("%s: %s", "final getsockopt()", strerror(errno));
		return;
	}
	debug("%s (%d) = %d", "final buffer size", socketDescriptor, optionValue);
}
#endif


/*
 * Open a Unix socket connection to the receiver, using it to replace the
 * given output file descriptor.  Returns true on success, false on error;
 * reports the error to stderr.
 */
static bool openUnixSocketSender( /*@unused@ */  __attribute__((unused))
				 struct scwState *state, struct scwCommandProcess *process,
				 struct sockaddr_un *receiverSocketAddress, int replaceDescriptor)
{
	struct sockaddr_un senderSocketAddress;
	int senderDescriptor;

	if (NULL == process->workDir)
		return false;

	senderDescriptor = socket(AF_UNIX, SOCK_DGRAM, 0);
	if (senderDescriptor < 0) {
		fprintf(stderr, "%s: %s[%d]: %s\n", PACKAGE_NAME, "socket", senderDescriptor, strerror(errno));
		return false;
	}
#if 0
/* Probably not needed (#36). */
	increaseUnixSocketBuffer(senderDescriptor, SO_SNDBUF);
#endif

	memset(&senderSocketAddress, 0, sizeof(senderSocketAddress));

	senderSocketAddress.sun_family = AF_UNIX;
	(void) snprintf(senderSocketAddress.sun_path, sizeof(senderSocketAddress.sun_path), "%s/%d",
			process->workDir, replaceDescriptor);
	if (bind
	    (senderDescriptor, (struct sockaddr *) &senderSocketAddress, (socklen_t) sizeof(senderSocketAddress)) < 0) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "bind", senderSocketAddress.sun_path,
			strerror(errno));
		return false;
	}
	if (connect
	    (senderDescriptor, (struct sockaddr *) receiverSocketAddress,
	     (socklen_t) sizeof(*receiverSocketAddress)) < 0) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, "connect",
			senderSocketAddress.sun_path, strerror(errno));
		return false;
	}

	if (shutdown(senderDescriptor, SHUT_RD) < 0) {
		fprintf(stderr, "%s: %s[%d]: %s\n", PACKAGE_NAME, "shutdown", replaceDescriptor, strerror(errno));
		return false;
	}

	if (dup2(senderDescriptor, replaceDescriptor) < 0) {
		fprintf(stderr, "%s: %s[%d]: %s\n", PACKAGE_NAME, "dup2", replaceDescriptor, strerror(errno));
		return false;
	}

	if (close(senderDescriptor) < 0) {
		fprintf(stderr, "%s: %s[%d/%d]: %s\n", PACKAGE_NAME, "close", senderDescriptor,
			replaceDescriptor, strerror(errno));
		return false;
	}

	return true;
}


/*
 * Spawn a relay process which reads from the read end of the given pipe and
 * writes it to the receiver Unix socket using the given file descriptor
 * number as a source.  In the parent, closes the read end of the pipe and
 * sets its value to -1 before returning.
 *
 * Returns true on success, false on error.
 */
static bool spawnRelay(struct scwState *state, struct scwCommandProcess *process,
		       struct sockaddr_un *receiverSocketAddress, int whichDescriptor, int *readPipe)
{
	pid_t relayPid;

	relayPid = (pid_t) fork();

	if (relayPid > 0) {
		/*
		 * Parent process - close the read end of the pipe and
		 * return.
		 */
		(void) close(readPipe[0]);
		readPipe[0] = -1;
		return true;
	} else if (process->pid < 0) {
		/* Fork failure. */
		return false;
	}

	/* Child process - connect to receiver socket and relay. */

	/* First - close the item lock descriptor. */
	if (process->itemLockDescriptor >= 0)
		(void) close(process->itemLockDescriptor);

	/* Close the read ends of the other pipes. */
	if (process->stdoutDescriptor >= 0 && process->stdoutDescriptor != readPipe[0])
		(void) close(process->stdoutDescriptor);
	if (process->stderrDescriptor >= 0 && process->stderrDescriptor != readPipe[0])
		(void) close(process->stderrDescriptor);
	if (process->statusDescriptor >= 0 && process->statusDescriptor != readPipe[0])
		(void) close(process->statusDescriptor);

	/* Close the receiving socket. */
	(void) close(process->combinedDescriptor);

	/* Close all output streams. */
	closeOutputStreams(state);

	/* Close the debug stream. */
	debugWriteOutput("", "", -1, "");

	/* Open the socket to relay to. */
	if (!openUnixSocketSender(state, process, receiverSocketAddress, whichDescriptor)) {
		exit(EXIT_FAILURE);
	}

	/* Close standard streams. */
	{
		int closeDescriptor;
		for (closeDescriptor = 0; closeDescriptor < 4; closeDescriptor++) {
			if (closeDescriptor == readPipe[0])
				continue;
			if (closeDescriptor == whichDescriptor)
				continue;
			(void) close(closeDescriptor);
		}
	}

	/* Relay data from the input to the socket until input EOF. */
	while (true) {
		char transferBuffer[SCW_MAX_RELAYBUFFER];	/* flawfinder: ignore */
		ssize_t bytesRead;
		size_t offset;
		size_t remaining;

		bytesRead = read(readPipe[0], transferBuffer, sizeof(transferBuffer));	/* flawfinder: ignore */
		/* flawfinder - read() is bounded, to stay within the buffer. */
		if (bytesRead < 0 && errno == EINTR) {
			perror(PACKAGE_NAME);
			continue;
		} else if (bytesRead < 0 && errno != EINTR) {
			perror(PACKAGE_NAME);
			break;
		} else if (0 == bytesRead) {
			/* EOF. */
			break;
		}

		remaining = (size_t) bytesRead;
		offset = 0;

		while (remaining > 0) {
			ssize_t bytesWritten;

			bytesWritten = send(whichDescriptor, transferBuffer + offset, remaining, 0);

			if (bytesWritten < 0) {
				if ((EINTR == errno) || (EAGAIN == errno)) {
					continue;
				}
				exit(EXIT_FAILURE);
			}
			if (bytesWritten < 1)
				exit(EXIT_FAILURE);

			remaining -= bytesWritten;
			offset += bytesWritten;
		}
	}

	exit(EXIT_SUCCESS);
}


/*
 * Spawn the child process for the currently selected item's command,
 * setting up the appropriate communications so that its output can be
 * received according to the selected strategy.
 *
 * In the child process, the itemLockDescriptor is closed beforehand, so the
 * child doesn't inherit the item lock.
 *
 * Populates *process.  Returns false if the process could not be created.
 */
static bool spawnCommand(struct scwState *state, struct scwCommandProcess *process)
{
	struct sockaddr_un receiverSocketAddress;
	int outputPipe[2] = { -1, -1 };
	int errorPipe[2] = { -1, -1 };
	int statusPipe[2] = { -1, -1 };
	bool usingPipes, usingSockets;
	bool usingStatusStream;

	memset(&receiverSocketAddress, 0, sizeof(receiverSocketAddress));

	process->pid = 0;
	process->ended = true;

	/* Early empty return if there's no command defined. */
	if (NULL == process->settings->command.expandedValue || process->settings->command.expandedLength < 1)
		return true;

	/* Determine whether we need a status stream. */
	usingStatusStream = false;
	if (SCW_STREAM_STATUS == state->streamForStatus)
		usingStatusStream = true;

	debug("%s: %s", "usingStatusStream", usingStatusStream ? "true" : "false");

	/* Determine whether we need pipes. */
	usingPipes = true;
	usingSockets = false;
	switch (process->strategy) {
	case SCW_RECEIVER_PIPE:
		usingPipes = true;
		usingSockets = false;
		break;
	case SCW_RECEIVER_UNIXSOCKET:
		usingPipes = false;
		usingSockets = true;
		break;
	case SCW_RECEIVER_RELAY:
		usingPipes = true;
		usingSockets = true;
		break;
	case SCW_RECEIVER_AUTO:
		/* Should not be reachable. */
		debug("%s", "reached SCW_RECEIVER_AUTO");
		break;
	}

	if (usingPipes) {
		/* Create the pipes for reading from the child process. */
		if (pipe(outputPipe) < 0)
			return false;
		if (pipe(errorPipe) < 0) {
			int oldError = errno;
			(void) close(outputPipe[0]);
			(void) close(outputPipe[1]);
			errno = oldError;
			return false;
		}
		if (usingStatusStream && pipe(statusPipe) < 0) {
			int oldError = errno;
			(void) close(outputPipe[0]);
			(void) close(outputPipe[1]);
			(void) close(errorPipe[0]);
			(void) close(errorPipe[1]);
			errno = oldError;
			return false;
		}

		/* Make the read ends of the pipes available to the caller. */
		process->stdoutDescriptor = outputPipe[0];
		process->stderrDescriptor = errorPipe[0];
		if (usingStatusStream)
			process->statusDescriptor = statusPipe[0];

		debug("%s: %d", "(usingPipes) stdoutDescriptor", process->stdoutDescriptor);
		debug("%s: %d", "(usingPipes) stderrDescriptor", process->stderrDescriptor);
		debug("%s: %d", "(usingPipes) statusDescriptor", process->statusDescriptor);
	}

	if (usingSockets) {

		/* Create a Unix socket for reading from the child process. */

		/* The sockets will live in a temporary directory. */

		process->workDir = createTemporaryDirectory(state);
		if (NULL == process->workDir)
			return false;

		/* Open and bind the receiver socket. */

		process->combinedDescriptor = socket(AF_UNIX, SOCK_DGRAM, 0);
		if (process->combinedDescriptor < 0) {
			int oldError = errno;
			(void) rmdir(process->workDir);
			errno = oldError;
			return false;
		}
#if 0
/* Probably not needed (#36). */
		increaseUnixSocketBuffer(process->combinedDescriptor, SO_RCVBUF);
#endif

		receiverSocketAddress.sun_family = AF_UNIX;
		(void) snprintf(receiverSocketAddress.sun_path, sizeof(receiverSocketAddress.sun_path), "%s/receiver",
				process->workDir);
		if (bind
		    (process->combinedDescriptor, (struct sockaddr *) &receiverSocketAddress,
		     (socklen_t) sizeof(receiverSocketAddress)) < 0) {
			int oldError = errno;
			(void) close(process->combinedDescriptor);
			process->combinedDescriptor = -1;
			(void) rmdir(process->workDir);
			errno = oldError;
			return false;
		}

		/*
		 * Note that on FreeBSD, this shutdown() call gives an error
		 * "socket not connected", so we ignore errors here.
		 */
		(void) shutdown(process->combinedDescriptor, SHUT_WR);
#if 0
		if (shutdown(process->combinedDescriptor, SHUT_WR) < 0) {
			int oldError = errno;
			(void) close(process->combinedDescriptor);
			process->combinedDescriptor = -1;
			(void) unlink(receiverSocketAddress.sun_path);
			(void) rmdir(process->workDir);
			errno = oldError;
			return false;
		}
#endif

		debug("%s: %d", "(usingSockets) combinedDescriptor", process->combinedDescriptor);
	}

	process->pid = (pid_t) fork();

	if (process->pid > 0) {
		/* Parent process. */

		process->ended = false;

		if (usingPipes) {
			/* Close the write ends of the pipes. */
			(void) close(outputPipe[1]);
			(void) close(errorPipe[1]);
			if (usingStatusStream)
				(void) close(statusPipe[1]);
		}

		if (usingPipes && usingSockets) {
			/*
			 * In relay mode, create a child process for each
			 * pipe, which reads from that pipe and relays it to
			 * the Unix socket, and close the read end of each
			 * pipe in the parent.
			 */
			bool relayCreationFailed = false;

			if (!spawnRelay(state, process, &receiverSocketAddress, 1, outputPipe)) {
				relayCreationFailed = true;
			} else if (!spawnRelay(state, process, &receiverSocketAddress, 2, errorPipe)) {
				relayCreationFailed = true;
			} else if (usingStatusStream) {
				if (!spawnRelay(state, process, &receiverSocketAddress, 3, statusPipe)) {
					relayCreationFailed = true;
				}
			}

			/*
			 * The caller shouldn't try to read from the pipes.
			 *
			 * We set these here and not above, so that
			 * spawnRelay's child processes can close their
			 * copies of these.
			 */
			process->stdoutDescriptor = -1;
			process->stderrDescriptor = -1;
			process->statusDescriptor = -1;

			if (relayCreationFailed) {
				int oldError = errno;
				(void) close(process->combinedDescriptor);
				process->combinedDescriptor = -1;
				(void) unlink(receiverSocketAddress.sun_path);
				(void) rmdir(process->workDir);
				if (outputPipe[0] > 0)
					(void) close(outputPipe[0]);
				if (errorPipe[0] > 0)
					(void) close(errorPipe[0]);
				if (usingStatusStream && statusPipe[0] > 0)
					(void) close(statusPipe[0]);
				errno = oldError;
				return false;
			}
		}

		return true;

	} else if (process->pid < 0) {
		/* Fork failure. */
		int oldError;

		oldError = errno;
		if (usingPipes) {
			(void) close(outputPipe[1]);
			(void) close(errorPipe[1]);
			if (usingStatusStream)
				(void) close(statusPipe[1]);
		}
		errno = oldError;

		return false;
	}

	/* Child process. */

	/* Close the item lock descriptor. */
	if (process->itemLockDescriptor >= 0)
		(void) close(process->itemLockDescriptor);

	if (usingPipes) {
		/* Close the read ends of the pipes we just created. */
		(void) close(outputPipe[0]);
		(void) close(errorPipe[0]);
		if (usingStatusStream)
			(void) close(statusPipe[0]);
	}

	if (usingSockets) {
		/* Close the receiving socket. */
		(void) close(process->combinedDescriptor);
	}

	/* Close all output streams. */
	closeOutputStreams(state);

	/* Close the debug stream. */
	debugWriteOutput("", "", -1, "");

	if (process->closeInput) {
		/* Replace stdin with /dev/null. */

		int nullDescriptor;

		nullDescriptor = open("/dev/null", O_RDONLY	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
				      | O_NOFOLLOW
#endif
		    );
		/*
		 * flawfinder warns of symlinks, redirections, etc when
		 * using open(); this is a fixed path, in read mode, and we
		 * use O_NOFOLLOW where available, so the risk is
		 * negligible.
		 */
		if ((nullDescriptor < 0) || (dup2(nullDescriptor, STDIN_FILENO) < 0)) {
			perror(PACKAGE_NAME);
			exit(EXIT_FAILURE);
		}
		(void) close(nullDescriptor);
	}

	if (usingPipes) {
		/*
		 * Replace stdout and stderr with the write ends of the
		 * pipes created earlier.
		 */
		if ((dup2(outputPipe[1], STDOUT_FILENO) < 0) || (dup2(errorPipe[1], STDERR_FILENO) < 0)) {
			perror(PACKAGE_NAME);
			exit(EXIT_FAILURE);
		}

		/*
		 * If using a status descriptor, and it's not already file
		 * descriptor 3, duplicate it to file descriptor 3.
		 */
		if (usingStatusStream && (3 != statusPipe[1]) && (dup2(statusPipe[1], 3) < 0)) {
			perror(PACKAGE_NAME);
			exit(EXIT_FAILURE);
		}

		/* Close the original pipes we just duplicated. */
		(void) close(outputPipe[1]);
		(void) close(errorPipe[1]);
		if (usingStatusStream && 3 != statusPipe[1])
			(void) close(statusPipe[1]);
	}

	/*
	 * Relay mode is "usingSockets && usingPipes", but in relay mode the
	 * child process running the command only uses the pipes, so in
	 * relay mode we don't do the part below, as we're using the pipes
	 * from the section above.
	 */

	if (usingSockets && !usingPipes) {
		int replaceDescriptor;

		/*
		 * Replace stdout, stderr, and (if using status) fd 3, with
		 * sockets connected to the receiver we opened earlier.
		 */

		for (replaceDescriptor = 1; replaceDescriptor <= 3; replaceDescriptor++) {

			/* Skip fd 3 if we're not using a status stream. */
			if (3 == replaceDescriptor && !usingStatusStream)
				continue;

			if (!openUnixSocketSender(state, process, &receiverSocketAddress, replaceDescriptor))
				exit(EXIT_FAILURE);
		}
	}

	if (!usingStatusStream)
		(void) dup2(2, 3);

	(void) execl("/bin/sh", "sh", "-c", process->settings->command.expandedValue, NULL);	/* flawfinder: ignore */
	/*
	 * flawfinder warns that execl() is difficult to use safely. 
	 * There's no way for us to make it safer to run a user-supplied
	 * command here, and we're always running what the user told us to
	 * anyway, so there is no extra access to gain.
	 */
	exit(EXIT_FAILURE);
}


/*
 * Select which receiver strategy to use, and return it.
 */
static scwReceiverStrategy selectReceiverStrategy( /*@unused@ */  __attribute__((unused))
						  struct scwState *state)
{
	struct utsname unameInfo;

	memset(&unameInfo, 0, sizeof(unameInfo));
	(void) uname(&unameInfo);

	/* Unix sockets work on FreeBSD and OpenBSD. */
	if ((0 == strncmp(unameInfo.sysname, "FreeBSD", 7))
	    || (0 == strncmp(unameInfo.sysname, "OpenBSD", 7))
	    ) {
		debug("%s (%s)", "compatible BSD - using Unix socket", unameInfo.sysname);
		return SCW_RECEIVER_UNIXSOCKET;
	}

	/*
	 * On Linux, only use Unix sockets directly if SELinux is not
	 * enforcing, because it can interfere with child processes writing
	 * to them, depending on their context.
	 */
	if (0 == strncmp(unameInfo.sysname, "Linux", 5)) {
		FILE *enforceStream;
		int enforcing;

		enforceStream = fileOpenStreamForRead("/sys/fs/selinux/enforce");
		if (NULL == enforceStream) {
			/* No SELinux info - probably no SELinux. */
			debug("%s", "Linux but no SELinux sysfs file - using Unix socket");
			return SCW_RECEIVER_UNIXSOCKET;
		}

		enforcing = 0;
		if (1 != fscanf(enforceStream, "%d", &enforcing))
			enforcing = 0;
		(void) fclose(enforceStream);

		/* If not enforcing, use Unix sockets. */
		if (0 == enforcing) {
			debug("%s", "Linux but SELinux is not enforcing - using Unix socket");
			return SCW_RECEIVER_UNIXSOCKET;
		}

		debug("%s", "Linux, and SELinux is enforcing - using relay");
		return SCW_RECEIVER_RELAY;
	}

	/* The default fallback is the subprocess strategy. */
	debug("%s", "fallback to pipe mode");
	return SCW_RECEIVER_PIPE;
}


/*
 * Run the selected scheduled command.
 *
 * Before the command is run, the concurrency, dependency, conflict, and
 * prerequisite checks must all succeed.
 *
 * The item's concurrency lock must only be active while either the checks
 * are being performed (during which time the CheckLockFile is locked), or
 * while the command really is running.  This is so that everything else can
 * always tell whether this item is running by attempting to lock the item's
 * concurrency lock file after first locking CheckLockFile.
 *
 * Returns zero on success, or an SCW_EXIT_* exit code on error.
 */
int runItem(struct scwState *state)
{
	struct scwSettings combinedSettings;
	const char *metricsDir;
	size_t metricsDirLength;
	int retcode;
	time_t lockCheckStart;
	double elapsed;
	unsigned int totalRunTime;
	int checkLockDescriptor, itemLockDescriptor;
	bool firstCheck, conditionsMet, markAsOverran, markAsFailed;
	char endMessage[256];		 /* flawfinder: ignore */
	bool ranWithoutLock;
	struct scwCommandProcess process;
	bool commandSucceeded, commandFailed, ambiguousExit;

	/*
	 * flawfinder warns about endmessage[] being a statically sized
	 * array that could overflow, but we only use it with a bounded
	 * snprint(), and zero it beforehand to ensure that when using it as
	 * a string, it will be null-terminated.
	 */

	/* Can't run without an item. */
	if (NULL == state->item)
		return SCW_EXIT_ERROR;

	/* Combine the settings for this item. */

	memset(&combinedSettings, 0, sizeof(combinedSettings));
	retcode = combineSettings(state, &combinedSettings);
	if (retcode != 0)
		return retcode;

	expandAllRawValues(state, &combinedSettings);
#ifdef ENABLE_DEBUGGING
	debugOutputAllSettings("combinedSettings", &combinedSettings);
#endif

	/* Check key settings are present. */

	/*@-mustfreefresh@ */
	/* gettext triggers splint warnings we can't resolve. */
	if (NULL == combinedSettings.checkLockFile.expandedValue) {
		fprintf(stderr, "%s: %.*s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength), state->item,
			"CheckLockFile", _("no value set"));
		return SCW_EXIT_ERROR;
	}
	if (NULL == combinedSettings.metricsDir.expandedValue) {
		fprintf(stderr, "%s: %.*s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength), state->item,
			"MetricsDir", _("no value set"));
		return SCW_EXIT_ERROR;
	}
	if (NULL == combinedSettings.command.expandedValue || combinedSettings.command.expandedLength < 1) {
		fprintf(stderr, "%s: %.*s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength), state->item,
			"Command", _("no value set"));
		return SCW_EXIT_RUN_COMMAND;
	}
	/*@+mustfreefresh@ */

	/* Try to ensure that there is a writable metrics directory. */

	metricsDir = combinedSettings.metricsDir.expandedValue;
	metricsDirLength = combinedSettings.metricsDir.expandedLength;

	if ((0 != directoryParentCreate(metricsDir, metricsDirLength))
	    || (0 != directoryCreate(metricsDir, metricsDirLength))
	    || (NULL != metricsDir && 0 != access(metricsDir, R_OK | W_OK | X_OK))) {	/* flawfinder: ignore */

		/*
		 * flawfinder warns of access() and race conditions.  Here,
		 * if the permissions change after we checked, no extra
		 * access is gained, the run just fails more messily, so
		 * there's no risk.
		 */

		/*@-mustfreefresh@ */
		/* gettext triggers splint warnings we can't resolve. */
		if (!state->continueWithoutMetrics) {
			fprintf(stderr, "%s: %.*s: %s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength), state->item,
				"MetricsDir", _("metrics directory unavailable"), _("item cannot run"));
			return SCW_EXIT_ERROR;
		}
		fprintf(stderr, "%s: %.*s: %s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength), state->item,
			"MetricsDir", _("metrics directory unavailable"), _("proceeding without metrics or checks"));
		/*@+mustfreefresh@ */
		metricsDir = NULL;
		metricsDirLength = 0;
		combinedSettings.metricsDir.expandedValue = NULL;
		combinedSettings.metricsDir.expandedLength = 0;
	}

	/* Initialise the command process structure. */

	memset(&process, 0, sizeof(process));
	process.settings = &combinedSettings;
	process.strategy = combinedSettings.enumReceiverStrategy;
	if (SCW_RECEIVER_AUTO == process.strategy)
		process.strategy = selectReceiverStrategy(state);
#if CLOSE_COMMAND_STDIN
	process.closeInput = true;
#else
	process.closeInput = false;
#endif

	/* Determine which stream will produce status updates. */

	state->streamForStatus = SCW_STREAM_STATUS;
	/*@-unrecog@ */
	/* splint doesn't know about strncasecmp(). */
	if ((NULL != combinedSettings.statusMode.expandedValue) && (6 == combinedSettings.statusMode.expandedLength)
	    && (0 == strncasecmp(combinedSettings.statusMode.expandedValue, "stdout", 6))) {
		state->streamForStatus = SCW_STREAM_STDOUT;
	} else if ((NULL != combinedSettings.statusMode.expandedValue)
		   && (6 == combinedSettings.statusMode.expandedLength)
		   && (0 == strncasecmp(combinedSettings.statusMode.expandedValue, "stderr", 6))) {
		state->streamForStatus = SCW_STREAM_STDERR;
	}
	/*@+unrecog@ */
	debug("%s: %d", "streamForStatus", state->streamForStatus);

	/*
	 * A broad outline of the running order follows.  It ignores some
	 * specifics and omits the metrics updates.
	 *
	 * A. Initial setup and early-return checks:
	 *  1. Early return if the item is disabled and forceRun is false.
	 *  2. Parse the output maps.
	 *  3. Run the item's Prerequisite command; early return on failure.
	 *  4. Delay up to RandomDelay seconds if not running on a terminal.
	 *  5. Early return if the MinInterval requirement is not met.
	 *
	 * B. Checks for concurrency locks, dependencies, and conflicts:
	 *  1. Lock the CheckLockFile.
	 *  2. Lock the item's lock file (wait time: ConcurrencyWait).
	 *  3. Check for DependsOn (wait time: DependencyWait).
	 *  4. Check for ConflictsWith (wait time: ConflictWait).
	 *  5. If checks failed, release both locks, repeat until expiry.
	 *
	 * At this point the CheckLockFile and the item concurrency lock
	 * file are both locked.
	 *
	 * C. If all of the above took 1 second or more, perform re-checks:
	 *  1. Early return if the item is disabled and forceRun is false.
	 *  2. Run the item's Prerequisite command; early return on failure.
	 *
	 * With these early returns, both locks are released first.
	 *
	 * D. Run the item's command:
	 *  1. Unlock the CheckLockFile but keep the item concurrency lock.
	 *  2. Open the streams for the item's OutputMap.
	 *  3. Run the item's Command.
	 *  4. Terminate the command if its MaxRunTime is reached.
	 *  5. Record its exit status and so on.
	 *  6. Release the item concurrency lock.
	 *  7. Deliver any output that was spooled for delivery on exit.
	 */

	/*
	 * (A) Initial setup and early-return checks.
	 */

	/* (A1) Check that either the item is enabled or forceRun is true. */
	if (!state->forceRun) {
		if (NULL != metricsDir && fileExistsAt(metricsDir, metricsDirLength, STATIC_STRING("disabled"))) {
			debug("%s", "disabled marker found");
			return SCW_EXIT_RUN_DISABLED;
		}
	}

	/*
	 * Replace the "success-interval" metrics file contents, in case the
	 * operator never runs the "update" action.
	 */
	if (NULL != metricsDir) {
		retcode =
		    fileReplaceContentsAt(metricsDir, metricsDirLength, STATIC_STRING("success-interval"), "%u\n",
					  combinedSettings.numSuccessInterval);
		if (retcode != 0)
			return retcode;
	}

	/* (A2) Parse the OutputMaps. */
	retcode = parseAllOutputMaps(state, &combinedSettings);
	if (retcode != 0)
		return retcode;

	/* (A3) Perform the initial Prerequisite check. */
	if (!prerequisiteCheck(&combinedSettings, -1, -1)) {
		debug("%s", "prerequisite not met");
		return SCW_EXIT_RUN_PREREQUISITE;
	}

	/* (A4) If not on a terminal, honour RandomDelay. */
	debug("%s: %s", "runningOnTerminal", state->runningOnTerminal ? "true" : "false");
	if (combinedSettings.numRandomDelay > 0 && !state->runningOnTerminal) {
		unsigned int delay;

		delay = randomNumber(combinedSettings.numRandomDelay);

		debug("%s: %u", "random delay", delay);

		/* Write a metrics file to say how long we will delay for. */
		if (NULL != metricsDir) {
			(void) fileReplaceContentsAt(metricsDir, metricsDirLength, STATIC_STRING("delay"), "%u\n",
						     delay);
		}

		(void) sleep(delay);
	}

	/* Remove the "delay" metrics file, as we are no longer delaying. */
	if (NULL != metricsDir) {
		(void) fileDeleteAt(metricsDir, metricsDirLength, STATIC_STRING("delay"));
	}

	/* (A5) Check that the MinInterval constraint is met. */
	if (NULL != metricsDir && combinedSettings.numMinInterval > 0) {
		time_t thisItemLastEnded;
		struct stat statBuf;
		double elapsedSinceLastEnded;

		thisItemLastEnded = 0;
		memset(&statBuf, 0, sizeof(statBuf));
		if (0 == fileStatAt(metricsDir, metricsDirLength, STATIC_STRING("ended"), &statBuf)) {
			thisItemLastEnded = (time_t) (statBuf.st_mtime);
		}

		elapsedSinceLastEnded = difftime(time(NULL), thisItemLastEnded);
		if (elapsedSinceLastEnded < (double) (combinedSettings.numMinInterval)) {
			debug("%s - %.0f < %u", "MinInterval constraint not met", elapsedSinceLastEnded,
			      combinedSettings.numMinInterval);
			return SCW_EXIT_RUN_MININTERVAL;
		}
	}

	/*
	 * (B) Checks for concurrency locks, dependencies, and conflicts.
	 */

	lockCheckStart = time(NULL);
	checkLockDescriptor = -1;
	itemLockDescriptor = -1;
	markAsFailed = false;
	markAsOverran = false;
	conditionsMet = false;
	firstCheck = true;

	/* Skip these checks if there's no usable MetricsDir. */
	if (NULL == metricsDir)
		conditionsMet = true;

	while (0 == retcode && !conditionsMet) {
		if (firstCheck) {
			firstCheck = false;
		} else {
			/*
			 * Not the first time around the loop: close all
			 * open locks and wait a short while.
			 *
			 * Note that we always release the item
			 * (concurrency) lock before the check lock, so that
			 * it's always true that while the check lock is
			 * inactive, the item lock can only be active if the
			 * item is actually running.
			 */
			if (itemLockDescriptor >= 0) {
				(void) flock(itemLockDescriptor, LOCK_UN);
				(void) close(itemLockDescriptor);
				itemLockDescriptor = -1;
			}
			if (checkLockDescriptor >= 0) {
				(void) flock(checkLockDescriptor, LOCK_UN);
				(void) close(checkLockDescriptor);
				checkLockDescriptor = -1;
			}
			(void) sleep(1);
		}
		conditionsMet =
		    checkItemCoordination(state, &combinedSettings, lockCheckStart, &checkLockDescriptor,
					  &itemLockDescriptor, &markAsFailed, &markAsOverran, &retcode);
	}

	/* Mark as overran or as failed, if appropriate. */
	if (markAsFailed) {
		debug("%s", "marking as failed");
		if (NULL != metricsDir) {
			(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("failed"), false);
		}
	}
	if (markAsOverran) {
		if (NULL != metricsDir) {
			debug("%s", "marking as overran");
			(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("overran"), false);
		}
	}

	/*
	 * (C) Perform re-checks if time elapsed and conditions were met.
	 */

	if (0 == retcode && difftime(time(NULL), lockCheckStart) > 0.9) {
		/* (C1) Re-check that either the item is enabled or forceRun is true. */
		if (!state->forceRun) {
			if (NULL != metricsDir && fileExistsAt(metricsDir, metricsDirLength, STATIC_STRING("disabled"))) {
				debug("%s", "disabled marker found");
				retcode = SCW_EXIT_RUN_DISABLED;
			}
		}
		/* (C2) Perform another Prerequisite check. */
		if (0 == retcode) {
			if (!prerequisiteCheck(&combinedSettings, itemLockDescriptor, checkLockDescriptor)) {
				debug("%s", "prerequisite not met");
				retcode = SCW_EXIT_RUN_PREREQUISITE;
			}
		}
	}

	/* Early return, releasing all locks, if the conditions were not met. */
	if (0 != retcode) {
		if (itemLockDescriptor >= 0) {
			/* Unlock the item first. */
			(void) flock(itemLockDescriptor, LOCK_UN);
			(void) close(itemLockDescriptor);
			itemLockDescriptor = -1;
		}
		if (checkLockDescriptor >= 0) {
			/* Release the check lock last. */
			(void) flock(checkLockDescriptor, LOCK_UN);
			(void) close(checkLockDescriptor);
			checkLockDescriptor = -1;
		}
		return retcode;
	}

	/*
	 * At this point, CheckLockFile and the item concurrency lock file
	 * are both locked, and we know that conditions for running this
	 * item have been met - or, either lock file was inaccessible, so
	 * we're running anyway.
	 */

	/*
	 * (D) Run the item's command.
	 */

	/* (D1) Release the CheckLockFile lock. */
	if (checkLockDescriptor >= 0) {
		(void) flock(checkLockDescriptor, LOCK_UN);
		(void) close(checkLockDescriptor);
		checkLockDescriptor = -1;
	}

	/*
	 * Now only the item lock file is locked, indicating that the item
	 * is running.
	 */

	/* (D2) Open the output streams for the OutputMaps. */
	retcode = openOutputStreams(state);
	if (0 != retcode) {
		/* Failed to open output - don't start; early return. */
		closeOutputStreams(state);
		removeSpooledOutput(state);
		if (itemLockDescriptor >= 0) {
			(void) flock(itemLockDescriptor, LOCK_UN);
			(void) close(itemLockDescriptor);
			itemLockDescriptor = -1;
		}
		return retcode;
	}

	/* Transfer across the UTC setting. */
	state->timestampUTC = combinedSettings.flagTimestampUTC;

	/* (D3) Fork and exec the item's Command. */
	process.exitStatus = 0;
	process.ended = false;
	process.timedOut = false;
	process.stdoutDescriptor = -1;
	process.stderrDescriptor = -1;
	process.statusDescriptor = -1;
	process.combinedDescriptor = -1;
	process.itemLockDescriptor = itemLockDescriptor;

	if (!spawnCommand(state, &process)) {
		char *errorString = strerror(errno);
		/* Failed to create child process - early return. */
		if (NULL != errorString)
			recordOutput(state, &combinedSettings, process.pid, SCW_STREAM_STDERR,
				     STATIC_STRING(errorString));
		closeOutputStreams(state);
		removeSpooledOutput(state);
		if (itemLockDescriptor >= 0) {
			(void) flock(itemLockDescriptor, LOCK_UN);
			(void) close(itemLockDescriptor);
			itemLockDescriptor = -1;
		}
		return SCW_EXIT_ERROR;
	}

	process.startTime = time(NULL);

	/* Update the "started" metrics file's timestamp. */
	if (NULL != metricsDir) {
		(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("started"), true);
	}

	/* Remove the "overran" metrics file. */
	if (NULL != metricsDir) {
		(void) fileDeleteAt(metricsDir, metricsDirLength, STATIC_STRING("overran"));
	}

	/* Replace the contents of the "pid" metrics file. */
	if (NULL != metricsDir) {
		(void) fileReplaceContentsAt(metricsDir, metricsDirLength, STATIC_STRING("pid"), "%d\n",
					     (int) (process.pid));
	}

	/* Generate a "begin" status message. */
	recordOutput(state, &combinedSettings, process.pid, SCW_STREAM_STATUS, STATIC_STRING("(begin)"));

	/* Record the output of the command, stopping it after MaxRunTime. */
	retcode = recordCommandOutput(state, &process);

	/* Determine the total running time. */
	elapsed = difftime(time(NULL), process.startTime);
	if (elapsed > 0.0) {
		totalRunTime = (unsigned int) elapsed;
	} else {
		totalRunTime = 0;
	}

	/*
	 * Check whether the command ran and succeeded, or ran and failed,
	 * or ran and had an ambiguous exit status (meaning neither success
	 * nor failure).
	 */
	commandSucceeded = false;
	commandFailed = false;
	ambiguousExit = false;
	if (0 == process.exitStatus) {
		commandSucceeded = true;
	} else if (process.exitStatus == (int) (combinedSettings.numAmbiguousExitStatus)) {
		ambiguousExit = true;
	} else {
		commandFailed = true;
	}
	debug("%s=%s, %s=%s, %s=%s", "commandSucceeded", commandSucceeded ? "true" : "false", "commandFailed",
	      commandFailed ? "true" : "false", "ambiguousExit", ambiguousExit ? "true" : "false");

	/* (D5) Write an "end" status message. */
	memset(endMessage, 0, sizeof(endMessage));
	/*@-mustfreefresh@ */
	/* gettext triggers splint warnings we can't resolve. */
	if (ambiguousExit) {
		(void) snprintf(endMessage, sizeof(endMessage), "(end) %s [%d], %s %us", _("ambiguous exit status"),
				process.exitStatus, _("elapsed time"), totalRunTime);
	} else {
		(void) snprintf(endMessage, sizeof(endMessage), "(end) %s (%d), %s %us", _("exit status"),
				process.exitStatus, _("elapsed time"), totalRunTime);
	}
	/*@+mustfreefresh@ */
	recordOutput(state, &combinedSettings, process.pid, SCW_STREAM_STATUS, STATIC_STRING(endMessage));

	/* Update the "ended" metrics file's timestamp. */
	if (NULL != metricsDir) {
		(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("ended"), true);
	}

	/* Replace the contents of the "run-time" metrics file. */
	if (NULL != metricsDir) {
		(void) fileReplaceContentsAt(metricsDir, metricsDirLength, STATIC_STRING("run-time"), "%u\n",
					     totalRunTime);
	}

	/* Remove the "pid" metrics file. */
	if (NULL != metricsDir) {
		(void) fileDeleteAt(metricsDir, metricsDirLength, STATIC_STRING("pid"));
	}

	/* Close all OutputMap streams. */
	closeOutputStreams(state);

	/*
	 * If the command succeeded, update its "succeeded" metrics file's
	 * timestamp, and remove the "failed" metrics file.
	 *
	 * If the command failed, create its "failed" metrics file if it
	 * doesn't already exist; don't update the file's timestamp if it
	 * already does exist, so the timestamp indicates when the command
	 * first failed.
	 *
	 * Do neither of these if the exit status matched the
	 * AmbiguousExitStatus.
	 */
	if (commandSucceeded && (NULL != metricsDir)) {
		(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("succeeded"), true);
		(void) fileDeleteAt(metricsDir, metricsDirLength, STATIC_STRING("failed"));
	} else if (commandFailed && (NULL != metricsDir)) {
		(void) fileCreateAt(metricsDir, metricsDirLength, STATIC_STRING("failed"), false);
	}

	/* (D6) Release the item's concurrency lock. */
	ranWithoutLock = true;
	if (itemLockDescriptor >= 0) {
		ranWithoutLock = false;
		(void) flock(itemLockDescriptor, LOCK_UN);
		(void) close(itemLockDescriptor);
		itemLockDescriptor = -1;
	}

	/* If the command timed out, set the return code. */
	if (0 == retcode && process.timedOut)
		retcode = SCW_EXIT_RUN_TIMEOUT;

	/* (D7) Process OutputMap spools, e.g. send email. */
	sendSpooledOutput(state, &combinedSettings, process.pid,
			  ((retcode == SCW_EXIT_RUN_TIMEOUT) || commandFailed) ? true : false);
	removeSpooledOutput(state);

	/* If the command ran without timeout, but failed, set the return code. */
	if (0 == retcode && commandFailed)
		retcode = SCW_EXIT_RUN_FAILED;

	/*
	 * Adjust the return code if there was a problem with the metrics
	 * file or the item lock.
	 */
	if (ranWithoutLock || NULL == metricsDir) {
		if (0 == retcode) {
			retcode = SCW_EXIT_RUN_NOCHECKS_OK;
		} else if (retcode < 3) {
			retcode = SCW_EXIT_RUN_NOCHECKS_FAIL;
		}
	}

	return retcode;
}
