/*
 * Random number generation.
 *
 * Copyright 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>


/*
 * Return a pseudo-random number that is less than the given maximum, or
 * unlimited if the maximum is zero.
 *
 * The random number generator is seeded automatically the first time this
 * function is called.
 */
/*@-unrecog@ */
/* splint doesn't know about srandom(), random(), clock_gettime(). */
unsigned int randomNumber(unsigned int upperBoundary)
{
	static bool firstRun = true;
	unsigned int value;

	if (firstRun) {
		unsigned int seed;
#if HAVE_CLOCK_GETTIME
		struct timespec monotonicTime;
#endif

		/* First run - seed the PRNG. */

		seed = (unsigned int) getpid();
		seed ^= (unsigned int) time(NULL);

#if HAVE_CLOCK_GETTIME
		memset(&monotonicTime, 0, sizeof(monotonicTime));
		(void) clock_gettime(CLOCK_MONOTONIC, &monotonicTime);
		seed ^= monotonicTime.tv_sec;
		seed ^= monotonicTime.tv_nsec;
#endif

		srandom(seed);		    /* flawfinder: ignore */

		firstRun = false;
	}

	value = random();		    /* flawfinder: ignore */

	/*
	 * flawfinder warns that srandom() and random() are not sufficiently
	 * random for security-related purposes (CWE-327).  However we're
	 * just using them for a random delay to avoid system congestion, or
	 * for MIME attachment boundaries to make them relatively unique,
	 * for which they are good enough - so we ignore the warning.
	 */

	if (upperBoundary > 0) {
		value = value % upperBoundary;
	}

	return value;
}

/*@+unrecog@ */
