/*
 * Functions relating to recording the output of an item.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include "sha-256.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/file.h>
#ifndef SPLINT
/* splint chokes on syslog.h */
#include <syslog.h>
#endif


/*
 * Parse one OutputMap setting to populate the target scwOutput structure.
 * If the target pointer is NULL, only the syntax is checked.  Returns a
 * nonzero SCW_EXIT_* exit status on error.
 */
int parseOutputMap( /*@null@ */ struct scwState *state, const char *string, size_t length,
		   /*@null@ */ struct scwOutput *target)
{
	size_t position, wordStart, wordLength, keywordIndex;
	struct {
		/*@null@ */ const char *keyword;
		scwOutputFormat format;
	} formatKeywords[] = {
		{ "raw", SCW_FORMAT_RAW },
		{ "stamped", SCW_FORMAT_STAMPED },
		{ "json", SCW_FORMAT_JSON },
		{ "form", SCW_FORMAT_FORM },
		{ NULL, 0 }
	};
	/*@-unrecog@ */
	/* we hid syslog.h from splint */
	struct {
		/*@null@ */ const char *name;
		int value;
	} syslogFacility[] = {
		{ "auth", LOG_AUTH },
		{ "authpriv", LOG_AUTHPRIV },
		{ "cron", LOG_CRON },
		{ "daemon", LOG_DAEMON },
		{ "lpr", LOG_LPR },
		{ "mail", LOG_MAIL },
		{ "news", LOG_NEWS },
		{ "user", LOG_USER },
		{ "uucp", LOG_UUCP },
		{ "local0", LOG_LOCAL0 },
		{ "local1", LOG_LOCAL1 },
		{ "local2", LOG_LOCAL2 },
		{ "local3", LOG_LOCAL3 },
		{ "local4", LOG_LOCAL4 },
		{ "local5", LOG_LOCAL5 },
		{ "local6", LOG_LOCAL6 },
		{ "local7", LOG_LOCAL7 },
		{ NULL, 0 }
	};
	struct {
		/*@null@ */ const char *name;
		int value;
	} syslogLevel[] = {
		{ "debug", LOG_DEBUG },
		{ "info", LOG_INFO },
		{ "notice", LOG_NOTICE },
		{ "warning", LOG_WARNING },
		{ "warn", LOG_WARNING },
		{ "err", LOG_ERR },
		{ "error", LOG_ERR },
		{ "crit", LOG_CRIT },
		{ "critical", LOG_CRIT },
		{ "alert", LOG_ALERT },
		{ "emerg", LOG_EMERG },
		{ "emergency", LOG_EMERG },
		{ "panic", LOG_EMERG },
		{ NULL, 0 }
	};
	/*@+unrecog@ */
	bool formatSpecified;
	scwOutputFormat format;
	scwOutputDestinationType destinationType;

	if (NULL == string)
		return SCW_EXIT_BAD_CONFIG;

	if (NULL != target) {
		memset(target, 0, sizeof(*target));
		target->writeDescriptor = -1;
		target->outputFileCheck = false;
	}

	/* First word is the stream spec. */
	position = 0;
	while (position < length && !isspace(string[position])) {
		switch (string[position]) {
		case 'o':
		case 'O':
		case '-':
			debug("%.*s: %s", length, string, "selected STDOUT");
			if (NULL != target)
				target->streamsIncluded |= SCW_STREAM_STDOUT;
			break;
		case 'e':
		case 'E':
			debug("%.*s: %s", length, string, "selected STDERR");
			if (NULL != target)
				target->streamsIncluded |= SCW_STREAM_STDERR;
			break;
		case 's':
		case 'S':
			debug("%.*s: %s", length, string, "selected STATUS");
			if (NULL != target)
				target->streamsIncluded |= SCW_STREAM_STATUS;
			break;
		case '!':
			debug("%.*s: %s", length, string, "set SEND-ON-FAILURE flag");
			if (NULL != target)
				target->sendOnFailure = true;
			break;
		default:
			debug("%.*s: %s: %c", length, string, "unknown character", string[position]);
			return SCW_EXIT_BAD_CONFIG;
		}
		position++;
	}

	/* Skip to the next word. */
	while (position < length && isspace(string[position]))
		position++;

	/* Find the length of the next word. */
	wordStart = position;
	while (position < length && !isspace(string[position]))
		position++;
	wordLength = position - wordStart;

	debug("%.*s: %s: [%.*s]", length, string, "second word", wordLength, string + wordStart);

	formatSpecified = false;
	format = 0;
	for (keywordIndex = 0; (!formatSpecified) && (NULL != formatKeywords[keywordIndex].keyword); keywordIndex++) {
		if (strlen(formatKeywords[keywordIndex].keyword) != wordLength)	/* flawfinder: ignore */
			continue;
		/*
		 * flawfinder warns about strlen() in case it's given an
		 * unterminated string, but the strings in this structure
		 * are all null-terminated, so we can ignore that.
		 */
		/*@-unrecog@ */
		/* splint doesn't recognise strncasecmp(). */
		if (0 != strncasecmp(string + wordStart, formatKeywords[keywordIndex].keyword, wordLength))
			continue;
		/*@+unrecog@ */
		format = formatKeywords[keywordIndex].format;
		if (NULL != target)
			target->format = format;
		formatSpecified = true;
		debug("%.*s: %s: %s", length, string, "format", formatKeywords[keywordIndex].keyword);
	}

	if (!formatSpecified) {
		debug("%.*s: %s: %.*s", length, string, "unknown format", wordLength, string + wordStart);
		return SCW_EXIT_BAD_CONFIG;
	}

	/* Skip to the next word. */
	while (position < length && isspace(string[position]))
		position++;
	wordStart = position;

	/* Find the length of the rest of the string. */
	wordLength = length - wordStart;

	/* Make sure it's not empty. */
	if (wordLength < 1) {
		debug("%s", "empty remainder (no destination)");
		return SCW_EXIT_BAD_CONFIG;
	}

	debug("%.*s: %s: [%.*s]", length, string, "destination", wordLength, string + wordStart);

	/* Allocate a copy of the destination, if there is a target. */
	if (NULL != target && NULL != state) {
		char *destinationCopy = stringCopy(state, string + wordStart, wordLength);
		if (NULL == destinationCopy)
			return SCW_EXIT_ERROR;
		target->destination = destinationCopy;
	}

	/*
	 * Determine the destination type by looking at its content.
	 */
	destinationType = SCW_DESTINATION_FILE;
	if (wordLength > 9 && (stringStartsWith(string + wordStart, "http://")
			       || stringStartsWith(string + wordStart, "https://"))) {

		/* URLs start "http://" or "https://". */
		debug("%s", "SCW_DESTINATION_URL");
		destinationType = SCW_DESTINATION_URL;

	} else if (NULL != memchr(string + wordStart, (int) '@', wordLength)
		   && NULL == memchr(string + wordStart, (int) '/', wordLength)) {

		/* Emails contain at least one '@' and no '/'. */

		debug("%s", "SCW_DESTINATION_EMAIL");
		destinationType = SCW_DESTINATION_EMAIL;

	} else if (NULL != memchr(string + wordStart, (int) '.', wordLength)
		   && NULL == memchr(string + wordStart, (int) '/', wordLength)) {

		/* Syslog "facility.level". */
		size_t nameIndex;
		int syslogPriority;
		bool syslogWordFound;

		debug("%s", "SCW_DESTINATION_SYSLOG");
		destinationType = SCW_DESTINATION_SYSLOG;

		syslogPriority = 0;

		syslogWordFound = false;
		for (nameIndex = 0; syslogFacility[nameIndex].name != NULL && !syslogWordFound; nameIndex++) {
			size_t nameLength = strlen(syslogFacility[nameIndex].name);	/* flawfinder: ignore */
			/*
			 * flawfinder warns about strlen() - it's being
			 * called on a static string, always
			 * null-terminated, so no risk.
			 */
			if (wordLength <= nameLength + 1)
				continue;
			if ('.' != string[wordStart + nameLength])
				continue;
			/*@-unrecog@ */
			/* splint doesn't recognise strncasecmp(). */
			if (0 != strncasecmp(syslogFacility[nameIndex].name, string + wordStart, nameLength))
				continue;
			/*@+unrecog@ */

			debug("%s: %s (%d)", "syslog facility", syslogFacility[nameIndex].name,
			      syslogFacility[nameIndex].value);
			syslogPriority |= syslogFacility[nameIndex].value;
			wordStart += nameLength + 1;
			wordLength -= nameLength + 1;
			syslogWordFound = true;
		}

		if (!syslogWordFound) {
			debug("%.*s: %s: %.*s", length, string, "unknown syslog facility", wordLength,
			      string + wordStart);
			return SCW_EXIT_BAD_CONFIG;
		}

		syslogWordFound = false;
		for (nameIndex = 0; syslogLevel[nameIndex].name != NULL && !syslogWordFound; nameIndex++) {
			size_t nameLength = strlen(syslogLevel[nameIndex].name);	/* flawfinder: ignore */
			/* flawfinder - strlen(); see above. */
			if (wordLength != nameLength)
				continue;
			/*@-unrecog@ */
			/* splint doesn't recognise strncasecmp(). */
			if (0 != strncasecmp(syslogLevel[nameIndex].name, string + wordStart, nameLength))
				continue;
			/*@+unrecog@ */

			debug("%s: %s (%d)", "syslog level", syslogLevel[nameIndex].name, syslogLevel[nameIndex].value);
			syslogPriority |= syslogLevel[nameIndex].value;
			syslogWordFound = true;
		}

		if (!syslogWordFound) {
			debug("%.*s: %s: %.*s", length, string, "unknown syslog level", wordLength, string + wordStart);
			return SCW_EXIT_BAD_CONFIG;
		}

		if (NULL != target)
			target->syslogPriority = syslogPriority;
	}

	if (NULL != target)
		target->destinationType = destinationType;

	/*
	 * Enable output check mode if the filename starts with ">".
	 */
	if (SCW_DESTINATION_FILE == destinationType && NULL != target && NULL != target->destination
	    && '>' == target->destination[0] && '\0' != target->destination[1]) {
		debug("%s", "output check mode");
		target->outputFileCheck = true;
		target->nextFileCheck = 1 + time(NULL);
		target->destination = 1 + target->destination;
	}

	/*
	 * Check that the destination type and format make sense together.
	 */
	switch (format) {
	case SCW_FORMAT_RAW:
	case SCW_FORMAT_STAMPED:
		switch (destinationType) {
		case SCW_DESTINATION_FILE:
		case SCW_DESTINATION_SYSLOG:
		case SCW_DESTINATION_EMAIL:
			break;
		case SCW_DESTINATION_URL:
			debug("%s", "cannot use raw or stamped format with a URL");
			return SCW_EXIT_BAD_CONFIG;
		}
		break;
	case SCW_FORMAT_JSON:
	case SCW_FORMAT_FORM:
		switch (destinationType) {
		case SCW_DESTINATION_URL:
			break;
		case SCW_DESTINATION_FILE:
		case SCW_DESTINATION_SYSLOG:
		case SCW_DESTINATION_EMAIL:
			debug("%s", "can only use json or form format with a URL");
			return SCW_EXIT_BAD_CONFIG;
		}
		break;
	}

	return 0;
}


/*
 * Parse the OutputMaps in the settings to populate state->output[],
 * returning zero on success, or a nonzero SCW_EXIT_* exit status on error.
 */
int parseAllOutputMaps(struct scwState *state, struct scwSettings *settings)
{
	size_t mapIndex;
	int retcode;

	for (mapIndex = 0; mapIndex < settings->countOutputMaps; mapIndex++) {
		const char *string;
		size_t length;

		string = settings->outputMap[mapIndex].expandedValue;
		length = settings->outputMap[mapIndex].expandedLength;
		if (NULL == string) {
			string = settings->outputMap[mapIndex].rawValue;
			length = settings->outputMap[mapIndex].rawLength;
		}
		if (NULL == string) {
			debug("%s[%d]: %s", "outputmap", mapIndex, "null value");
			return SCW_EXIT_BAD_CONFIG;
		}
		retcode = parseOutputMap(state, string, length, &(state->output[mapIndex]));
		if (0 != retcode) {
			return retcode;
		}
	}

	state->outputCount = settings->countOutputMaps;

	return 0;
}


/*
 * Open all output streams that the OutputMap defines, where appropriate
 * (file streams for files and spools, syslog setup).  Returns a nonzero
 * SCW_EXIT_* code on error.
 */
int openOutputStreams(struct scwState *state)
{
	size_t mapIndex;
	bool syslogOpened = false;

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {
		if (state->output[mapIndex].sendOnFailure
		    || SCW_DESTINATION_EMAIL == state->output[mapIndex].destinationType
		    || SCW_DESTINATION_URL == state->output[mapIndex].destinationType) {

			/*
			 * All streams flagged as send-on-failure, and all
			 * email and URL destinations, write to a temporary
			 * spool file.
			 */
			char *tempPath = NULL;
			int tempDescriptor = -1;

			if (!tempFileCreate(state, &tempPath, &tempDescriptor))
				return SCW_EXIT_ERROR;

			state->output[mapIndex].temporarySpool = tempPath;
			state->output[mapIndex].writeDescriptor = tempDescriptor;

			debug("%s[%d]: %s: %s[%d]", "outputMap", mapIndex, "temporary spool", tempPath, tempDescriptor);

		} else if (SCW_DESTINATION_FILE == state->output[mapIndex].destinationType) {

			/* Append to the destination file. */

			int outputDescriptor;

			/* Try to make sure the log directory exists. */
			(void) directoryParentCreate(STATIC_STRING(state->output[mapIndex].destination));

			/* Open the file. */
			outputDescriptor = fileOpenForAppend(state->output[mapIndex].destination);
			if (outputDescriptor < 0) {
				/*@-mustfreefresh@ */
				/* gettext triggers splint warnings we can't resolve. */
				if (!state->continueWithoutOutputFile) {
					fprintf(stderr, "%s: %.*s: %s: %s: %s\n", PACKAGE_NAME,
						(int) (state->itemLength), NULL == state->item ? "(null)" : state->item,
						"OutputMap", _("cannot write to file"), _("item cannot run"));
					return SCW_EXIT_ERROR;
				}
				fprintf(stderr, "%s: %.*s: %s: %s: %s\n", PACKAGE_NAME, (int) (state->itemLength),
					NULL == state->item ? "(null)" : state->item, "OutputMap",
					_("cannot write to file"), _("skipping this output"));
				/*@+mustfreefresh@ */
			}
			state->output[mapIndex].writeDescriptor = outputDescriptor;

			debug("%s[%d]: %s: %s[%d]", "outputMap", mapIndex, "file", state->output[mapIndex].destination,
			      outputDescriptor);

			if (state->output[mapIndex].outputFileCheck) {
				/*
				 * In output check mode, record the inode
				 * number of the file we just opened, so
				 * that before each write, we can check
				 * whether the destination file has been
				 * rotated out by comparing inode numbers.
				 */
				struct stat statBuf;
				if (0 != fstat(state->output[mapIndex].writeDescriptor, &statBuf)) {
					perror(PACKAGE_NAME);
				} else {
					state->output[mapIndex].destinationInode = (ino_t) (statBuf.st_ino);
				}
			}

		} else if (SCW_DESTINATION_SYSLOG == state->output[mapIndex].destinationType && !syslogOpened) {

			/* Open the system logger if not yet opened. */
			/*@-unrecog @ */
			/* we hid syslog.h from splint */
			openlog(PACKAGE_NAME, 0, LOG_DAEMON);
			/*@+unrecog @ */
			syslogOpened = true;

			debug("%s[%d]: %s", "outputMap", mapIndex, "syslog");

		}
	}

	return 0;
}


/*
 * Close all open output file streams.  Any temporary spool files are
 * closed, but not deleted.
 */
void closeOutputStreams(struct scwState *state)
{
	size_t mapIndex;

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {
		if (state->output[mapIndex].writeDescriptor < 0)
			continue;
		if (0 != close(state->output[mapIndex].writeDescriptor)) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
				NULL ==
				state->output[mapIndex].temporarySpool ? state->output[mapIndex].
				destination : state->output[mapIndex].temporarySpool, strerror(errno));
		}
		state->output[mapIndex].writeDescriptor = -1;
	}
}


/*
 * Append the contents of the temporary spool to the destination file.
 */
static void copySpoolToFile(const char *spoolFile, const char *destinationFile)
{
	FILE *readStream = NULL;
	int writeDescriptor = -1;
	FILE *writeStream = NULL;
	char line[SCW_MAX_LINELENGTH];	 /* flawfinder: ignore */

	/*
	 * flawfinder rationale: zeroed before use, and only written to by
	 * fgets() which is bounded by the size passed to it.
	 */

	debug("%s -> %s", spoolFile, destinationFile);

	readStream = fileOpenStreamForRead(spoolFile);
	if (NULL == readStream) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, spoolFile, strerror(errno));
		return;
	}
	writeDescriptor = fileOpenForAppend(destinationFile);
	if (writeDescriptor < 0) {
		(void) fclose(readStream);
		return;
	}
	writeStream = fdopen(writeDescriptor, "a");
	if (NULL == writeStream) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, destinationFile, strerror(errno));
		(void) close(writeDescriptor);
		(void) fclose(readStream);
		return;
	}

#if HAVE_POSIX_FADVISE
	/* Advise the OS that we will only be reading sequentially. */
	(void) posix_fadvise(fileno(readStream), 0, 0, POSIX_FADV_SEQUENTIAL);
#endif

	memset(line, 0, sizeof(line));
	while (NULL != fgets(line, (int) (sizeof(line)), readStream)) {
		(void) fprintf(writeStream, "%s", line);
	}
	(void) fclose(readStream);
	if (0 != fclose(writeStream)) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, destinationFile, strerror(errno));
	}
}


/*
 * Write the contents of the temporary spool to syslog.
 */
static void copySpoolToSyslog(struct scwState *state, const char *spoolFile, pid_t pid, int syslogPriority)
{
	FILE *readStream = NULL;
	char line[SCW_MAX_LINELENGTH];	 /* flawfinder: ignore */

	debug("%s -> %s", spoolFile, "syslog");

	/*
	 * flawfinder rationale: zeroed before use, and only written to by
	 * fgets() which is bounded by the size passed to it.
	 */

	readStream = fileOpenStreamForRead(spoolFile);
	if (NULL == readStream) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, spoolFile, strerror(errno));
		return;
	}

#if HAVE_POSIX_FADVISE
	/* Advise the OS that we will only be reading sequentially. */
	(void) posix_fadvise(fileno(readStream), 0, 0, POSIX_FADV_SEQUENTIAL);
#endif

	memset(line, 0, sizeof(line));
	while (NULL != fgets(line, (int) (sizeof(line)), readStream)) {
		char *eol;

		/* Remove the trailing newline. */
		eol = strchr(line, '\n');
		if (NULL != eol)
			eol[0] = '\0';

		/*@-unrecog@ *//* we hid syslog.h from splint. */
		syslog(syslogPriority, "%.*s[%d]: %s", (int) (state->itemLength), state->item, (int) pid, line);
		/*@+unrecog@ */
	}
	(void) fclose(readStream);
}


/*
 * Send the temporary spool by email.
 *
 * This done by writing a temporary message file containing a formatted
 * email message derived from the contents of "spoolFile", and then running
 * the command settings->sendmail.expandedValue with that temporary file as
 * its standard input.
 */
static void sendSpoolByEmail(struct scwState *state, struct scwSettings *settings, size_t mapIndex)
{
	char *sendmailCommand;
	const char *spoolFile;
	int spoolDescriptor;
	char *messageFile = NULL;
	int messageDescriptor = -1;
	struct stat statBuf;
	bool sendAsAttachment = false;
	char mimeBoundary[128];		 /* flawfinder: ignore */
	char transferBuffer[16384];	 /* flawfinder: ignore */
	pid_t pid;

	/*
	 * flawfinder rationale - mimeBoundary is zeroed before use and
	 * snprintf() is bounded by sizeof(); transferBuffer is bounded by
	 * sizeof().
	 */

	sendmailCommand = settings->sendmail.expandedValue;
	if (NULL == sendmailCommand)
		return;

	spoolFile = state->output[mapIndex].temporarySpool;
	if (NULL == spoolFile)
		return;

	memset(mimeBoundary, 0, sizeof(mimeBoundary));

	debug("%s -> %s: [%s]", spoolFile, "email", sendmailCommand);

	spoolDescriptor = open(spoolFile, O_RDONLY	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
			       | O_NOFOLLOW
#endif
	    );
	if (spoolDescriptor < 0) {
		perror(PACKAGE_NAME);
		return;
	}

	/* Read the size of the spool file. */
	if (0 != fstat(spoolDescriptor, &statBuf)) {
		perror(PACKAGE_NAME);
		(void) close(spoolDescriptor);
		return;
	}

	/* Check whether to send as an attachment. */
	if ((settings->numEmailMaxBodySize != (unsigned int) -1)	/* not unlimited */
	    &&((size_t) (statBuf.st_size) > (size_t) (settings->numEmailMaxBodySize))	/* over the threshold */
	    ) {
		sendAsAttachment = true;
	}

	if (!tempFileCreate(state, &messageFile, &messageDescriptor)) {
		(void) close(spoolDescriptor);
		return;
	}
	if (NULL == messageFile) {
		(void) close(spoolDescriptor);
		return;
	}

	/* Write the common message headers. */

	/* The "From:" header. */
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("From: "));
	if (NULL != settings->emailSender.expandedValue) {
		/* Use the EmailSender setting if present. */
		(void) fileWriteBuffer(messageDescriptor, settings->emailSender.expandedValue,
				       settings->emailSender.expandedLength);
	} else {
		/* If there's no EmailSender, default to a cron-like sender. */
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\"(Cron Daemon)\" <"));
		if (NULL != state->username)
			(void) fileWriteBuffer(messageDescriptor, state->username, state->usernameLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(">"));
	}
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));

	/* The "To:" header. */
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("To: "));
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(state->output[mapIndex].destination));
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));

	/* The "Subject:" header. */
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Subject: "));
	if (NULL != settings->emailSubject.expandedValue) {
		/* Use the EmailSubject setting if present. */
		(void) fileWriteBuffer(messageDescriptor, settings->emailSubject.expandedValue,
				       settings->emailSubject.expandedLength);
	} else {
		/* If there's no EmailSubject, default to a cron-like subject. */
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Subject: Cron <"));
		if (NULL != state->username)
			(void) fileWriteBuffer(messageDescriptor, state->username, state->usernameLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("@"));
		if (NULL != state->hostname)
			(void) fileWriteBuffer(messageDescriptor, state->hostname, state->hostnameLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("> "));
		if (NULL != settings->command.expandedValue)
			(void) fileWriteBuffer(messageDescriptor, settings->command.expandedValue,
					       settings->command.expandedLength);
	}
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));

	/* Indicate that this is an auto-generated email. */
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Auto-Submitted: auto-generated\n"));

	/* MIME headers if sending as an attachment. */
	if (sendAsAttachment) {
		/* Generate the MIME boundary. */
		(void) snprintf(mimeBoundary, sizeof(mimeBoundary), "--%s-%s-%d-%u", PACKAGE_NAME, "boundary",
				(int) getpid(), randomNumber(0));

		/* Write the MIME headers. */
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("MIME-Version: 1.0\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Content-Type: multipart/mixed; boundary=\""));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(mimeBoundary));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\"\n"));
	}

	/* Blank line to end the headers. */
	(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));

	/* MIME components if sending as an attachment. */
	if (sendAsAttachment) {
		(void) fileWriteBuffer(messageDescriptor,
				       STATIC_STRING("This is a multi-part message in MIME format.\n\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("--"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(mimeBoundary));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));
		(void) fileWriteBuffer(messageDescriptor,
				       STATIC_STRING("Content-Type: text/plain; charset=\"UTF-8\"\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Content-Transfer-Encoding: 8bit\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));
		if (NULL != settings->emailBodyText.expandedValue)
			(void) fileWriteBuffer(messageDescriptor, settings->emailBodyText.expandedValue,
					       settings->emailBodyText.expandedLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));

		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("--"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(mimeBoundary));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));
		(void) fileWriteBuffer(messageDescriptor,
				       STATIC_STRING("Content-Type: text/plain; charset=\"UTF-8\"; name=\""));
		if (NULL != settings->emailAttachmentName.expandedValue)
			(void) fileWriteBuffer(messageDescriptor, settings->emailAttachmentName.expandedValue,
					       settings->emailAttachmentName.expandedLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\"\n"));
		(void) fileWriteBuffer(messageDescriptor,
				       STATIC_STRING("Content-Disposition: attachment; filename=\""));
		if (NULL != settings->emailAttachmentName.expandedValue)
			(void) fileWriteBuffer(messageDescriptor, settings->emailAttachmentName.expandedValue,
					       settings->emailAttachmentName.expandedLength);
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\"\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("Content-Transfer-Encoding: 8bit\n"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n"));
	}

	/* Copy the spooled output to the message file. */
	while (true) {
		ssize_t bytesRead, bytesWritten, writeOffset;

		bytesRead = read(spoolDescriptor, transferBuffer, sizeof(transferBuffer));	/* flawfinder: ignore */
		/*
		 * flawfinder warns of boundaries when read() is in a loop,
		 * but here we always read into the same point in the buffer
		 * and never larger than its size.
		 */
		if (bytesRead < 0) {
			if (errno == EINTR)
				continue;
			perror(PACKAGE_NAME);
			break;
		} else if (0 == bytesRead) {
			break;
		}

		writeOffset = 0;
		while (writeOffset < bytesRead) {
			bytesWritten =
			    write(messageDescriptor, transferBuffer + writeOffset, (size_t) (bytesRead - writeOffset));
			if (bytesWritten < 0) {
				if (errno == EINTR)
					continue;
				perror(PACKAGE_NAME);
				break;
			} else if (0 == bytesWritten) {
				break;
			}
			writeOffset += bytesWritten;
		}
	}

	(void) close(spoolDescriptor);

	/* Final MIME components if sending as an attachment. */
	if (sendAsAttachment) {
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("\n--"));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING(mimeBoundary));
		(void) fileWriteBuffer(messageDescriptor, STATIC_STRING("--\n"));
	}

	if (0 != close(messageDescriptor)) {
		perror(PACKAGE_NAME);
		(void) remove(messageFile);
		return;
	}

	/*
	 * Fork, run SendmailCommand with "sh -c" in the child process with
	 * its stdin connected to the temporary message file, its stdout and
	 * stderr connected to /dev/null, and all other file descriptors
	 * closed, and in the parent, wait for exit.
	 */
	pid = (pid_t) fork();
	if (pid < 0) {

		/* Fork failure. */
		perror(PACKAGE_NAME);
		return;

	} else if (pid == 0) {

		/* Child process. */
		int nullDescriptor;

		/* Replace stdin with the message file. */
		messageDescriptor = open(messageFile, O_RDONLY	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
					 | O_NOFOLLOW
#endif
		    );
		if ((messageDescriptor < 0) || (dup2(messageDescriptor, STDIN_FILENO) < 0)) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, messageFile, strerror(errno));
			exit(EXIT_FAILURE);
		}
		(void) close(messageDescriptor);

		/* Replace stdout/err with /dev/null. */
		nullDescriptor = open("/dev/null", O_APPEND	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
				      | O_NOFOLLOW
#endif
				      , 0600);

		/*
		 * flawfinder warns of symlinks, redirections, etc when
		 * using open(); this is an internally generated temporary
		 * path (spoolFile) or a fixed path (/dev/null), without
		 * O_CREAT or O_TRUNC, and we use O_NOFOLLOW where
		 * available, so the risk is negligible.
		 */
		if (nullDescriptor < 0) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "/dev/null", strerror(errno));
		} else if ((dup2(nullDescriptor, STDOUT_FILENO) < 0)
			   || (dup2(nullDescriptor, STDERR_FILENO) < 0)) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, "/dev/null", strerror(errno));
		} else {
			(void) close(nullDescriptor);
			debugWriteOutput("", "", -1, "");	/* close debug stream. */
			(void) execl("/bin/sh", "sh", "-c", sendmailCommand, NULL);	/* flawfinder: ignore */
			/*
			 * flawfinder warns that execl() is difficult to use
			 * safely.  There's no way for us to make it safer
			 * to run a user-supplied command here.
			 */
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, sendmailCommand, strerror(errno));
		}
		exit(EXIT_FAILURE);

	} else {

		/* Parent process. */
		while (true) {
			pid_t waited;
			int status;

			status = 0;
			/*@-type@ */
			/* splint disagreement about __pid_t vs pid_t. */
			waited = waitpid(pid, &status, 0);
			/*@+type@ */
			if (waited < 0 && errno == EINTR) {
				perror(PACKAGE_NAME);
				continue;
			} else if (waited < 0 && errno != EINTR) {
				perror(PACKAGE_NAME);
				(void) remove(messageFile);
				return;
			}

			debug("%s=%d, %s=%d", "waited", (int) pid, "status", status);
			(void) remove(messageFile);
			return;
		}
	}
}


/*
 * Send the contents of the temporary spool to a URL.
 */
static void sendSpoolToUrl(struct scwSettings *settings, const char *spoolFile, const char *url, scwOutputFormat format)
{
	pid_t pid;

	debug("%s -> %s: [%d]", spoolFile, url, format);

	/*
	 * Fork, run the transmission command in the child process with its
	 * standard input and output connected to /dev/null, and in the
	 * parent, wait for exit.
	 */

	pid = (pid_t) fork();
	if (pid < 0) {

		/* Fork failure. */
		perror(PACKAGE_NAME);
		return;

	} else if (pid == 0) {

		/* Child process. */
		int nullDescriptor;

		/* Replace stdin/out/err with /dev/null. */
		nullDescriptor = open("/dev/null", O_RDWR	/* flawfinder: ignore */
#ifdef O_NOFOLLOW
				      | O_NOFOLLOW
#endif
				      , 0600);
		/*
		 * flawfinder warns of symlinks, redirections, etc when
		 * using open(); this is a fixed path, without O_CREAT or
		 * O_TRUNC, and we use O_NOFOLLOW where available, so the
		 * risk is negligible.
		 */
		if (nullDescriptor < 0) {
			perror(PACKAGE_NAME);
		} else if ((dup2(nullDescriptor, STDIN_FILENO) < 0) || (dup2(nullDescriptor, STDOUT_FILENO) < 0)) {
			perror(PACKAGE_NAME);
		} else {
			char numberBuf[16];	/* flawfinder: ignore */
			struct scwSettingValue *commandSetting = NULL;
			const char *commandString = NULL;

			/*
			 * flawfinder - buffer is zeroed and bounded;
			 * spoolFile is null-terminated.
			 */
			memset(numberBuf, 0, sizeof(numberBuf));
			(void) snprintf(numberBuf, sizeof(numberBuf), "%u",
					settings->numHTTPTimeout > 0 ? settings->numHTTPTimeout : 1);

			(void) close(nullDescriptor);
			debugWriteOutput("", "", -1, "");	/* close debug stream. */

			/*@-unrecog@ *//* splint doesn't know of setenv() or unsetenv(). */
			(void) unsetenv("SCW_TIMEOUT");
			(void) unsetenv("SCW_FILE");
			(void) unsetenv("SCW_URL");
			(void) setenv("SCW_TIMEOUT", numberBuf, 1);
			(void) setenv("SCW_FILE", spoolFile, 1);
			(void) setenv("SCW_URL", url, 1);
			/*@+unrecog@ */

			if (SCW_FORMAT_JSON == format) {
				commandSetting = &(settings->transmitJson);
			} else {
				commandSetting = &(settings->transmitForm);
			}

			if (NULL != commandSetting) {
				commandString = commandSetting->expandedValue;
				if (NULL == commandString)
					commandString = commandSetting->rawValue;
			}

			if (NULL != commandString) {
				(void) execl("/bin/sh", "sh", "-c", commandString, NULL);	/* flawfinder: ignore */
			} else {
				/*@-mustfreefresh@ */
				/* gettext triggers splint warnings we can't resolve. */
				fprintf(stderr, "%s: %s\n", PACKAGE_NAME, _("no transmission command defined"));
				/*@+mustfreefresh@ */
			}

			/*
			 * flawfinder warns that execl() is difficult to use
			 * safely.  There's no way for us to make it safer
			 * to run a user-supplied command here.
			 */
		}
		exit(EXIT_FAILURE);

	} else {

		/* Parent process. */
		while (true) {
			pid_t waited;
			int status;

			status = 0;
			/*@-type@ */
			/* splint disagreement about __pid_t vs pid_t. */
			waited = waitpid(pid, &status, 0);
			/*@+type@ */
			if (waited < 0 && errno == EINTR) {
				/* unblocked signal or SIGCHLD - ignore. */
				/* perror(PACKAGE_NAME); */
				continue;
			} else if (waited < 0 && errno != EINTR) {
				perror(PACKAGE_NAME);
				return;
			}

			debug("%s=%d, %s=%d", "waited", (int) pid, "status", status);

			/*@-predboolint@ */
			/* splint sees WIF...() as non-boolean. */
			if (WIFEXITED(status)) {
				int exitStatus = WEXITSTATUS(status);
				debug("%s: %d", "transmitter exit status", exitStatus);
				exitStatus = exitStatus;	/* dummy for splint */
				break;
			} else if (WIFSIGNALED(status)) {
				debug("%s", "transmitter terminated by signal");
				break;
			}
			/*@+predboolint@ */

			debug("%s: %d", "unexpected parent state, returning false - status value", status);
			break;
		}
	}
}


/*
 * Send all spooled / deferred output ("!" or email) as per the OutputMaps;
 * this is called after all the command has exited and all of the outputs
 * have been closed.
 */
void sendSpooledOutput(struct scwState *state, struct scwSettings *settings, pid_t pid, bool commandFailed)
{
	size_t mapIndex;

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {
		struct stat statBuf;

		if (NULL == state->output[mapIndex].temporarySpool)
			continue;

		/* If it's send-on-failure, don't send if the command succeeded. */
		if (state->output[mapIndex].sendOnFailure && !commandFailed)
			continue;

		debug("%s[%d]: %s=%s", "outputMap", mapIndex, "anythingWritten",
		      state->output[mapIndex].anythingWritten ? "true" : "false");

		/* Ignore the spool if it contains nothing. */
		if (!state->output[mapIndex].anythingWritten)
			continue;
		memset(&statBuf, 0, sizeof(statBuf));
		if (0 != fileStatAt("", 0, STATIC_STRING(state->output[mapIndex].temporarySpool), &statBuf))
			continue;
		if (0 == statBuf.st_size)
			continue;

		switch (state->output[mapIndex].destinationType) {
		case SCW_DESTINATION_FILE:
			if (NULL != state->output[mapIndex].destination)
				copySpoolToFile(state->output[mapIndex].temporarySpool,
						state->output[mapIndex].destination);
			break;

		case SCW_DESTINATION_SYSLOG:
			copySpoolToSyslog(state, state->output[mapIndex].temporarySpool, pid,
					  state->output[mapIndex].syslogPriority);
			break;

		case SCW_DESTINATION_EMAIL:
			if (NULL != state->output[mapIndex].destination && NULL != settings->sendmail.expandedValue)
				sendSpoolByEmail(state, settings, mapIndex);
			break;

		case SCW_DESTINATION_URL:
			if (NULL != state->output[mapIndex].destination)
				sendSpoolToUrl(settings, state->output[mapIndex].temporarySpool,
					       state->output[mapIndex].destination, state->output[mapIndex].format);
			break;
		}
	}
}


/*
 * For any outputs which are to be sent to a URL, for which any data has
 * been written, and whose spoolUntil time has been reached, transmit the
 * spool, and empty it.
 */
void transmitDelayedUrlSpools(struct scwState *state, struct scwSettings *settings)
{
	size_t mapIndex;
	time_t now;

	now = time(NULL);

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {

		/* This applies to URLs only. */
		if (SCW_DESTINATION_URL != state->output[mapIndex].destinationType)
			continue;

		/* Nothing to do if we're only sending at the end, on failure. */
		if (state->output[mapIndex].sendOnFailure)
			continue;

		/* Nothing to do if there is no spool. */
		if (NULL == state->output[mapIndex].temporarySpool)
			continue;

		/* Nothing to do if nothing has been written. */
		if (!state->output[mapIndex].anythingWritten)
			continue;

		/* Nothing to do if we're supposed to wait some more. */
		if (now < state->output[mapIndex].spoolUntil)
			continue;

		/* Lock the file. */
		(void) flock(state->output[mapIndex].writeDescriptor, LOCK_EX);

		/* Transmit the spool. */
		sendSpoolToUrl(settings, state->output[mapIndex].temporarySpool,
			       state->output[mapIndex].destination, state->output[mapIndex].format);

		/* Truncate the spool to start it again. */
		if (-1 == lseek(state->output[mapIndex].writeDescriptor, 0, SEEK_SET))
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
				state->output[mapIndex].temporarySpool, strerror(errno));
		if (0 != ftruncate(state->output[mapIndex].writeDescriptor, 0))
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
				state->output[mapIndex].temporarySpool, strerror(errno));
		state->output[mapIndex].anythingWritten = false;

		/* Unlock the file. */
		(void) flock(state->output[mapIndex].writeDescriptor, LOCK_UN);

		/* Adjust the time until the next transmission. */
		state->output[mapIndex].spoolUntil = now + settings->numHTTPInterval;
	}
}


/*
 * Remove all temporary files for spooled output.
 */
void removeSpooledOutput(struct scwState *state)
{
	size_t mapIndex;

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {
		if (NULL == state->output[mapIndex].temporarySpool)
			continue;
		if (0 != remove(state->output[mapIndex].temporarySpool)) {
			fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, state->output[mapIndex].temporarySpool,
				strerror(errno));
		}
		state->output[mapIndex].temporarySpool = NULL;
	}
}


/*
 * Return a string containing the message formatted according to the
 * intended output type.  The caller should free() the string.
 */
/*@null@ */
static char *formatMessage(struct scwState *state, struct scwSettings *settings, struct scwOutput *output, pid_t pid,
			   scwOutputStreamId whichStream, const char *message, size_t messageLength, time_t epochTime,
			   const char *timestamp, size_t timestampLength, size_t *formattedLengthPointer)
{
	char *formattedString = NULL;
	size_t formattedBufferSize = 0;
	size_t formattedLength = 0;
	size_t position = 0;
	size_t escapedLength = 0;
	int bytesWritten;
	struct Sha_256 hashState;
	uint8_t hashResult[32];
	char hashHexString[65];		 /* flawfinder: ignore */

	/*
	 * flawfinder rationale: needs room for 64 hex digits and a null
	 * terminator.  Explicitly zeroed initially to ensure termination.
	 */

	memset(&hashState, 0, sizeof(hashState));
	memset(&hashResult, 0, sizeof(hashResult));
	memset(&hashHexString, 0, sizeof(hashHexString));

	/* Generate the hash, if one is needed. */
	if (settings->sharedSecret.rawLength > 0
	    && (SCW_FORMAT_JSON == output->format || SCW_FORMAT_FORM == output->format)) {
		char intAsString[24];	 /* flawfinder: ignore */
		size_t offset;

		/*
		 * flawfinder rationale: buffer is large enough for any
		 * integer <2^63 plus a null terminator, and is explicitly
		 * zeroed before each use to ensure termination.
		 */

		sha_256_init(&hashState, hashResult);

		/* epoch */
		memset(intAsString, 0, sizeof(intAsString));
		(void) snprintf(intAsString, sizeof(intAsString), "%lu", (unsigned long) epochTime);
		sha_256_write(&hashState, intAsString, strlen(intAsString));	/* flawfinder: ignore */
		debug("%s: %s", "add to hash", intAsString);

		/* pid */
		memset(intAsString, 0, sizeof(intAsString));
		(void) snprintf(intAsString, sizeof(intAsString), "%lu", (unsigned long) pid);
		sha_256_write(&hashState, intAsString, strlen(intAsString));	/* flawfinder: ignore */
		debug("%s: %s", "add to hash", intAsString);

		/*
		 * flawfinder - the intAsString buffer is always going to be
		 * null-terminated, so strlen() can't overrun.
		 */

		/* hostname */
		if (state->hostnameLength > 0 && NULL != state->hostname) {
			sha_256_write(&hashState, state->hostname, state->hostnameLength);
			debug("%s: %.*s", "add to hash", state->hostnameLength, state->hostname);
		}

		/* user */
		if (state->usernameLength > 0 && NULL != state->username) {
			sha_256_write(&hashState, state->username, state->usernameLength);
			debug("%s: %.*s", "add to hash", state->usernameLength, state->username);
		}

		/* item */
		if (state->itemLength > 0 && NULL != state->item) {
			sha_256_write(&hashState, state->item, state->itemLength);
			debug("%s: %.*s", "add to hash", state->itemLength, state->item);
		}

		/* stream */
		switch (whichStream) {
		case SCW_STREAM_STDOUT:
			sha_256_write(&hashState, STATIC_STRING("stdout"));
			debug("%s: %s", "add to hash", "stdout");
			break;
		case SCW_STREAM_STDERR:
			sha_256_write(&hashState, STATIC_STRING("stderr"));
			debug("%s: %s", "add to hash", "stderr");
			break;
		case SCW_STREAM_STATUS:
			sha_256_write(&hashState, STATIC_STRING("status"));
			debug("%s: %s", "add to hash", "status");
			break;
		}

		/* message */
		if (messageLength > 0) {
			sha_256_write(&hashState, message, messageLength);
			debug("%s: %.*s", "add to hash", messageLength, message);
		}

		/* shared secret */
		if (settings->sharedSecret.rawLength > 0 && NULL != settings->sharedSecret.rawValue) {
			sha_256_write(&hashState, settings->sharedSecret.rawValue, settings->sharedSecret.rawLength);
			debug("%s: %.*s", "add to hash", settings->sharedSecret.rawLength,
			      settings->sharedSecret.rawValue);
		}

		(void) sha_256_close(&hashState);

		/* Express hash as a hex string */
		for (offset = 0; offset < 32; offset++) {
			(void) snprintf(hashHexString + 2 * offset, 3, "%02x", (unsigned int) (hashResult[offset]));
		}

		debug("%s: %s", "hash value", hashHexString);
	}

	switch (output->format) {
	case SCW_FORMAT_RAW:
	case SCW_FORMAT_STAMPED:
		/* Timestamp + space + message. */
		formattedBufferSize = timestampLength + 1 + messageLength;

		/* Multi streams, so also need " [x]" (x=stream) */
		if (output->streamsIncluded != whichStream)
			formattedBufferSize += 4;

		/* No timestamp in raw format. */
		if (SCW_FORMAT_RAW == output->format)
			formattedBufferSize = messageLength;

		/*
		 * If this is a status line but its first word isn't
		 * surrounded by brackets, we need space for the brackets,
		 * since we will add them - even in raw mode.
		 */
		if (SCW_STREAM_STATUS == whichStream && messageLength > 0 && message[0] != '(')
			formattedBufferSize += 2;

		/* Add one byte for the null terminator. */
		formattedBufferSize++;

		/* Allocate the buffer. */
		formattedString = malloc(formattedBufferSize);
		if (NULL == formattedString) {
			perror(PACKAGE_NAME);
			return NULL;
		}
		memset(formattedString, 0, formattedBufferSize);

		position = 0;

		if (SCW_FORMAT_STAMPED == output->format) {
			memcpy(formattedString, timestamp, timestampLength);	/* flawfinder: ignore */
			/* flawfinder - buffer is sized appropriately. */
			position += timestampLength;
			formattedString[position++] = ' ';

			if (output->streamsIncluded != whichStream) {
				formattedString[position++] = '[';
				switch (whichStream) {
				case SCW_STREAM_STDOUT:
					formattedString[position++] = '-';
					break;
				case SCW_STREAM_STDERR:
					formattedString[position++] = 'E';
					break;
				case SCW_STREAM_STATUS:
					formattedString[position++] = 's';
					break;
				}
				formattedString[position++] = ']';
				formattedString[position++] = ' ';
			}
		}

		/* Ensure that a status line's first word is bracketed (). */
		if (SCW_STREAM_STATUS == whichStream && messageLength > 0 && message[0] != '(') {
			formattedString[position++] = '(';
			while (messageLength > 0 && !isspace(message[0])) {
				formattedString[position++] = message[0];
				message++;
				messageLength--;
			}
			formattedString[position++] = ')';
		}

		memmove(formattedString + position, message, messageLength);
		formattedString[position + messageLength] = '\0';

		formattedLength = position + messageLength;
		break;

	case SCW_FORMAT_JSON:
		/*
		 * JSON worst-case string size is 6 output characters per
		 * input character ("\uXXXX"), plus start and end quotes.
		 * Each item also needs a ":" between key and value, and a
		 * "," after the value.  Assume maximum 15 digits for
		 * integers.  Allocate enough for the worst-case.
		 */
		formattedBufferSize = 2;    /* surrounding "{ }". */
		formattedBufferSize += 3 + STATIC_STRLEN("epoch") + 15;
		formattedBufferSize += 3 + STATIC_STRLEN("pid") + 15;
		formattedBufferSize += 3 + STATIC_STRLEN("line") + 15;
		formattedBufferSize += 5 + STATIC_STRLEN("hostname") + 6 * state->hostnameLength;
		formattedBufferSize += 5 + STATIC_STRLEN("user") + 6 * state->usernameLength;
		formattedBufferSize += 5 + STATIC_STRLEN("item") + 6 * state->itemLength;
		formattedBufferSize += 5 + STATIC_STRLEN("stream") + STATIC_STRLEN("status");	/* not escaped */
		formattedBufferSize += 5 + STATIC_STRLEN("message") + 6 * messageLength;
		if (settings->sharedSecret.rawLength > 0)
			formattedBufferSize += 5 + STATIC_STRLEN("hash") + 6 * 64;
		/*
		 * flawfinder - calling strlen() on a static string is OK as
		 * it's always null-terminated.
		 */

		/* Add one byte for the null terminator. */
		formattedBufferSize++;

		/* Allocate the buffer. */
		formattedString = malloc(formattedBufferSize);
		if (NULL == formattedString) {
			perror(PACKAGE_NAME);
			return NULL;
		}
		memset(formattedString, 0, formattedBufferSize);

		/* Construct the string. */
		bytesWritten =
		    snprintf(formattedString, formattedBufferSize, "{\"epoch\":%lu,\"pid\":%lu,\"line\":%u",
			     (unsigned long) epochTime, (unsigned long) pid, output->lineNumber);
		position = 0;
		if (bytesWritten > 0) {
			position = (size_t) bytesWritten;
		} else {
			perror(PACKAGE_NAME);
		}

		position =
		    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING(",\"hostname\":\""));
		(void) jsonEscapeString(formattedString + position, formattedBufferSize - position, state->hostname,
					state->hostnameLength, &escapedLength);
		position += escapedLength;

		position =
		    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("\",\"user\":\""));
		(void) jsonEscapeString(formattedString + position, formattedBufferSize - position, state->username,
					state->usernameLength, &escapedLength);
		position += escapedLength;

		position =
		    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("\",\"item\":\""));
		(void) jsonEscapeString(formattedString + position, formattedBufferSize - position, state->item,
					state->itemLength, &escapedLength);
		position += escapedLength;

		position =
		    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("\",\"stream\":\""));
		switch (whichStream) {
		case SCW_STREAM_STDOUT:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("stdout"));
			break;
		case SCW_STREAM_STDERR:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("stderr"));
			break;
		case SCW_STREAM_STATUS:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("status"));
			break;
		}

		position =
		    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("\",\"message\":\""));
		(void) jsonEscapeString(formattedString + position, formattedBufferSize - position, message,
					messageLength, &escapedLength);
		position += escapedLength;

		if (settings->sharedSecret.rawLength > 0) {
			position =
			    stringAppend(formattedBufferSize, formattedString, position,
					 STATIC_STRING("\",\"hash\":\""));
			(void) jsonEscapeString(formattedString + position, formattedBufferSize - position,
						hashHexString, 64, &escapedLength);
			position += escapedLength;
		}

		position = stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("\"}"));

		formattedLength = position;

		break;

	case SCW_FORMAT_FORM:
		/*
		 * URL-encoding worst-case string size is 3 output
		 * characters per input character ("%XX").  Each item also
		 * needs a "=" between key and value, and a "&" between
		 * key=value pairs.  Also, each key is suffixed with an
		 * integer.  Assume maximum 15 digits for integers.
		 * Allocate enough for the worst-case.
		 */
		formattedBufferSize = 0;
		formattedBufferSize += 1 + STATIC_STRLEN("epoch") + 15 + 15;
		formattedBufferSize += 2 + STATIC_STRLEN("pid") + 15 + 15;
		formattedBufferSize += 2 + STATIC_STRLEN("hostname") + 15 + 3 * state->hostnameLength;
		formattedBufferSize += 2 + STATIC_STRLEN("user") + 15 + 3 * state->usernameLength;
		formattedBufferSize += 2 + STATIC_STRLEN("item") + 15 + 3 * state->itemLength;
		formattedBufferSize += 2 + STATIC_STRLEN("stream") + 15 + STATIC_STRLEN("status");	/* not escaped */
		formattedBufferSize += 2 + STATIC_STRLEN("message") + 15 + 3 * messageLength;
		if (settings->sharedSecret.rawLength > 0)
			formattedBufferSize += 2 + STATIC_STRLEN("hash") + 15 + 3 * 64;
		/*
		 * flawfinder - calling strlen() on a static string is OK as
		 * it's always null-terminated.
		 */

		/* Add one byte for the null terminator. */
		formattedBufferSize++;

		/* Allocate the buffer. */
		formattedString = malloc(formattedBufferSize);
		if (NULL == formattedString) {
			perror(PACKAGE_NAME);
			return NULL;
		}
		memset(formattedString, 0, formattedBufferSize);

		/* Construct the string. */
		bytesWritten =
		    snprintf(formattedString, formattedBufferSize, "epoch%u=%lu&pid%u=%lu", output->lineNumber,
			     (unsigned long) epochTime, output->lineNumber, (unsigned long) pid);
		position = 0;
		if (bytesWritten > 0) {
			position = (size_t) bytesWritten;
		} else {
			perror(PACKAGE_NAME);
		}
#define appendKey(x) \
		{ \
			bytesWritten = snprintf(formattedString + position, formattedBufferSize - position, x, output->lineNumber); /* flawfinder: ignore */ \
			if (bytesWritten > 0) { \
				position += (size_t) bytesWritten; \
			} else { \
				perror(PACKAGE_NAME); \
			} \
		}
		/*
		 * flawfinder warns of snprintf() format strings being
		 * influenced by an attacker, but we pass it static format
		 * strings when calling this macro, so this is a false
		 * positive.
		 */
		appendKey("&hostname%u=");
		(void) urlEncodeString(formattedString + position, formattedBufferSize - position, state->hostname,
				       state->hostnameLength, &escapedLength);
		position += escapedLength;

		appendKey("&user%u=");
		(void) urlEncodeString(formattedString + position, formattedBufferSize - position, state->username,
				       state->usernameLength, &escapedLength);
		position += escapedLength;

		appendKey("&item%u=");
		(void) urlEncodeString(formattedString + position, formattedBufferSize - position, state->item,
				       state->itemLength, &escapedLength);
		position += escapedLength;

		appendKey("&stream%u=");
		switch (whichStream) {
		case SCW_STREAM_STDOUT:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("stdout"));
			break;
		case SCW_STREAM_STDERR:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("stderr"));
			break;
		case SCW_STREAM_STATUS:
			position =
			    stringAppend(formattedBufferSize, formattedString, position, STATIC_STRING("status"));
			break;
		}

		appendKey("&message%u=");
		(void) urlEncodeString(formattedString + position, formattedBufferSize - position, message,
				       messageLength, &escapedLength);
		position += escapedLength;

		if (settings->sharedSecret.rawLength > 0) {
			appendKey("&hash%u=");
			(void) urlEncodeString(formattedString + position, formattedBufferSize - position,
					       hashHexString, 64, &escapedLength);
			position += escapedLength;
		}

		formattedLength = position;

		break;
	}

	*formattedLengthPointer = formattedLength;
	return formattedString;
}


/*
 * Create a per-euid, per-tty, lockfile in ${TMPDIR:-${TMP:-/tmp}} for the
 * terminal attached to stdout, and return an open file descriptor for the
 * lock.
 */
static int terminalLockFile(void)
{
	char lockFile[1024];		 /* flawfinder: ignore */
	char *terminalDevice;
	char *terminalName;
	char *tmpDir;
	int openFlags;
	int fd = -1;

	/* flawfinder - the buffer is used with boundaries, and zeroed. */

	terminalDevice = ttyname(STDOUT_FILENO);
	if (NULL == terminalDevice) {
		debug("%s: %s", "failed to get terminal name", strerror(errno));
		return -1;
	}
	terminalName = strrchr(terminalDevice, '/');
	if (NULL == terminalName) {
		terminalName = terminalDevice;
	} else {
		terminalName = 1 + terminalDevice;
	}

	tmpDir = (char *) getenv("TMPDIR"); /* flawfinder: ignore */
	if ((NULL == tmpDir) || ('\0' == tmpDir[0]))
		tmpDir = (char *) getenv("TMP");	/* flawfinder: ignore */
	if ((NULL == tmpDir) || ('\0' == tmpDir[0]))
		tmpDir = "/tmp";

	/*
	 * flawfinder rationale: null and zero-size values of $TMPDIR and
	 * $TMP are rejected, and the destination buffer is bounded.
	 */

	memset(lockFile, 0, sizeof(lockFile));
	(void) snprintf(lockFile, sizeof(lockFile), "%s/scw-%s-%i.lock", tmpDir, terminalName, (int) geteuid());

#ifdef O_NOFOLLOW
	openFlags = O_RDWR | O_CREAT | O_NOFOLLOW;
#else
	openFlags = O_RDWR | O_CREAT;
#endif

	fd = open(lockFile, openFlags, 0600);	/* flawfinder: ignore */

	/*
	 * flawfinder rationale: we aren't truncating the lock file, we
	 * don't change its contents, and we are attempting to use
	 * O_NOFOLLOW where possible to avoid symlink attacks, so this
	 * open() is as safe as we can make it.
	 */

	if (fd < 0) {
		debug("%s: %s: %s", "failed to open lock file", lockFile, strerror(errno));
	}

	return fd;
}


/*
 * Acquire a file lock representing the terminal, returning the associated
 * file descriptor.
 *
 * We use fcntl() because flock(LOCK_EX) on a terminal doesn't seem to work.
 */
static int lockTerminal(void)
{
	struct flock lock;
	int fd = STDOUT_FILENO;

	memset(&lock, 0, sizeof(lock));
	lock.l_type = (short) F_WRLCK;
	lock.l_whence = (short) SEEK_SET;
	lock.l_start = 0;
	lock.l_len = 1;
	while (fd >= 0 && fcntl(fd, F_SETLKW, &lock) < 0) {
		if (errno != EINTR) {
			if (STDOUT_FILENO == fd) {
				/* Fall back to a lock file. */
				debug("%s", "using a terminal lock file");
				fd = terminalLockFile();
			} else {
				debug("%s: %s", "lock attempt failed", strerror(errno));
				return -1;
			}
		}
	}

	debug("%s: fd=%d", "terminal locked", fd);
	return fd;
}


/*
 * Unlock the terminal lock acquired by lockTerminal(), and close the file
 * descriptor if appropriate.
 */
static void unlockTerminal(int fd)
{
	struct flock lock;

	memset(&lock, 0, sizeof(lock));
	lock.l_type = (short) F_UNLCK;
	lock.l_whence = (short) SEEK_SET;
	lock.l_start = 0;
	lock.l_len = 1;
	(void) fcntl(fd, F_SETLK, &lock);

	if (STDOUT_FILENO != fd)
		(void) close(fd);

	debug("%s", "terminal lock released");
}


/*
 * Record a line of item output to the appropriate places according to the
 * OutputMaps, and possibly also to the terminal.
 */
void recordOutput(struct scwState *state, struct scwSettings *settings, pid_t pid, scwOutputStreamId whichStream,
		  const char *message, size_t messageLength)
{
	char timestampBuffer[128];	 /* flawfinder: ignore */
	size_t timestampLength;
	time_t epochTime;
	size_t mapIndex;

	/*
	 * flawfinder notes: timestampBuffer is only written to by
	 * strftime(), which takes a buffer size, and we enforce string
	 * termination.
	 */

	/*
	 * First, if StatusMode is not "fd" (state->streamForStatus is not
	 * SCW_STREAM_STATUS) - and whichStream matches the stream we're
	 * reading status from, look for the StatusTag.  If it matches, we
	 * call ourselves with the status message, to simulate reading that
	 * message from a virtual status stream.
	 */
	if (whichStream != SCW_STREAM_STATUS && whichStream == state->streamForStatus) {
		if (NULL == settings->statusTag.expandedValue) {
			/* No StatusTag, so the whole message is status. */
			recordOutput(state, settings, pid, SCW_STREAM_STATUS, message, messageLength);
		} else if (settings->statusTag.expandedLength < messageLength
			   && 0 == strncmp(message, settings->statusTag.expandedValue,
					   settings->statusTag.expandedLength)) {
			size_t statusOffset = settings->statusTag.expandedLength;
			/* Skip spaces after the status tag. */
			while (statusOffset < messageLength && isspace(message[statusOffset]))
				statusOffset++;
			recordOutput(state, settings, pid, SCW_STREAM_STATUS, message + statusOffset,
				     messageLength - statusOffset);
		}
	}

	/* Construct the timestamp, in case it is needed. */
	epochTime = time(NULL);
	timestampBuffer[0] = '\0';
	if (state->timestampUTC) {
		struct tm *brokenDownTime = gmtime(&epochTime);
		if (0 == strftime(timestampBuffer, sizeof(timestampBuffer), "%Y-%m-%dT%H:%M:%SZ", brokenDownTime)) {
			timestampBuffer[0] = '\0';
		}
	} else {
		struct tm *brokenDownTime = localtime(&epochTime);
		if (0 == strftime(timestampBuffer, sizeof(timestampBuffer), "%Y-%m-%dT%H:%M:%S", brokenDownTime)) {
			timestampBuffer[0] = '\0';
		}
	}
	timestampBuffer[sizeof(timestampBuffer) - 1] = '\0';	/* enforce termination */
	timestampLength = strlen(timestampBuffer);	/* flawfinder: ignore */
	/* flawfinder - strlen() has been given a null-terminated string. */

	debug("[%.*s] -> %d", messageLength, message, whichStream);

	for (mapIndex = 0; mapIndex < state->outputCount; mapIndex++) {
		char *outputString = NULL;
		size_t outputLength = 0;

		/* Skip if this output does not include this stream. */
		if (0 == (state->output[mapIndex].streamsIncluded & whichStream))
			continue;

		/* Increment the output line counter. */
		state->output[mapIndex].lineNumber++;

		outputString =
		    formatMessage(state, settings, &(state->output[mapIndex]), pid, whichStream, message, messageLength,
				  epochTime, timestampBuffer, timestampLength, &outputLength);
		if (NULL == outputString)
			continue;

		/*
		 * If output check mode is enabled, check the destination
		 * still has the same inode number, and re-open it if not.
		 */
		if (state->output[mapIndex].outputFileCheck && NULL != state->output[mapIndex].destination
		    && SCW_DESTINATION_FILE == state->output[mapIndex].destinationType
		    && epochTime >= state->output[mapIndex].nextFileCheck) {
			struct stat statBuf;
			int statResult;

			/* Check no more than once per second. */
			state->output[mapIndex].nextFileCheck = 1 + epochTime;

			statResult = stat(state->output[mapIndex].destination, &statBuf);
			if (((0 == statResult)
			     && ((ino_t) (statBuf.st_ino) != state->output[mapIndex].destinationInode)
			    ) || ((statResult != 0) && (ENOENT == errno))
			    ) {
				int newWriteDescriptor;

				/* Attempt to re-open the file. */
				newWriteDescriptor = fileOpenForAppend(state->output[mapIndex].destination);
				if (newWriteDescriptor >= 0) {
					/*
					 * Success - close the old file
					 * descriptor and start using the
					 * new one.
					 */
					state->output[mapIndex].destinationInode = (ino_t) (statBuf.st_ino);
					if (state->output[mapIndex].writeDescriptor >= 0) {
						if (0 != close(state->output[mapIndex].writeDescriptor)) {
							fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
								NULL ==
								state->output[mapIndex].
								temporarySpool ? state->output[mapIndex].
								destination : state->output[mapIndex].temporarySpool,
								strerror(errno));
						}
					}
					state->output[mapIndex].writeDescriptor = newWriteDescriptor;
					debug("%s: %s", "reopened", state->output[mapIndex].destination);
				}
			}
		}

		/* Lock the output so multiple streams don't mix. */
		if (state->output[mapIndex].writeDescriptor >= 0) {
			off_t filePosition;

			(void) flock(state->output[mapIndex].writeDescriptor, LOCK_EX);
			/*
			 * Move to the end of the file in case some other
			 * process appended to it, or truncated it.
			 */
			filePosition = (off_t) lseek(state->output[mapIndex].writeDescriptor, 0, SEEK_END);
			if (filePosition > 2 && SCW_FORMAT_JSON == state->output[mapIndex].format) {
				/* JSON format - move back 2 for "\n]" - see below. */
				(void) lseek(state->output[mapIndex].writeDescriptor, -2, SEEK_CUR);
			}
		}

		switch (state->output[mapIndex].destinationType) {
		case SCW_DESTINATION_EMAIL:
			/*
			 * When sending an email, first spool the output to
			 * a file, then later in sendSpoolByEmail() we will
			 * prepend headers and, if sending as an attachment,
			 * introduce the appropriate MIME formatting.
			 */
			/*@fallthrough@ */
		case SCW_DESTINATION_FILE:
			(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor, outputString, outputLength);
			(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor, "\n", 1);
			state->output[mapIndex].anythingWritten = true;
			break;

		case SCW_DESTINATION_SYSLOG:
			/*@-unrecog@ *//* we hid syslog.h from splint. */
			syslog(state->output[mapIndex].syslogPriority, "%.*s[%d]: %.*s", (int) (state->itemLength),
			       state->item, (int) pid, (int) outputLength, outputString);
			/*@+unrecog@ */
			state->output[mapIndex].anythingWritten = true;
			break;

		case SCW_DESTINATION_URL:
			if (SCW_FORMAT_JSON == state->output[mapIndex].format) {
				/*
				 * With JSON format, the first line is
				 * preceded by "[" to open the array, and
				 * anything other than the first line is
				 * preceded by "," before the newline, to
				 * continue the array.
				 */
				if (!state->output[mapIndex].anythingWritten) {
					(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor,
							       STATIC_STRING("[\n"));
				} else {
					(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor,
							       STATIC_STRING(",\n"));
				}
			} else if (SCW_FORMAT_FORM == state->output[mapIndex].format) {
				/*
				 * With form format, anything but the first
				 * write is preceded by "&".
				 */
				if (state->output[mapIndex].anythingWritten) {
					(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor,
							       STATIC_STRING("&"));
				}
			}

			/* Add the formatted string to the output spool. */
			(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor, outputString, outputLength);
			state->output[mapIndex].anythingWritten = true;

			/*
			 * In JSON format, add a "\n]" to close the array, but
			 * move back two bytes so if we add another line it
			 * overwrites that with ",\n".
			 */
			if (SCW_FORMAT_JSON == state->output[mapIndex].format
			    && NULL != state->output[mapIndex].temporarySpool) {
				(void) fileWriteBuffer(state->output[mapIndex].writeDescriptor, STATIC_STRING("\n]"));
				if (-1 == lseek(state->output[mapIndex].writeDescriptor, -2, SEEK_CUR))
					fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
						state->output[mapIndex].temporarySpool, strerror(errno));
			}

			/*
			 * Note that we don't actually transmit here - the
			 * main run loop calls transmitDelayedUrlSpools()
			 * which does that, according to the spoolUntil time
			 * that's based on the HTTPInterval.
			 */

			break;
		}

		/* Unlock the output again. */
		if (state->output[mapIndex].writeDescriptor >= 0)
			(void) flock(state->output[mapIndex].writeDescriptor, LOCK_UN);

		free(outputString);
		outputString = NULL;
	}

	/* Status updates replace the "last-status" metrics file contents. */
	if (SCW_STREAM_STATUS == whichStream && NULL != settings->metricsDir.expandedValue) {
		(void) fileReplaceContentsAt(settings->metricsDir.expandedValue, settings->metricsDir.expandedLength,
					     STATIC_STRING("last-status"), "%.*s\n", (int) messageLength, message);
	}

	/* Colourised terminal output. */
	if (state->runningOnTerminal) {
		char *outputString = NULL;
		size_t outputLength = 0;
		struct scwOutput terminal;

		memset(&terminal, 0, sizeof(terminal));
		terminal.format = SCW_FORMAT_STAMPED;
		terminal.destinationType = SCW_DESTINATION_FILE;
		terminal.streamsIncluded = SCW_STREAM_STDOUT | SCW_STREAM_STDERR | SCW_STREAM_STATUS;

		outputString =
		    formatMessage(state, settings, &terminal, pid, whichStream, message, messageLength, epochTime,
				  timestampBuffer, timestampLength, &outputLength);
		if (NULL != outputString) {
			bool needColourReset = false;
			int terminalLockDescriptor;

			terminalLockDescriptor = lockTerminal();

			if (SCW_STREAM_STDERR == whichStream) {
				needColourReset = true;
				(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[33m"));
			} else if (SCW_STREAM_STATUS == whichStream) {
				needColourReset = true;
				if (stringStartsWith(message, "(begin)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[35;1m"));
				} else if (stringStartsWith(message, "(notice)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[36;1m"));
				} else if (stringStartsWith(message, "notice")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[36;1m"));
				} else if (stringStartsWith(message, "(ok)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[32;1m"));
				} else if (stringStartsWith(message, "ok")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[32;1m"));
				} else if (stringStartsWith(message, "(warning)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[33;1m"));
				} else if (stringStartsWith(message, "warning")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[33;1m"));
				} else if (stringStartsWith(message, "(error)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[31;1m"));
				} else if (stringStartsWith(message, "error")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[31;1m"));
				} else if (stringStartsWith(message, "(end)")) {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[35;1m"));
				} else {
					(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[1m"));
				}
			}

			(void) fileWriteBuffer(STDOUT_FILENO, outputString, outputLength);
			free(outputString);
			outputString = NULL;

			if (needColourReset)
				(void) fileWriteBuffer(STDOUT_FILENO, STATIC_STRING("\033[m"));

			(void) fileWriteBuffer(STDOUT_FILENO, "\n", 1);

			unlockTerminal(terminalLockDescriptor);
		}
	}
}
