/*
 * Parse command-line options.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif


/*
 * Parse the given command-line arguments into the supplied state, handling
 * the "help" and "version" options internally.
 *
 * Returns nonzero if there is a fatal error.
 *
 * Note that the contents of *argv[] (i.e. the command line parameters)
 * aren't copied anywhere, just the pointers are copied, so make sure the
 * command line data isn't overwritten or argv[1] free()d or whatever.
 */
int parseOptions(struct scwState *state, int argc, char **argv)
{
#ifdef HAVE_GETOPT_LONG
	/*@-nullassign@ */
	/* splint rationale: NULL is allowed for "flags" in long options. */
	struct option optionsLong[] = {
		{ "help", 0, NULL, (int) 'h' },
		{ "version", 0, NULL, (int) 'V' },
		{ "config", 1, NULL, (int) 'c' },
		{ "set", 1, NULL, (int) 's' },
		{ "force", 0, NULL, (int) 'f' },
		{ "strict", 0, NULL, (int) 'S' },
		{ "enabled", 0, NULL, (int) 'e' },
		{ "disabled", 0, NULL, (int) 'd' },
		{ "info", 0, NULL, (int) 'i' },
		{ "all-users", 0, NULL, (int) 'a' },
		{ "all", 0, NULL, (int) 'a' },
		{ "debug", 1, NULL, (int) '!' },
		{ NULL, 0, NULL, 0 }
	};
	/*@+nullassign@ */
	int optionIndex = 0;
#endif				/* HAVE_GETOPT_LONG */
	char *optionsShort = "hVc:s:fSedia!:";
	int optionChar;
	bool tooManyArguments, requiresItem;
	size_t settingsOptionCount;
	bool strictMode = false;
	int retcode;

	/* Allocate an array to hold a copy of the arguments. */
	state->argumentCount = 0;
	state->arguments = calloc((size_t) (argc + 1), sizeof(char *));
	if (NULL == state->arguments) {
		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME,
			_("option structure argv allocation failed"), strerror(errno));
		return SCW_EXIT_ERROR;
		/*@+mustfreefresh@ */
	}

	settingsOptionCount = 0;

	do {
#ifdef HAVE_GETOPT_LONG
		optionChar = getopt_long(argc, argv, optionsShort, optionsLong, &optionIndex);	/* flawfinder: ignore */
#else
		optionChar = getopt(argc, argv, optionsShort);	/* flawfinder: ignore */
#endif
		/*
		 * flawfinder rationale: we have to pass argv to getopt, and
		 * limiting the argument sizes would be impractical and
		 * cumbersome (and likely lead to more bugs); so we have to
		 * trust the system getopt to not have internal buffer
		 * overflows.
		 */

		if (optionChar < 0)
			continue;

		/*
		 * Parse each command line option.
		 */
		switch (optionChar) {
		case 'c':
			state->configFile = optarg;
			break;
		case 's':
			settingsOptionCount++;
			/*@-mustfreefresh@ */
			/* splint note: gettext triggers a warning we can't resolve. */
			retcode = parseSetting(optarg, strlen(optarg), &(state->commandLineSettings), SCW_SOURCE_COMMANDLINE, _("command line settings"), settingsOptionCount);	/* flawfinder: ignore */
			/*@+mustfreefresh@ */
			/*
			 * flawfinder rationale: warns against using
			 * strlen() on strings that may not be
			 * null-terminated, but this is an option argument
			 * and they are always null-terminated.
			 */

			if (0 != retcode)
				return retcode;	/* early return */

			break;
		case 'f':
			state->forceRun = true;
			break;
		case 'S':
			strictMode = true;
			state->continueWithoutCheckLock = false;
			state->continueWithoutItemLock = false;
			state->continueWithoutMetrics = false;
			state->continueWithoutOutputFile = false;
			break;
		case 'e':
			state->listEnabledOnly = true;
			break;
		case 'd':
			state->listDisabledOnly = true;
			break;
		case 'i':
			state->listWithInfo = true;
			break;
		case 'a':
			state->allUsers = true;
			break;
		case 'h':
			state->action = SCW_ACTION_HELP;
			return 0;	    /* early return */
		case 'V':
			state->action = SCW_ACTION_VERSION;
			return 0;	    /* early return */
		case '!':
#ifdef ENABLE_DEBUGGING
			debugSetDestination(optarg);
#else
			/*@-mustfreefresh@ *//* see above */
			fprintf(stderr, "%s\n", _("Debugging is not enabled in this build."));
			return SCW_EXIT_BAD_ARGS;	/* early return */
			/*@+mustfreefresh@ */
#endif				/* ENABLE_DEBUGGING */
			break;
		default:
			/*@-mustfreefresh@ *//* see above */
			/*@-formatconst@ */
#ifdef HAVE_GETOPT_LONG
			fprintf(stderr, _("Try `%s --help' for more information."), PACKAGE_NAME);
#else
			fprintf(stderr, _("Try `%s -h' for more information."), PACKAGE_NAME);
#endif
			/*@+formatconst@ */
			/*
			 * splint note: formatconst is warning about the use
			 * of a non constant (translatable) format string;
			 * this is unavoidable here and the only attack
			 * vector is through the message catalogue.
			 */
			fprintf(stderr, "\n");
			return SCW_EXIT_BAD_ARGS;	/* early return */
			/*@+mustfreefresh@ */
		}

	} while (optionChar != -1);

	/*
	 * Store remaining command-line arguments.
	 */
	while (optind < argc) {
		state->arguments[state->argumentCount++] = argv[optind++];
	}

	if (state->argumentCount < 1) {
		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */
		fprintf(stderr, "%s: %s\n", PACKAGE_NAME, _("no action specified"));
		return SCW_EXIT_BAD_ARGS;   /* early return */
		/*@+mustfreefresh@ */
	}

	/*
	 * Determine which action has been selected.
	 */

	tooManyArguments = false;
	requiresItem = false;

	/*@-unrecog@ */
	/* splint note: splint doesn't recognise strcasecmp. */
	if (0 == strcasecmp(state->arguments[0], "run")) {
		state->action = SCW_ACTION_RUN;
		requiresItem = true;
		if (state->argumentCount > 2)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "enable")) {
		state->action = SCW_ACTION_ENABLE;
		requiresItem = true;
		if (state->argumentCount > 2)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "disable")) {
		state->action = SCW_ACTION_DISABLE;
		requiresItem = true;
		if (state->argumentCount > 2)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "status")) {
		state->action = SCW_ACTION_STATUS;
		requiresItem = true;
		if (state->argumentCount > 2)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "list")) {
		state->action = SCW_ACTION_LIST;
		requiresItem = false;
		if (state->argumentCount > 1)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "update")) {
		state->action = SCW_ACTION_UPDATE;
		requiresItem = false;
		if (state->argumentCount > 1)
			tooManyArguments = true;
	} else if (0 == strcasecmp(state->arguments[0], "faults")) {
		state->action = SCW_ACTION_FAULTS;
		requiresItem = false;
		if (state->argumentCount > 1)
			tooManyArguments = true;
	} else {
		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, state->arguments[0], _("unknown action"));
		return SCW_EXIT_BAD_ARGS;
		/*@+mustfreefresh@ */
	}
	/*@+unrecog@ */

	/*
	 * Check that the options and arguments are appropriate for the
	 * selected action.
	 */

	/*@-mustfreefresh@ */
	/* splint note: gettext triggers a warning we can't resolve. */
	if (tooManyArguments) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, state->arguments[0], _("too many arguments"));
		return SCW_EXIT_BAD_ARGS;
	} else if (requiresItem && state->argumentCount < 2) {
		fprintf(stderr, "%s: %s: %s\n", PACKAGE_NAME, state->arguments[0], _("missing item name"));
		return SCW_EXIT_BAD_ARGS;
	} else if (state->forceRun && state->action != SCW_ACTION_RUN) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--force",
#else
			"-f",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	} else if (strictMode && state->action != SCW_ACTION_RUN) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--strict",
#else
			"-S",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	} else if (state->listEnabledOnly && state->action != SCW_ACTION_LIST) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--enabled",
#else
			"-e",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	} else if (state->listDisabledOnly && state->action != SCW_ACTION_LIST) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--disabled",
#else
			"-e",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	} else if (state->listWithInfo && state->action != SCW_ACTION_LIST) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--info",
#else
			"-i",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	} else if (state->allUsers && state->action != SCW_ACTION_LIST
		   && state->action != SCW_ACTION_UPDATE && state->action != SCW_ACTION_FAULTS) {
		fprintf(stderr, "%s: %s: %s: %s\n", PACKAGE_NAME, state->arguments[0],
#ifdef HAVE_GETOPT_LONG
			"--all-users",
#else
			"-a",
#endif
			_("option not applicable to this action"));
		return SCW_EXIT_BAD_ARGS;
	}
	/*@+mustfreefresh@ */

	/* Store the selected item, if the action requires one. */
	if (requiresItem) {
		state->item = state->arguments[1];
		state->itemLength = strlen(state->item);	/* flawfinder: ignore */
		/* flawfinder - see strlen() note above. */
	}

#ifdef ENABLE_DEBUGGING
	debugShowStringValue("after parsing", state->configFile);
	debugShowStringValue("after parsing", state->item);
	debugShowIntegerValue("after parsing", state->action);
	debugShowIntegerValue("after parsing", state->argumentCount);
	debugShowBooleanValue("after parsing", state->forceRun);
	debugShowBooleanValue("after parsing", state->listEnabledOnly);
	debugShowBooleanValue("after parsing", state->listDisabledOnly);
	debugShowBooleanValue("after parsing", state->listWithInfo);
	{
		size_t arg;
		for (arg = 0; arg < state->argumentCount; arg++) {
			debugShowStringValue("after parsing", state->arguments[arg]);
		}
	}
#endif

	return 0;
}
