/*
 * Functions for the "faults" action.
 *
 * Copyright 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <string.h>
#include <time.h>


/*
 * Using the metrics files for each item, list all current faults.  Errors,
 * such as items being disabled or having not run for too long, are written
 * to standard error.  Warnings, such as items overrunning, are written to
 * standard output.
 */
int listFaults(struct scwState *state)
{
	struct scwSettings combinedSettings;
	char **usernameArray = NULL;
	size_t usernameCount = 0;
	size_t usernameIndex;

	/* Combine the global and command-line settings. */
	memset(&combinedSettings, 0, sizeof(combinedSettings));
	if (0 != combineSettings(state, &combinedSettings))
		return SCW_EXIT_ERROR;

	/* Enumerate the users to find items for. */

	if (state->allUsers) {
		enumerateAllUsers(state, &usernameArray, &usernameCount);
	} else {
		usernameArray = (char **) stringCopy(state, NULL, sizeof(char *) - 1);
		if (NULL == usernameArray) {
			return SCW_EXIT_ERROR;
		}
		/*@-dependenttrans@ */
		usernameArray[0] = state->username;
		/*@+dependenttrans@ */
		usernameCount = 1;
	}

	for (usernameIndex = 0; NULL != usernameArray && usernameIndex < usernameCount; usernameIndex++) {
		char *originalUsername;
		size_t originalUsernameLength;
		char *username;
		size_t usernameLength;
		char **itemArray = NULL;
		size_t itemCount = 0;
		size_t itemIndex;

		username = usernameArray[usernameIndex];
		if (NULL == username)
			continue;
		usernameLength = strlen(username);	/* flawfinder: ignore */
		/*
		 * flawfinder warns of strlen() with non-null-terminated
		 * strings; the array contains strings we have explicitly
		 * null terminated.
		 */

		originalUsername = state->username;
		originalUsernameLength = state->usernameLength;
		state->username = username;
		state->usernameLength = usernameLength;

		/* Load the per-user configuration for this user. */
		memset(&(state->userSettings), 0, sizeof(state->userSettings));
		if (0 != loadUserConfig(state)) {
			state->username = originalUsername;
			state->usernameLength = originalUsernameLength;
			continue;
		}

		/* Find the items for this user. */
		enumerateUserItems(state, &itemArray, &itemCount);

		/*@-mustfreefresh@ */
		/* splint note: gettext triggers a warning we can't resolve. */

		/* Load each item and display its information. */
		for (itemIndex = 0; NULL != itemArray && itemIndex < itemCount; itemIndex++) {
			struct scwSettings combinedItemSettings;
			struct scwItemStatus itemStatus;

			state->item = itemArray[itemIndex];
			if (NULL == state->item)
				continue;
			state->itemLength = strlen(state->item);	/* flawfinder: ignore */
			/*
			 * flawfinder warns of strlen() with
			 * non-null-terminated strings; the array contains
			 * strings we have explicitly null terminated.
			 */

			debug("%s: %.*s / %.*s", "considering item", state->usernameLength, state->username,
			      state->itemLength, state->item);

			/* Load the item's settings. */
			memset(&(state->itemSettings), 0, sizeof(state->itemSettings));
			if (0 != loadCurrentItemSettings(state))
				continue;

			/* Combine the settings for this item. */
			memset(&combinedItemSettings, 0, sizeof(combinedItemSettings));
			if (0 != combineSettings(state, &combinedItemSettings))
				continue;

			/*
			 * Unconditionally expand the metricsDir value.
			 * This is unconditional though it wastes memory,
			 * because otherwise it could get expanded once in
			 * the global settings and then never re-expanded
			 * when the username or item changes.
			 */
			expandRawValue(state, &(combinedSettings.metricsDir));

			if (NULL == combinedSettings.metricsDir.expandedValue) {
				debug("%s", "no MetricsDir");
				continue;
			}

			debug("%s=[%.*s]", "MetricsDir", combinedSettings.metricsDir.expandedLength,
			      combinedSettings.metricsDir.expandedValue);
#ifdef ENABLE_DEBUGGING
			debugOutputAllSettings(state->item, &combinedSettings);
#endif

			memset(&itemStatus, 0, sizeof(itemStatus));
			if (!loadStatusFromMetrics(&(combinedSettings.metricsDir), &itemStatus, true)) {
				debug("%s", "metrics load failed");
				continue;
			}

			/* Check whether the item is disabled. */
			if (itemStatus.isDisabled) {
				debug("%s", "item is disabled");
				fprintf(stderr, "%.*s/%.*s: %s\n", (int) (state->usernameLength), state->username,
					(int) (state->itemLength), state->item, _("This item is currently disabled."));
				/* No further checks if disabled. */
				continue;
			}

			/* Check whether the item's prerequisites have been met. */
			if (!itemStatus.hasPrerequisitesMet) {
				debug("%s", "prerequisites not met");
				/* Not met - silently skip remaining checks. */
				continue;
			}

			/* Check the success interval has not been exceeded. */
			if (itemStatus.tooLongSinceLastSuccess) {
				debug("%s", "too long since last success");
				fprintf(stderr, "%.*s/%.*s: %s\n", (int) (state->usernameLength),
					state->username, (int) (state->itemLength), state->item,
					_("This item has gone too long without a successful run."));
			}

			/* Check whether the item is currently overrunning. */
			if (itemStatus.isOverrunning && (0 != itemStatus.lastStarted)) {
				debug("%s", "overrunning");
				printf("%.*s/%.*s: %s\n", (int) (state->usernameLength),
				       state->username, (int) (state->itemLength), state->item,
				       _("This item is currently overrunning."));
			}
		}

		/*@+mustfreefresh@ */

		state->item = NULL;
		state->itemLength = 0;

		state->username = originalUsername;
		state->usernameLength = originalUsernameLength;
	}

	return 0;
}
