/*
 * Output debugging information.
 *
 * Copyright 2024-2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
 */

#include "scw-internal.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>


#ifdef ENABLE_DEBUGGING
/*@null@*/ static const char *debugFilename = NULL;

#define DEBUG_FD 10			    /* avoid debug fd being 3 and tripping up status fds */

/*
 * Set the destination for debugging information.
 */
void debugSetDestination(const char *filename)
{
	debugFilename = filename;
}


/*
 * Output debugging information to the file specified earlier by a call to
 * debugSetDestination(), if any.
 *
 * If the line is < 0, closes the currently open debug stream, if any.
 */
void debugWriteOutput(const char *function, const char *file, int line, const char *format, ...)
{
	static bool openAttempted = false;
	static FILE *debugStream = NULL;
	va_list varArgs;
	time_t epochTime;
	struct tm *brokenDownTime;
	char timestampBuffer[128];	 /* flawfinder: ignore */

	/*
	 * flawfinder note: timestampBuffer is only written to by strftime()
	 * which takes its size, and we enforce string termination.
	 */

	if (line < 0) {
		if (NULL != debugStream)
			(void) fclose(debugStream);
		debugStream = NULL;
		return;
	}

	if (false == openAttempted) {
		if (NULL != debugFilename) {
			int debugDescriptor;
			debugDescriptor = fileOpenForAppend(debugFilename);
			if (debugDescriptor >= 0 && debugDescriptor < DEBUG_FD) {
				if (dup2(debugDescriptor, DEBUG_FD) >= 0) {
					(void) close(debugDescriptor);
					debugDescriptor = DEBUG_FD;
				}
			}
			if (debugDescriptor >= 0)
				debugStream = fdopen(debugDescriptor, "a");
			if (NULL != debugStream)
				(void) fprintf(debugStream, "(%s: %d)\n", "debugDescriptor", debugDescriptor);
			openAttempted = true;
		}
	}

	if (NULL == debugStream) {
		return;
	}

	/*
	 * Note that here we use gmtime() rather than localtime(), to avoid
	 * complications from the complexity of localtime().
	 */

	epochTime = time(NULL);
	brokenDownTime = gmtime(&epochTime);
	timestampBuffer[0] = '\0';
	if (0 == strftime(timestampBuffer, sizeof(timestampBuffer), "%Y-%m-%dT%H:%M:%SZ", brokenDownTime)) {
		timestampBuffer[0] = '\0';
	}
	timestampBuffer[sizeof(timestampBuffer) - 1] = '\0';	/* enforce termination */

	(void) fprintf(debugStream, "%s @%d %s (%s:%d): ", timestampBuffer, getpid(), function, file, line);

	va_start(varArgs, format);
	(void) vfprintf(debugStream, format, varArgs);	/* flawfinder: ignore */
	va_end(varArgs);

	/*
	 * flawfinder note: vfprintf format is explicitly controlled by the
	 * caller of this function - no mitigation possible or desirable.
	 */

	(void) fprintf(debugStream, "\n");
	(void) fflush(debugStream);
}

#else				/* ! ENABLE_DEBUGGING */

/*
 * Stub debugging destination function.
 */
void debugSetDestination( __attribute__((unused))
			 /*@unused@ */
			 const char *filename)
{
}

/*
 * Stub debugging output function.
 */
void debugWriteOutput( __attribute__((unused))
		      /*@unused@ */
		      const char *function, __attribute__((unused))
		      /*@unused@ */
		      const char *file, __attribute__((unused))
		      /*@unused@ */
		      int line, __attribute__((unused))
		      /*@unused@ */
		      const char *format, ...)
{
}

#endif				/* ENABLE_DEBUGGING */
