/*
 * Watch the given directory, producing output files containing lists of
 * files changed.
 *
 * Copyright 2014, 2021, 2023, 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

/* TODO: save memory by only allocating leafnames, not the whole absolute path. */

/* File and subdirectory array allocation chunk size */
#define DIRCONTENTS_ALLOC_CHUNK	128

/* Directory index array allocation chunk size */
#define DIR_INDEX_ALLOC_CHUNK 1024

/* Change queue allocation chunk size */
#define CHANGE_QUEUE_ALLOC_CHUNK 1024

/* Changed paths list allocation chunk size */
#define CHANGEDPATH_ALLOC_CHUNK 1024

#define _ATFILE_SOURCE

#include "config.h"
#include "common.h"
#include "watch.h"

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdint.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <search.h>
#ifndef SPLINT
/* splint 3.1.2 chokes on syslog.h */
#include <syslog.h>
#endif
#include <limits.h>
#include <sys/inotify.h>
#include <sys/select.h>
#include <poll.h>
#include <fnmatch.h>
#include <assert.h>


/*
 * Actions to take on inotify events.
 */
typedef enum {
	IN_ACTION_CREATE,
	IN_ACTION_UPDATE,
	IN_ACTION_DELETE,
	IN_ACTION_NONE
} inotify_action_t;

struct ds_file_s;
typedef struct ds_file_s *ds_file_t;
struct ds_dir_s;
typedef struct ds_dir_s *ds_dir_t;
struct ds_topdir_s;
typedef struct ds_topdir_s *ds_topdir_t;
struct ds_watch_index_s;
typedef struct ds_watch_index_s *ds_watch_index_t;
struct ds_change_queue_s;
typedef struct ds_change_queue_s *ds_change_queue_t;

typedef /*@null@ */ void *nullable_pointer;


/*
 * Structure holding information about a file.  An example path would be
 * "0/12/12345/foo.txt"; the leaf would be "foo.txt"; the absolute_path
 * would be "/top/dir/0/12/12345/foo.txt" if the top level directory was
 * "/top/dir".
 *
 * The "absolute_path" is malloc()ed; "path" and "leaf" point within it, so
 * aren't separately allocated.
 */
struct ds_file_s {
	char *absolute_path;		 /* absolute path to file */
	/*@null@ *//*@dependent@ */ char *path;
	/* path relative to top level */
	/*@null@ *//*@dependent@ */ char *leaf;
	/* leafname of this file */
	time_t mtime;			 /* file last-modification time */
	off_t size;			 /* file size */
	/*@null@ *//*@dependent@ */ ds_dir_t parent;
	/* containing directory */
	bool seen_in_rescan;		 /* set during dir rescan */
};


/*
 * Structure holding information about a directory.  An example path would
 * be "0/12/12345" with a leaf of "12345".
 *
 * The "absolute_path" is malloc()ed; "path" and "leaf" point within it, so
 * aren't separately allocated.
 */
struct ds_dir_s {
	char *absolute_path;		 /* absolute path to directory */
	/*@null@ *//*@dependent@ */
	char *path;			 /* path relative to top level */
	/*@null@ *//*@dependent@ */
	char *leaf;			 /* leafname of this directory */
	int wd;				 /* inotify watch fd (if dir) */
	unsigned int depth;		 /* subdirs deep from top level */
	size_t file_count;		 /* number of files in directory */
	size_t subdir_count;		 /* number of immediate subdirs */
	/*@null@ */
	ds_file_t *files;		 /* array of files */
	/*@null@ */
	ds_dir_t *subdirs;		 /* array of subdirectories */
	size_t file_array_alloced;	 /* file entries allocated */
	size_t subdir_array_alloced;	 /* subdir entries allocated */
	nullable_pointer tree_files;	 /* binary tree of files */
	nullable_pointer tree_subdirs;	 /* binary tree of subdirs */
	/*@null@ *//*@dependent@ */
	ds_dir_t parent;		 /* pointer to parent directory */
	/*@null@ *//*@dependent@ */
	ds_topdir_t topdir;		 /* pointer to top directory */
	/* Booleans, at the end to minimise wasted space due to padding. */
	bool files_unsorted;		 /* set if files array needs sorting */
	bool subdirs_unsorted;		 /* set if subdirs array needs sorting */
	bool seen_in_rescan;		 /* set during dir rescan */
};


/*
 * Structure holding information about a top-level directory.
 */
struct ds_topdir_s {
	struct ds_dir_s dir;		 /* the directory's details */
	/*
	 * The watch_index array maps each inotify watch descriptor to a
	 * directory pointer. Each pointer is an alias of the original.
	 */
	/*@null@ */
	ds_watch_index_t watch_index;	 /* array of watch descriptors */
	size_t watch_index_length;	 /* number of entries in array */
	size_t watch_index_alloced;	 /* number of entries allocated */
	/*
	 * The change_queue array is populated when inotify events are
	 * received, and contains the file or directory pointers associated
	 * with each event.  Processing the change queue means checking each
	 * of these items to see whether they really have changed, in which
	 * case their path gets added to the changed_paths, and for
	 * directories, they are rescanned.
	 *
	 * These file and dir pointers are aliases of the originals, so
	 * before file or dir structures are freed, their pointers must be
	 * removed from this array as well.
	 *
	 * The tree_change_queue pointer is a binary tree (tfind() etc)
	 * containing aliases of the absolute_path string pointers from
	 * change queue array entries.  It is used to check for duplicates.
	 */
	/*@null@ */
	ds_change_queue_t change_queue;	 /* array of changes needed */
	size_t change_queue_length;	 /* number of changes in queue */
	size_t change_queue_alloced;	 /* array size allocated */
	nullable_pointer tree_change_queue;	/* tree of change queue paths */
	/*
	 * The changed_paths array contains the paths that are known to have
	 * changed, as allocated strings (copies, not aliases of the file or
	 * dir path pointers).  It is populated by change queue processing
	 * and emptied by dump_changed_paths().
	 *
	 * The tree_changed_paths pointer is a binary tree (tfind() etc)
	 * containing aliases of the string pointers from the changed_paths
	 * array.  It is used to check for duplicates.
	 */
	/*@null@ */
	char **changed_paths;		 /* array of changed paths */
	size_t changed_paths_length;	 /* number of paths in array */
	size_t changed_paths_alloced;	 /* array size allocated */
	nullable_pointer tree_changed_paths;	/* binary tree of changed paths */
	/* Integers, at the end to avoid padding if they're smaller than a pointer. */
	int fd_inotify;			 /* directory watch file descriptor */
	int highest_wd;			 /* highest watch descriptor value */
	/* Booleans, at the end to minimise wasted space due to padding. */
	bool watch_index_unsorted;	 /* set if watch index array needs sorting */
};


/*
 * Structure for indexing directory structures by watch identifier.
 */
struct ds_watch_index_s {
	int wd;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


/*
 * Structure for a file check or directory scan queue entry.
 */
struct ds_change_queue_s {
	time_t when;
	/*@null@ *//*@dependent@ */ ds_file_t file;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


static int ds_filename_valid(const char *name);

static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add( /*@dependent@ */ ds_dir_t dir, const char *name);
static void ds_file_remove(ds_file_t file);
static int ds_file_checkchanged(ds_file_t file);

static /*@null@ */ ds_topdir_t ds_dir_toplevel(int fd_inotify, const char *top_path);
/*@dependent@ */
static /*@null@ */ ds_dir_t ds_dir_add( /*@dependent@ */ ds_dir_t dir, const char *name);
static void ds_dir_remove(ds_dir_t dir);
static int ds_dir_scan( /*@dependent@ */ ds_dir_t dir, bool no_recurse);

static ssize_t ds_dir_file_lookup(ds_dir_t dir, ds_file_t file);
static ssize_t ds_dir_subdir_lookup(ds_dir_t dir, ds_dir_t subdir);

static void ds_watch_index_add( /*@dependent@ */ ds_dir_t dir, int wd);
static void ds_watch_index_remove(ds_topdir_t topdir, int wd);
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_topdir_t topdir, int wd, /*@null@ */ size_t *idxptr);

static void ds_change_queue_file_add( /*@dependent@ */ ds_file_t file, time_t when);
static void ds_change_queue_file_remove(ds_file_t file);
static void ds_change_queue_dir_add( /*@dependent@ */ ds_dir_t dir, time_t when);
static void ds_change_queue_dir_remove(ds_dir_t dir);

static void ds_change_queue_process(ds_topdir_t topdir, time_t work_until);

static void mark_file_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_file_t file);
static void mark_dir_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_dir_t dir);
static void dump_changed_paths(ds_topdir_t topdir, const char *changedpath_dir);

static bool extend_array( /*@null@ */ nullable_pointer * array_ptr, /*@null@ */ size_t *length_ptr, size_t entry_size,
			 size_t new_length);


static struct watch_dir_params_s *watch_dir_params;
static bool watch_dir_exit_now = false;


/*
 * Dummy function which does nothing, for tdestroy() to call.
 */
static void _tdestroy_noop( /*@unused@ */  __attribute__((unused))
			   void *ptr)
{
}


/*
 * Extend an array, calling die() and returning false on failure.
 */
bool extend_array( /*@null@ */ nullable_pointer * array_ptr, /*@null@ */ size_t *length_ptr, size_t entry_size,
		  size_t new_length)
{
	void *old_ptr;
	void *new_ptr;

	if ((NULL == array_ptr) || (NULL == length_ptr))
		return false;

	old_ptr = *array_ptr;
	new_ptr = realloc(old_ptr, new_length * entry_size);
	if (NULL == new_ptr) {
		die("%s: %s", "realloc", strerror(errno));
		*array_ptr = NULL;
		*length_ptr = 0;
		return false;
	}

	*array_ptr = new_ptr;
	*length_ptr = new_length;
	return true;
}


/*
 * Add the given watch descriptor to the directory index.
 */
static void ds_watch_index_add( /*@dependent@ */ ds_dir_t dir, int wd)
{
	if (NULL == dir)
		return;
	if (wd < 0)
		return;
	if (NULL == dir->topdir)
		return;

	/*
	 * Extend the array if we need to.
	 */
	if (dir->topdir->watch_index_length >= dir->topdir->watch_index_alloced) {
		if (!extend_array
		    ((void **) (&(dir->topdir->watch_index)), &(dir->topdir->watch_index_alloced),
		     sizeof(dir->topdir->watch_index[0]), dir->topdir->watch_index_alloced + DIR_INDEX_ALLOC_CHUNK))
			return;
	}
	assert(NULL != dir->topdir->watch_index);
	if (NULL == dir->topdir->watch_index)
		return;

	memset(&(dir->topdir->watch_index[dir->topdir->watch_index_length]), 0, sizeof(dir->topdir->watch_index[0]));

	dir->topdir->watch_index[dir->topdir->watch_index_length].wd = wd;
	dir->topdir->watch_index[dir->topdir->watch_index_length].dir = dir;

	dir->topdir->watch_index_length++;

	/*
	 * Mark the watch index array as unsorted, if the new wd is not
	 * greater than the highest wd seen so far (if it's greater than or
	 * equal to, then the array is still in wd order).
	 */
	if (wd < dir->topdir->highest_wd) {
		dir->topdir->watch_index_unsorted = true;
	} else {
		dir->topdir->highest_wd = wd;
	}
}


/*
 * Remove the given watch descriptor from the directory index.
 */
static void ds_watch_index_remove(ds_topdir_t topdir, int wd)
{
	size_t startidx, readidx, writeidx;

	if (NULL == topdir)
		return;
	if (NULL == topdir->watch_index)
		return;

	startidx = 0;
	(void) ds_watch_index_lookup(topdir, wd, &startidx);

	for (readidx = startidx, writeidx = startidx; readidx < topdir->watch_index_length; readidx++) {
		if (topdir->watch_index[readidx].wd == wd) {
			continue;
		}
		if (readidx != writeidx) {
			topdir->watch_index[writeidx] = topdir->watch_index[readidx];
		}
		writeidx++;
	}
	topdir->watch_index_length = writeidx;
}


/*
 * Comparison function for the directory index.
 */
static int ds_watch_index_compare(const void *a, const void *b)
{
	if (((ds_watch_index_t) a)->wd < ((ds_watch_index_t) b)->wd)
		return -1;
	if (((ds_watch_index_t) a)->wd > ((ds_watch_index_t) b)->wd)
		return 1;
	return 0;
}


/*
 * Return the directory structure associated with the given watch
 * descriptor, or NULL if none.  Populates *idxptr with the associated
 * watch_index array index number, if idxptr is not NULL.
 */
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_topdir_t topdir, int wd, /*@null@ */ size_t *idxptr)
{
	struct ds_watch_index_s key;
	/*@null@ */ ds_watch_index_t result;

	if (NULL == topdir)
		return NULL;
	if (NULL == topdir->watch_index)
		return NULL;

	if ((topdir->watch_index_unsorted)
	    && (topdir->watch_index_length > 0)) {
		debug("%s: %s", topdir->dir.absolute_path, "sorting watch index");
		qsort(topdir->watch_index, topdir->watch_index_length,
		      sizeof(topdir->watch_index[0]), ds_watch_index_compare);
		topdir->watch_index_unsorted = false;
		topdir->highest_wd = topdir->watch_index[topdir->watch_index_length - 1].wd;
	}

	key.wd = wd;
	debug("%s: %d: %s", topdir->dir.absolute_path, wd, "searching for watch index");
	result =
	    bsearch(&key, topdir->watch_index, topdir->watch_index_length,
		    sizeof(topdir->watch_index[0]), ds_watch_index_compare);

	if (NULL == result) {
		debug("%s: %d: %s", topdir->dir.absolute_path, wd, "watch index not found");
		return NULL;
	}

	debug("%s: %d: %s: %p (- %p) = %s", topdir->dir.absolute_path, wd, "bsearch result", result,
	      topdir->watch_index, NULL == result->dir ? "(null)" : result->dir->absolute_path);

	if (NULL != idxptr) {
		assert(result >= topdir->watch_index);
		if (result < topdir->watch_index)
			return result->dir;
		/*
		 * Subtracting the pointers gives us the array index since
		 * the pointers are of type ds_watch_index_t.
		 */
		*idxptr = (size_t) (result - topdir->watch_index);
	}

	return result->dir;
}


/*
 * Comparison function that compares two pointers to ds_file_t pointers by
 * their leafname.
 */
static int ds_fileptr_leaf_compare(const void *a, const void *b)
{
	ds_file_t ptr_a, ptr_b;
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	ptr_a = *((ds_file_t *) a);
	ptr_b = *((ds_file_t *) b);

	if ((NULL == ptr_a) || (NULL == ptr_b))
		return 0;

	x = ptr_a->leaf;
	y = ptr_b->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Return the index into dir->files[] that contains the given file, or -1 if
 * it's not present.  If the files array is unsorted, it is sorted first.
 */
static ssize_t ds_dir_file_lookup(ds_dir_t dir, ds_file_t file)
{
	/*@null@ */ ds_file_t *result;
	size_t index;

	if ((NULL == dir) || (NULL == file) || (NULL == dir->files) || (NULL == file->leaf))
		return -1;

	/* Sort the array if it needs to be sorted. */
	if (dir->files_unsorted) {
		debug("%s: %s", dir->absolute_path, "sorting files");
		qsort(dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);
		dir->files_unsorted = false;
#if ENABLE_DEBUGGING
		{
			size_t idx;
			for (idx = 0; NULL != dir->files && idx < dir->file_count; idx++) {
				debug("%s: %s[%d]: [%s]", dir->absolute_path, "files", idx, dir->files[idx]->leaf);
			}
		}
#endif
	}

	debug("%s: %s: %s", dir->absolute_path, file->leaf, "searching for file entry");
	result = bsearch(&file, dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);

	if (NULL == result) {
		debug("%s: %s: %s", dir->absolute_path, file->leaf, "file entry not found");
		return -1;
	}

	assert(result >= dir->files);
	if (result < dir->files)
		return -1;

	/*
	 * Subtracting the pointers gives us the array index since the
	 * pointers are of type ds_file_t.
	 */
	index = (size_t) (result - dir->files);

	debug("%s: %s: %s: %p (- %p) -> [%d]", dir->absolute_path, file->leaf, "bsearch result", result, dir->files,
	      index);

	assert(index < dir->file_count);
	if (index >= dir->file_count)
		return -1;

	return (ssize_t) index;
}


/*
 * Comparison function that compares two pointers to ds_dir_t pointers by
 * their leafname.
 */
static int ds_dirptr_leaf_compare(const void *a, const void *b)
{
	ds_dir_t ptr_a, ptr_b;
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	ptr_a = *((ds_dir_t *) a);
	ptr_b = *((ds_dir_t *) b);

	if ((NULL == ptr_a) || (NULL == ptr_b))
		return 0;

	x = ptr_a->leaf;
	y = ptr_b->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Return the index into dir->subdirs[] that contains the given
 * subdirectory, or -1 if it's not present.  If the subdirs array is
 * unsorted, it is sorted first.
 */
static ssize_t ds_dir_subdir_lookup(ds_dir_t dir, ds_dir_t subdir)
{
	/*@null@ */ ds_dir_t *result;
	size_t index;

	if ((NULL == dir) || (NULL == subdir) || (NULL == dir->subdirs) || (NULL == subdir->leaf))
		return -1;

	/* Sort the array if it needs to be sorted. */
	if (dir->subdirs_unsorted) {
		debug("%s: %s", dir->absolute_path, "sorting subdirs");
		qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
		dir->subdirs_unsorted = false;
#if ENABLE_DEBUGGING
		{
			size_t idx;
			for (idx = 0; NULL != dir->subdirs && idx < dir->subdir_count; idx++) {
				debug("%s: %s[%d]: [%s]", dir->absolute_path, "subdirs", idx, dir->subdirs[idx]->leaf);
			}
		}
#endif
	}

	debug("%s: %s: %s", dir->absolute_path, subdir->leaf, "searching for subdir entry");
	result = bsearch(&subdir, dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);

	if (NULL == result) {
		debug("%s: %s: %s", dir->absolute_path, subdir->leaf, "subdir entry not found");
		return -1;
	}

	assert(result >= dir->subdirs);
	if (result < dir->subdirs)
		return -1;

	/*
	 * Subtracting the pointers gives us the array index since the
	 * pointers are of type ds_dir_t.
	 */
	index = (size_t) (result - dir->subdirs);

	debug("%s: %s: %s: %p (- %p) -> [%d]", dir->absolute_path, subdir->leaf, "bsearch result", result,
	      dir->subdirs, index);

	assert(index < dir->subdir_count);
	if (index >= dir->subdir_count)
		return -1;

	return (ssize_t) index;
}


/*
 * Add an entry to the change queue.
 *
 * In no_file_tracking mode, if this is a file, then immediately add the
 * file to the changed paths list without checks, and then remove the file
 * from its parent directory structure, freeing it (via ds_file_remove).
 */
static void _ds_change_queue_add( /*@null@ */ ds_topdir_t topdir, time_t when,	/*@null@ *//*@dependent@ */
				 ds_file_t file,
				 /*@null@ *//*@dependent@ */ ds_dir_t dir)
{
	char *absolute_path;

	if (NULL == topdir)
		return;

	if ((NULL == file) && (NULL == dir))
		return;

	/*
	 * Shortcut straight to marking the path as changed, and forgetting
	 * the file, if this is a file and we're in no_file_tracking mode.
	 */
	if (watch_dir_params->no_file_tracking && (NULL != file)) {
		if (NULL != file->parent) {
			mark_file_path_changed(file->parent->topdir, file);
		}
		ds_file_remove(file);
		return;
	}

	/*
	 * Check the change isn't already queued - don't queue it twice.
	 */
	absolute_path = NULL;
	if (NULL != file)
		absolute_path = file->absolute_path;
	if (NULL != dir)
		absolute_path = dir->absolute_path;
	/*@-mustfreefresh@ */
	if (NULL != absolute_path && NULL != topdir->tree_change_queue) {
		nullable_string result;
		result = tfind(absolute_path, &(topdir->tree_change_queue), compare_with_strcmp);
		if (NULL != result) {
			debug("%s: %s", absolute_path, "path is already in the change queue");
			return;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the array if necessary.
	 */
	if (topdir->change_queue_length >= topdir->change_queue_alloced) {
		if (!extend_array
		    ((void **) (&(topdir->change_queue)), &(topdir->change_queue_alloced),
		     sizeof(topdir->change_queue[0]), topdir->change_queue_alloced + CHANGE_QUEUE_ALLOC_CHUNK))
			return;
	}
	assert(NULL != topdir->change_queue);
	if (NULL == topdir->change_queue)
		return;

	if (NULL != file) {
		debug("%s: %s: %s", "adding to change queue", "check file", file->path);
	} else if (NULL != dir) {
		debug("%s: %s: %s", "adding to change queue", "scan directory", dir->path);
	}

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	topdir->change_queue[topdir->change_queue_length].when = when;
	topdir->change_queue[topdir->change_queue_length].file = file;
	topdir->change_queue[topdir->change_queue_length].dir = dir;

	/* Add the absolute path to the binary tree. */
	/*@-nullstate@ */
	if (NULL != absolute_path)
		(void) tsearch(absolute_path, &(topdir->tree_change_queue), compare_with_strcmp);
	/*@+nullstate@ *//* topdir->tree_change_queue is OK to be NULL here. */

	topdir->change_queue_length++;
}


/*
 * Queue a file check.
 *
 * Note that in no_file_tracking mode, file will be freed, so never use file
 * again after calling this function (see _ds_change_queue_add).
 */
static void ds_change_queue_file_add( /*@dependent@ */ ds_file_t file, time_t when)
{
	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;

	if (0 == when) {
		/*
		 * If no specific time was given, choose a time that's a
		 * couple of seconds in the future so it's more likely it's
		 * finished being changed before we take any action.
		 *
		 * The larger the file is, the longer we give it.
		 */
		if (file->size < 102400) {  /* < 100KiB */
			when = time(NULL) + 2;
		} else if (file->size < 1048576) {	/* < 1MiB */
			when = time(NULL) + 3;
		} else if (file->size < 10485760) {	/* < 10MiB */
			when = time(NULL) + 4;
		} else if (file->size < 104857600) {	/* < 100MiB */
			when = time(NULL) + 5;
		} else {		    /* >= 100 MiB */
			when = time(NULL) + 6;
		}
	}

	_ds_change_queue_add(file->parent->topdir, when, file, NULL);
}


/*
 * Remove a file from the change queue.
 */
static void ds_change_queue_file_remove(ds_file_t file)
{
	ds_topdir_t topdir;
	size_t idx;

	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;
	if (NULL == file->parent->topdir->change_queue)
		return;

	topdir = file->parent->topdir;

	/* Remove the path from the change queue binary tree. */
	if (NULL != topdir->tree_change_queue)
		(void) tdelete(file->absolute_path, &(topdir->tree_change_queue), compare_with_strcmp);

	/* Clear the entry from the change queue array. */
	for (idx = 0; idx < topdir->change_queue_length; idx++) {
		if (topdir->change_queue[idx].file == file) {
			topdir->change_queue[idx].file = NULL;
		}
	}
}


/*
 * Queue a directory scan.
 */
static void ds_change_queue_dir_add( /*@dependent@ */ ds_dir_t dir, time_t when)
{
	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;
	if (0 == when)
		when = time(NULL);
	_ds_change_queue_add(dir->topdir, when, NULL, dir);
}


/*
 * Remove a directory from the change queue.
 */
static void ds_change_queue_dir_remove(ds_dir_t dir)
{
	size_t idx;

	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;
	if (NULL == dir->topdir->change_queue)
		return;

	/* Remove the path from the change queue binary tree. */
	if (NULL != dir->topdir->tree_change_queue)
		(void) tdelete(dir->absolute_path, &(dir->topdir->tree_change_queue), compare_with_strcmp);

	/* Clear the entry from the change queue array. */
	for (idx = 0; idx < dir->topdir->change_queue_length; idx++) {
		if (dir->topdir->change_queue[idx].dir == dir) {
			dir->topdir->change_queue[idx].dir = NULL;
		}
	}
}


/*
 * Comparison function that compares two ds_file_t pointers by their
 * leafname.
 */
static int ds_file_leaf_compare(const void *a, const void *b)
{
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	x = ((ds_file_t) a)->leaf;
	y = ((ds_file_t) b)->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Add a file to the list of files in the given directory; if the file is
 * already in the list, return the existing file.
 *
 * The "name" string should contain the name of the file relative to the
 * directory (not the full path), e.g. "somefile".
 *
 * Returns the file, or NULL on error.
 */
static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add( /*@dependent@ */ ds_dir_t dir, const char *name)
{
	ds_file_t file;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;
	if (NULL == dir->absolute_path)
		return NULL;

	/*
	 * Use a binary tree to find this file if we've already added it to
	 * this directory - if we have, return a pointer to its structure.
	 */
	/*@-mustfreefresh@ */
	if (NULL != dir->tree_files) {
		struct ds_file_s key;
		ds_file_t *result;

		key.leaf = (char *) name;
		result = tfind(&key, &(dir->tree_files), ds_file_leaf_compare);
		if (NULL != result) {
			debug("%s: %s: %s", dir->absolute_path, name, "skipping duplicate file");
			return *result;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the file array in the directory structure if we need to.
	 */
	if (dir->file_count >= dir->file_array_alloced) {
		size_t target_array_alloced;

		target_array_alloced = dir->file_array_alloced;
		while (target_array_alloced <= dir->file_count) {
			target_array_alloced += DIRCONTENTS_ALLOC_CHUNK;
		}
		if (!extend_array
		    ((void **) (&(dir->files)), &(dir->file_array_alloced), sizeof(dir->files[0]),
		     target_array_alloced))
			return NULL;
	}

	/*
	 * At this point, dir->files should have been allocated.
	 */
	assert(NULL != dir->files);
	if (NULL == dir->files)
		return NULL;

	/*
	 * Allocate a new file structure.
	 */
	file = calloc(1, sizeof(*file));
	if (NULL == file) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the file structure.
	 */
	/*@-unrecog@ */
	/*@-compdestroy@ */
	/* splint doesn't know about asprintf(). */
	if (asprintf(&(file->absolute_path), "%s/%s", dir->absolute_path, name) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		free(file);
		return NULL;
	}
	/*@+unrecog@ *//*@+compdestroy@ */
	if (NULL != dir->topdir) {
		size_t topdir_absolute_path_len;
		/*
		 * If there's a topmost directory pointer, the new
		 * file's relative path is relative to that topmost
		 * directory.
		 */
		topdir_absolute_path_len = strlen(dir->topdir->dir.absolute_path);	/* flawfinder: ignore */
		file->path = &(file->absolute_path[topdir_absolute_path_len + 1]);
	} else {
		size_t dir_absolute_path_len;
		/*
		 * If there's no topmost directory pointer, the new file's
		 * relative path is relative only to this directory.
		 */
		dir_absolute_path_len = strlen(dir->absolute_path);	/* flawfinder: ignore */
		file->path = &(file->absolute_path[dir_absolute_path_len + 1]);
	}
	/* flawfinder - strlen() on asprintf()-created null-terminated strings. */
	file->leaf = ds_leafname(file->absolute_path);
	file->parent = dir;
	file->seen_in_rescan = false;

	/*
	 * Add the file to the directory structure.
	 */
	dir->files[dir->file_count] = file;
	dir->file_count++;

	/* Add the file to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(file, &(dir->tree_files), ds_file_leaf_compare);
	/*@+nullstate@ *//* dir->tree_files is OK to be NULL here. */

	/* Mark the files array as unsorted. */
	dir->files_unsorted = true;

	return file;
}


/*
 * Free a file information structure and its contents.
 *
 * The parent directory's "files" list is updated to remove this file from
 * it.
 */
static void ds_file_remove(ds_file_t file)
{
	if (NULL == file)
		return;
	if (NULL == file->path)
		return;

	debug("%s: %s", file->absolute_path, "removing from file list");

	/*
	 * Remove this file from our parent directory's file listing, if we
	 * have a parent.
	 */
	if ((NULL != file->parent) && (NULL != file->leaf)) {
		size_t readidx, writeidx;
		ssize_t startidx;

		startidx = ds_dir_file_lookup(file->parent, file);

		if (startidx >= 0) {
			debug("%s: %s: %s: %s[%d]", file->absolute_path, file->leaf, "removing from parent files index",
			      file->parent->absolute_path, startidx);
			for (readidx = (size_t) startidx + 1, writeidx = (size_t) startidx;
			     NULL != file->parent->files && readidx < file->parent->file_count; readidx++, writeidx++) {
				file->parent->files[writeidx] = file->parent->files[readidx];
			}
			file->parent->file_count = writeidx;
		}
	}

	/* Remove the file from the change queue. */
	ds_change_queue_file_remove(file);

	/* Remove the file from its directory's binary tree. */
	if (NULL != file->parent && NULL != file->parent->tree_files) {
		(void) tdelete(file, &(file->parent->tree_files), ds_file_leaf_compare);
	}

	/*
	 * Free the memory used by the pathname.
	 */
	free(file->absolute_path);
	file->absolute_path = NULL;
	file->path = NULL;
	file->leaf = NULL;

	/* Free the file structure itself. */
	free(file);
}


/*
 * Check the given file's mtime and size; if either have changed, return 1.
 *
 * Returns 0 if nothing has changed, 1 if it has, or -1 if the file does not
 * exist or is not a regular file.
 *
 * If the file is successfully opened but there is an error while reading
 * it, 0 is returned as if it had not changed.
 */
static int ds_file_checkchanged(ds_file_t file)
{
	struct stat sb;

	if (NULL == file)
		return -1;

	if (NULL == file->absolute_path)
		return -1;

	memset(&sb, 0, sizeof(sb));

	if (lstat(file->absolute_path, &sb) != 0)
		return -1;

	if (!S_ISREG((mode_t) (sb.st_mode)))
		return -1;

	if (((time_t) (sb.st_mtime) == file->mtime) && ((off_t) (sb.st_size) == file->size))
		return 0;

	debug("%s: %s", NULL == file->path ? "(null)" : file->path, "file changed");

	file->mtime = (time_t) (sb.st_mtime);
	file->size = (off_t) (sb.st_size);

	return 1;
}


/*
 * Allocate and return a new top-level directory absolutely rooted at
 * "top_path".  All reported paths within the structure will be relative to
 * "top_path".
 *
 * The "fd_inotify" parameter should be the file descriptor to add directory
 * watches to for inotify, or -1 if inotify is not being used.
 */
static /*@null@ */ ds_topdir_t ds_dir_toplevel(int fd_inotify, const char *top_path)
{
	/*@null@ */ ds_topdir_t topdir;
	size_t absolute_path_len;

	if (strlen(top_path) > PATH_MAX) {  /* flawfinder: ignore */
		die("%s: %s", top_path, strerror(ENAMETOOLONG));
		return NULL;
	}
	/* flawfinder - top_path is guaranteed null-terminated by the caller. */

	topdir = calloc(1, sizeof(*topdir));
	if (NULL == topdir) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*@-unrecog@ */
	/*@-mustfreeonly@ */
	/* splint doesn't know about realpath */
	topdir->dir.absolute_path = realpath(top_path, NULL);	/* flawfinder: ignore */
	/*@+unrecog@ *//*@+mustfreeonly@ */
	if (NULL == topdir->dir.absolute_path) {
		die("%s: %s", "realpath", strerror(errno));
		free(topdir);
		return NULL;
	}

	absolute_path_len = strlen(topdir->dir.absolute_path);	/* flawfinder: ignore */

	/*
	 * flawfinder rationale: realpath()'s destination buffer is
	 * allocated by realpath() itself here, and we've checked that
	 * top_path is not too long; realpath() null-terminates the string
	 * it outputs, so calling strlen() on it will be OK.
	 */

	topdir->dir.path = &(topdir->dir.absolute_path[absolute_path_len]);
	topdir->dir.leaf = topdir->dir.path;
	topdir->dir.wd = -1;
	topdir->dir.depth = 0;
	topdir->dir.parent = NULL;
	topdir->dir.topdir = topdir;
	topdir->dir.seen_in_rescan = false;

	topdir->fd_inotify = fd_inotify;
	topdir->highest_wd = -1;

	return topdir;
}


/*
 * Comparison function that compares two ds_dir_t pointers by their
 * leafname.
 */
static int ds_dir_leaf_compare(const void *a, const void *b)
{
	char *x;
	char *y;

	if ((NULL == a) || (NULL == b))
		return 0;

	x = ((ds_dir_t) a)->leaf;
	y = ((ds_dir_t) b)->leaf;

	if ((NULL == x) || (NULL == y))
		return 0;

	return strcmp(x, y);
}


/*
 * Add a subdirectory to the list of subdirectories in the given parent
 * directory; if the subdirectory is already in the list, just return the
 * existing subdirectory.
 *
 * The "name" string should contain the name of the directory relative to
 * the parent directory (not the full path), e.g. "somedir".
 *
 * Returns the subdirectory, or NULL on error.
 */
/*@dependent@ */
static /*@null@ */ ds_dir_t ds_dir_add( /*@dependent@ */ ds_dir_t dir, const char *name)
{
	/*@null@ */ ds_dir_t subdir;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;

	/*
	 * Check that this subdirectory wouldn't be too deep.
	 */
	if (dir->depth >= watch_dir_params->max_dir_depth) {
		debug("%s/%s: %s", dir->absolute_path, name, "too deep - not adding");
		return NULL;
	}

	/*
	 * Use a binary tree to find this subdirectory if we already know
	 * about it - if we do, return a pointer to its structure.
	 */
	/*@-mustfreefresh@ */
	if (NULL != dir->tree_subdirs) {
		struct ds_dir_s key;
		ds_dir_t *result;
		key.leaf = (char *) name;
		result = tfind(&key, &(dir->tree_subdirs), ds_dir_leaf_compare);
		if (NULL != result) {
			debug("%s: %s: %s", dir->absolute_path, name, "skipping duplicate directory");
			return *result;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the subdirectory array in the directory structure if we
	 * need to.
	 */
	if (dir->subdir_count >= dir->subdir_array_alloced) {
		size_t target_array_alloced;

		target_array_alloced = dir->subdir_array_alloced;
		while (target_array_alloced <= dir->subdir_count) {
			target_array_alloced += DIRCONTENTS_ALLOC_CHUNK;
		}
		if (!extend_array
		    ((void **) (&(dir->subdirs)), &(dir->subdir_array_alloced), sizeof(dir->subdirs[0]),
		     target_array_alloced))
			return NULL;
	}

	/*
	 * At this point, dir->subdirs should have been allocated.
	 */
	assert(NULL != dir->subdirs);
	if (NULL == dir->subdirs)
		return NULL;

	/*
	 * Allocate a new directory structure for the subdirectory.
	 */
	subdir = calloc(1, sizeof(*subdir));
	if (NULL == subdir) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the new subdirectory structure.
	 */
	/*@-compdestroy@ */
	/* splint thinks a failed asprintf() creates storage we must free. */
	if (asprintf(&(subdir->absolute_path), "%s/%s", dir->absolute_path, name) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		free(subdir);
		return NULL;
	}
	/*@+compdestroy@ */
	if (NULL != dir->topdir) {
		size_t topdir_absolute_path_len;
		/*
		 * If there's a topmost directory pointer, the new
		 * subdirectory's relative path is relative to that topmost
		 * directory.
		 */
		topdir_absolute_path_len = strlen(dir->topdir->dir.absolute_path);	/* flawfinder: ignore */
		subdir->path = &(subdir->absolute_path[topdir_absolute_path_len + 1]);
	} else {
		size_t dir_absolute_path_len;
		/*
		 * If there's no topmost directory pointer, the new
		 * subdirectory's relative path is relative only to this
		 * directory.
		 */
		dir_absolute_path_len = strlen(dir->absolute_path);	/* flawfinder: ignore */
		subdir->path = &(subdir->absolute_path[dir_absolute_path_len + 1]);
	}
	/* flawfinder - strlen() on asprintf()-created null-terminated strings. */
	subdir->leaf = ds_leafname(subdir->absolute_path);

	subdir->wd = -1;
	subdir->depth = dir->depth + 1;
	subdir->parent = dir;
	subdir->topdir = dir->topdir;
	subdir->seen_in_rescan = false;

	/*
	 * Add the subdirectory to the directory structure.
	 */
	dir->subdirs[dir->subdir_count] = subdir;
	dir->subdir_count++;

	/* Add the subdirectory to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(subdir, &(dir->tree_subdirs), ds_dir_leaf_compare);
	/*@+nullstate@ *//* dir->tree_subdirs is OK to be NULL here. */

	/* Mark the subdirectories array as unsorted. */
	dir->subdirs_unsorted = true;

	return subdir;
}


/*
 * Free a directory information structure.  This includes all files and
 * directories it contains.
 *
 * The parent directory's "subdirs" list is updated to remove this
 * subdirectory from it.
 *
 * If this the top directory, it is not freed (since in that case it will be
 * part of a ds_topdir_s structure, not allocated in its own right), but all
 * its values are set to NULL.
 */
static void ds_dir_remove(ds_dir_t dir)
{
	bool is_top_dir;
	size_t item;

	if (NULL == dir)
		return;

	debug("%s: %s", dir->absolute_path, "removing from directory list");

	is_top_dir = (0 == dir->depth) ? true : false;

	/*
	 * Remove the watch on this directory.
	 */
	if ((dir->wd >= 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s (%d:%d)", dir->absolute_path, "removing watch", dir->topdir->fd_inotify, dir->wd);
		if (inotify_rm_watch(dir->topdir->fd_inotify, dir->wd) != 0) {
			/*
			 * We can get EINVAL if the directory was deleted,
			 * so just ignore that.
			 */
			if (errno != EINVAL) {
				error("%s: %s", "inotify_rm_watch", strerror(errno));
			}
		}
		ds_watch_index_remove(dir->topdir, dir->wd);
		dir->wd = -1;
	}

	/*
	 * Remove all files from this directory.
	 */
	if (NULL != dir->files) {
		for (item = 0; item < dir->file_count; item++) {
			/* wipe parent field to avoid wasted work */
			dir->files[item]->parent = NULL;
			ds_file_remove(dir->files[item]);
		}
		free(dir->files);
		dir->files = NULL;
		dir->file_count = 0;
		dir->file_array_alloced = 0;
	}

	/*
	 * Remove all subdirectories from this directory (recursive).
	 */
	if (NULL != dir->subdirs) {
		/*
		 * First ensure subdirs is sorted, so the order can't change
		 * during removal.
		 */
		if (dir->subdir_count > 0 && dir->subdirs_unsorted) {
			qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
			dir->subdirs_unsorted = false;
		}
		for (item = 0; item < dir->subdir_count; item++) {
			dir->subdirs[item]->parent = NULL;
			ds_dir_remove(dir->subdirs[item]);
		}
		free(dir->subdirs);
		dir->subdirs = NULL;
		dir->subdir_count = 0;
		dir->subdir_array_alloced = 0;
	}

	/*
	 * Remove this subdirectory from our parent's directory listing, if
	 * we have a parent.  Note that when we call ourselves, above, we've
	 * wiped the parent in the subdirectory, to avoid wasted work.
	 */
	if ((NULL != dir->parent) && (NULL != dir->leaf)) {
		size_t readidx, writeidx;
		ssize_t startidx;

		startidx = ds_dir_subdir_lookup(dir->parent, dir);

		if (startidx >= 0) {
			for (readidx = (size_t) startidx + 1, writeidx = (size_t) startidx;
			     NULL != dir->parent->subdirs && readidx < dir->parent->subdir_count;
			     readidx++, writeidx++) {
				dir->parent->subdirs[writeidx] = dir->parent->subdirs[readidx];
			}
			dir->parent->subdir_count = writeidx;
		}

		/* Remove this directory from its parent directory's binary tree. */
		if (NULL != dir->parent->tree_subdirs)
			(void) tdelete(dir, &(dir->parent->tree_subdirs), ds_dir_leaf_compare);
	}

	/* Remove the directory from the change queue. */
	ds_change_queue_dir_remove(dir);

	/*
	 * Free the memory used by the pathname.
	 */
	if (NULL != dir->absolute_path) {
		free(dir->absolute_path);
		dir->absolute_path = NULL;
		dir->path = NULL;
		dir->leaf = NULL;
	}

	/* Clear out the binary trees. */
	if (NULL != dir->tree_files) {
		tdestroy(dir->tree_files, _tdestroy_noop);
		dir->tree_files = NULL;
	}
	if (NULL != dir->tree_subdirs) {
		tdestroy(dir->tree_subdirs, _tdestroy_noop);
		dir->tree_subdirs = NULL;
	}

	/* Free the directory structure itself, if it's not the top dir. */
	if (!is_top_dir)
		free(dir);
}


/*
 * Free the allocated items inside a topdir.
 */
static void ds_topdir_free_contents(ds_topdir_t topdir)
{
	if (NULL == topdir)
		return;

	/*
	 * Free the directory details.
	 */
	ds_dir_remove(&(topdir->dir));

	/*
	 * Free the watch index.
	 */
	if (NULL != topdir->watch_index) {
		free(topdir->watch_index);
		topdir->watch_index = NULL;
	}

	/*
	 * Free the change queue.
	 */
	if (NULL != topdir->change_queue) {
		free(topdir->change_queue);
		topdir->change_queue = NULL;
	}

	/*
	 * Free the changed paths list.
	 */
	if (NULL != topdir->changed_paths) {
		size_t idx;
		for (idx = 0; idx < topdir->changed_paths_length; idx++) {
			free(topdir->changed_paths[idx]);
		}
		free(topdir->changed_paths);
		topdir->changed_paths = NULL;
		topdir->changed_paths_length = 0;
		topdir->changed_paths_alloced = 0;
	}

	/*
	 * Clear out the binary trees.
	 */
	if (NULL != topdir->tree_change_queue) {
		tdestroy(topdir->tree_change_queue, _tdestroy_noop);
		topdir->tree_change_queue = NULL;
	}
	if (NULL != topdir->tree_changed_paths) {
		tdestroy(topdir->tree_changed_paths, _tdestroy_noop);
		topdir->tree_changed_paths = NULL;
	}
}


/*
 * Filter for any filename.
 *
 * Ignore anything ending in .tmp or ~ by default, or if exclude_count is
 * >0, ignore anything matching any of the patterns in excludes[].
 *
 * Returns 1 if the file should be included, 0 if it should be ignored.
 */
static int ds_filename_valid(const char *leafname)
{
	if (leafname[0] == '\0')
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '\0'))
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '.')
	    && (leafname[2] == '\0'))
		return 0;

	if (watch_dir_params->exclude_count > 0) {
		/*
		 * If given an exclusion list, use it.
		 */
		size_t eidx;
		for (eidx = 0; eidx < watch_dir_params->exclude_count; eidx++) {
			if (NULL == watch_dir_params->excludes[eidx])
				continue;
			if (fnmatch(watch_dir_params->excludes[eidx], leafname, 0) == 0)
				return 0;
		}
	} else {
		/*
		 * Default is to exclude *~ and *.tmp
		 */
		size_t namelen;
		namelen = strlen(leafname); /* flawfinder: ignore */
		/* flawfinder - caller-supplied leafname must be null-terminated. */
		if ((1 <= namelen) && (leafname[namelen - 1] == '~'))
			return 0;
		if ((namelen > 4)
		    && (strcmp(&(leafname[namelen - 4]), ".tmp") == 0))
			return 0;
	}

	return 1;
}


/*
 * Filter for scanning directories - only include filenames we should be
 * processing.
 */
static int scan_directory_filter(const struct dirent *d)
{
	return ds_filename_valid(d->d_name);
}


/*
 * Recursively scan the given directory.  Also checks files for changes. 
 * Returns nonzero if the scan failed, in which case the directory will have
 * been deleted from the lists.
 *
 * If no_recurse is true, then no subdirectories are scanned, though
 * subdirectories are still added and removed as necessary.
 */
static int ds_dir_scan( /*@dependent@ */ ds_dir_t dir, bool no_recurse)
{
	struct dirent **namelist = NULL;
	int namelist_length, itemidx, diridx, fileidx;
	struct stat dirsb;

	if (NULL == dir)
		return 1;
	if (NULL == dir->absolute_path)
		return 1;

	debug("%s (%s %d): %s", dir->absolute_path, "depth", dir->depth, "starting rescan");

	if (dir->depth > watch_dir_params->max_dir_depth) {
		debug("%s: %s", dir->absolute_path, "too deep - removing");
		ds_dir_remove(dir);
		return 1;
	}

	memset(&dirsb, 0, sizeof(dirsb));

	if (lstat(dir->absolute_path, &dirsb) != 0) {
		error("%s: %s: %s", dir->absolute_path, "lstat", strerror(errno));
		ds_dir_remove(dir);
		return 1;
	}

	/*@-compdef@ */
	/* splint thinks &namelist is not fully defined. */
	/*@-null@ */
	/* namelist starts off NULL; scandir fills it in. */
	namelist_length = scandir(dir->absolute_path, &namelist, scan_directory_filter, alphasort);
	/*@+null@ */
	/*@+compdef@ */
	if (namelist_length < 0) {
		error("%s: %s: %s", dir->absolute_path, "scandir", strerror(errno));
		ds_dir_remove(dir);
		return 1;
	}

	/*
	 * Mark all subdirectories and files as having not been seen, so we
	 * can spot which ones have been removed after we've scanned the
	 * directory.
	 */
	for (diridx = 0; NULL != dir->subdirs && diridx < (int) (dir->subdir_count); diridx++) {
		dir->subdirs[diridx]->seen_in_rescan = false;
	}
	for (fileidx = 0; NULL != dir->files && fileidx < (int) (dir->file_count); fileidx++) {
		dir->files[fileidx]->seen_in_rescan = false;
	}

	/*
	 * Go through the directory scan results, adding new items to the
	 * arrays.
	 */
	for (itemidx = 0; NULL != namelist && itemidx < namelist_length; itemidx++) {
		/*@null@ */ char *item_full_path;
		char *item_leaf;
		struct stat sb;
		int d_type;

		/*
		 * In no_file_tracking mode, if scandir()'s d_type field is
		 * populated, skip directory entries that are definitely not
		 * subdirectories so we can avoid calling lstat() on files.
		 */
		d_type = (int) (namelist[itemidx]->d_type);
		if (watch_dir_params->no_file_tracking && d_type != DT_UNKNOWN && d_type != DT_DIR) {
			free(namelist[itemidx]);
			continue;
		}

		item_full_path = NULL;
		if (asprintf(&item_full_path, "%s/%s", dir->absolute_path, namelist[itemidx]->d_name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return 1;
		}
		if (NULL == item_full_path) {
			die("%s: %s", "asprintf", "NULL");
			return 1;
		}
		item_leaf = ds_leafname(item_full_path);

		free(namelist[itemidx]);

		if ((NULL == item_leaf) || (ds_filename_valid(item_leaf) == 0)) {
			free(item_full_path);
			continue;
		}

		memset(&sb, 0, sizeof(sb));

		if (lstat(item_full_path, &sb) != 0) {
			free(item_full_path);
			continue;
		}

		if (S_ISREG((mode_t) (sb.st_mode))) {
			ds_file_t file;
			file = ds_file_add(dir, item_leaf);
			if (NULL != file)
				file->seen_in_rescan = true;
		} else if (S_ISDIR((mode_t) (sb.st_mode))) {
			ds_dir_t subdir;
			if (sb.st_dev == dirsb.st_dev) {
				subdir = ds_dir_add(dir, item_leaf);
				if (NULL != subdir)
					subdir->seen_in_rescan = true;
			} else {
				debug("%s/%s: %s", dir->absolute_path, item_leaf, "skipping - different filesystem");
			}
		}

		free(item_full_path);
	}

	free(namelist);

	/*
	 * Sort the subdirectory array if it needs to be sorted.
	 *
	 * Below, while looping through the subdirectories array, after
	 * removing a subdirectory we backtrack diridx by one since the
	 * array just got shorter.  Removing a subdir causes our subdirs
	 * array to get sorted (via ds_dir_remove -> ds_dir_subdir_lookup)
	 * so if it wasn't already in order, that will change the order of
	 * the entries, and the loop will not function as intended.
	 */
	if (NULL != dir->subdirs && dir->subdir_count > 0 && dir->subdirs_unsorted) {
		qsort(dir->subdirs, dir->subdir_count, sizeof(dir->subdirs[0]), ds_dirptr_leaf_compare);
		dir->subdirs_unsorted = false;
	}

	/*
	 * Delete any subdirectories that we did not see on rescan, and
	 * recursively scan those that we did.
	 */
	for (diridx = 0; NULL != dir->subdirs && diridx < (int) (dir->subdir_count); diridx++) {
		if (dir->subdirs[diridx]->seen_in_rescan) {
			if (no_recurse)
				continue;
			if (ds_dir_scan(dir->subdirs[diridx], false) != 0) {
				/* Go back one, as this diridx has now gone */
				diridx--;
			}
		} else {
			debug("%s[%d]: %s: %s", dir->absolute_path, diridx, dir->subdirs[diridx]->leaf,
			      "subdirectory not seen in rescan - removing");
			ds_dir_remove(dir->subdirs[diridx]);
			/* Go back one, as this diridx has now gone */
			diridx--;
		}
	}

	/*
	 * Sort the files for the same reason as above.
	 */
	if (NULL != dir->files && dir->file_count > 0 && dir->files_unsorted) {
		qsort(dir->files, dir->file_count, sizeof(dir->files[0]), ds_fileptr_leaf_compare);
		dir->files_unsorted = false;
	}

	/*
	 * Delete any files that we did not see on rescan, if we're not in
	 * no_file_tracking mode.
	 */
	for (fileidx = 0;
	     NULL != dir->files && (!watch_dir_params->no_file_tracking) && fileidx < (int) (dir->file_count);
	     fileidx++) {
		if (dir->files[fileidx]->seen_in_rescan)
			continue;
		debug("%s[%d]: %s: %s", dir->absolute_path, fileidx, dir->files[fileidx]->leaf,
		      "file not seen in rescan - removing");
		ds_file_remove(dir->files[fileidx]);
		/* Go back one, as this fileidx has now gone */
		fileidx--;
	}

	/*
	 * Check all files for changes, if we're not in no_file_tracking
	 * mode.
	 */
	for (fileidx = 0;
	     NULL != dir->files && (!watch_dir_params->no_file_tracking) && fileidx < (int) (dir->file_count);
	     fileidx++) {
		int changed;
		changed = ds_file_checkchanged(dir->files[fileidx]);
		if (changed < 0) {
			debug("%s[%d]: %s: %s", dir->absolute_path, fileidx, dir->files[fileidx]->leaf,
			      "file no longer present - removing");
			ds_file_remove(dir->files[fileidx]);
			/* Go back one, as this fileidx has now gone */
			fileidx--;
		}
	}

	/*
	 * Add an inotify watch to this directory if there isn't one
	 * already.
	 */
	if ((dir->wd < 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s", dir->absolute_path, "adding watch");
		dir->wd =
		    inotify_add_watch(dir->topdir->fd_inotify,
				      dir->absolute_path,
				      (uint32_t) (IN_CREATE | IN_DELETE | IN_MODIFY | IN_DELETE_SELF | IN_MOVED_FROM |
						  IN_MOVED_TO));
		if (dir->wd < 0) {
			error("%s: %s: %s", dir->absolute_path, "inotify_add_watch", strerror(errno));
		} else {
			/*
			 * Add this watch descriptor to the directory index,
			 * so that we can find this directory structure when
			 * an inotify event arrives.
			 */
			ds_watch_index_add(dir, dir->wd);
		}
	}

	return 0;
}


/*
 * Process queued changes until the given time or until all queue entries
 * we're ready to process have been done.
 */
static void ds_change_queue_process(ds_topdir_t topdir, time_t work_until)
{
	size_t readidx, writeidx;

	if (NULL == topdir)
		return;

	if (topdir->change_queue_length < 1)
		return;

	if (NULL == topdir->change_queue)
		return;

	debug("%s: %s: %d", "change queue: starting run, queue length", topdir->dir.absolute_path,
	      topdir->change_queue_length);

	/*
	 * The "writeidx" position is moved along whenever we skip a change
	 * queue array entry, so that at the end of the loop, the change
	 * queue contains only the skipped items, and writeidx is the number
	 * of items left.
	 *
	 * If nothing was skipped, writeidx remains at 0, so the change
	 * queue length is set to zero at the end.
	 */

	for (readidx = 0, writeidx = 0; readidx < topdir->change_queue_length; readidx++) {
		ds_change_queue_t entry;
		time_t now;

		entry = &(topdir->change_queue[readidx]);
		if ((entry->file == NULL) && (entry->dir == NULL))
			continue;

		(void) time(&now);

		/*
		 * Skip if it's not yet time for this item, or if we've
		 * reached our work_until time.
		 */
		if ((entry->when > now) || (now >= work_until)) {
			/*
			 * When skipping an item, copy it into the write
			 * position if the read position is ahead of the
			 * write; and move the write index along one place
			 * regardless.
			 */
			if (readidx > writeidx) {
				memcpy(&(topdir->change_queue[writeidx]),	/* flawfinder: ignore */
				       entry, sizeof(*entry));
				/*
				 * flawfinder rationale - we're copying
				 * array entries, all of the same size.
				 */
			}
			writeidx++;
			continue;
		}

		if (NULL != entry->file) {
			int changed;
			ds_file_t file;

			file = entry->file;

			/* Clear the change queue array entry. */
			entry->file = NULL;

			/* Remove the path from the change queue binary tree. */
			if (NULL != topdir->tree_change_queue)
				(void) tdelete(file->absolute_path, &(topdir->tree_change_queue), compare_with_strcmp);

			debug("%s: %s", file->absolute_path, "checking for changes");
			changed = ds_file_checkchanged(file);

			if (changed < 0) {
				if ((NULL != file->parent) && (NULL != file->parent->path)
				    && (!watch_dir_params->only_list_files)) {
					mark_dir_path_changed(file->parent->topdir, file->parent);
				}
				ds_file_remove(file);
			} else if (changed > 0) {
				if ((NULL != file->parent) && (NULL != file->path)
				    && (!watch_dir_params->only_list_directories)) {
					mark_file_path_changed(file->parent->topdir, file);
				}
			}
		} else if (NULL != entry->dir) {
			ds_dir_t dir;

			dir = entry->dir;

			/* Clear the change queue array entry. */
			entry->dir = NULL;

			/* Remove the path from the change queue binary tree. */
			if (NULL != topdir->tree_change_queue)
				(void) tdelete(dir->absolute_path, &(topdir->tree_change_queue), compare_with_strcmp);

			debug("%s: %s", dir->absolute_path, "triggering scan");
			(void) ds_dir_scan(dir, false);
		}
	}

	topdir->change_queue_length = writeidx;

	debug("%s: %s: %d", "change queue: run ended, queue length", topdir->dir.absolute_path,
	      topdir->change_queue_length);
}


/*
 * Process a change to a directory inside a watched directory.
 */
static void process_dir_change(struct inotify_event *event, /*@dependent@ */ ds_dir_t dir)
{
	ssize_t subdir_index;
	ds_dir_t subdir;
	inotify_action_t action;
	char *fullpath = NULL;
	struct stat sb;
	ds_dir_t newdir;

	/*
	 * Find the directory structure to which this event refers, if
	 * known.
	 */
	subdir = NULL;
	/*@-compdestroy@ */
	{
		struct ds_dir_s key;
		/* Put this in its own block so "key" has tiny scope. */
		memset(&key, 0, sizeof(key));
		key.leaf = event->name;
		subdir_index = ds_dir_subdir_lookup(dir, &key);
	}
	/*@+compdestroy@ *//* splint thinks key's other fields are used. */
	if ((subdir_index >= 0) && (NULL != dir->subdirs))
		subdir = dir->subdirs[subdir_index];

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != subdir)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != subdir)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new directory we've not seen before.
		 */

		/*
		 * Ignore the directory if it doesn't pass the filename
		 * filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		if (asprintf(&fullpath, "%s/%s", dir->absolute_path, event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the directory if it doesn't exist.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		}

		/*
		 * Ignore it if it's not a directory.
		 */
		if (!S_ISDIR((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Add the new directory and queue a scan for it.
		 */
		debug("%s: %s", fullpath, "adding new subdirectory");
		newdir = ds_dir_add(dir, event->name);
		free(fullpath);
		if (NULL != newdir)
			ds_change_queue_dir_add(newdir, 0);

		/*
		 * Mark this as a changed path.
		 */
		if (NULL != newdir && !watch_dir_params->only_list_files) {
			mark_dir_path_changed(dir->topdir, newdir);
		}

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a directory we've seen before, so queue a rescan for
		 * it.
		 */
		if (NULL != subdir) {
			debug("%s: %s", subdir->absolute_path, "queueing rescan");
			ds_change_queue_dir_add(subdir, 0);
		}
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this directory before, delete its
		 * structure.
		 */
		if (NULL != subdir) {
			debug("%s: %s", subdir->absolute_path, "triggering removal");
			ds_dir_remove(subdir);
		}
		/*
		 * Mark the parent directory as a changed path.
		 */
		if ((NULL != dir->path) && (!watch_dir_params->only_list_files)) {
			mark_dir_path_changed(dir->topdir, dir);
		}
		break;
	}
}


/*
 * Process a change to a file inside a watched directory.
 */
static void process_file_change(struct inotify_event *event, /*@dependent@ */ ds_dir_t dir)
{
	ssize_t file_index;
	ds_file_t file;
	inotify_action_t action;
	char *fullpath;
	struct stat sb;
	ds_file_t newfile;

	/*
	 * Find the file structure to which this event refers, if known.
	 */
	file = NULL;
	/*@-compdestroy@ */
	{
		struct ds_file_s key;
		/* Put this in its own block so "key" has tiny scope. */
		memset(&key, 0, sizeof(key));
		key.leaf = event->name;
		file_index = ds_dir_file_lookup(dir, &key);
	}
	/*@+compdestroy@ *//* splint thinks key's other fields are used. */
	if ((file_index >= 0) && (NULL != dir->files))
		file = dir->files[file_index];

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != file)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != file)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new file we've not seen before.
		 */

		/*
		 * Ignore the file if it doesn't pass the filename filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		fullpath = NULL;
		if (asprintf(&fullpath, "%s/%s", dir->absolute_path, event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the file if it doesn't exist or it isn't a regular
		 * file.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		} else if (!S_ISREG((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Add the new file and queue it to be checked.
		 */
		debug("%s: %s", fullpath, "adding new file");
		newfile = ds_file_add(dir, event->name);
		if (NULL != newfile)
			ds_change_queue_file_add(newfile, 0);

		free(fullpath);

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a file we've seen before, so queue a check for it.
		 */
		if (NULL != file)
			ds_change_queue_file_add(file, 0);
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this file before, delete its structure.
		 */
		if (NULL != file)
			debug("%s: %s", file->absolute_path, "triggering removal");
		/*
		 * Mark the parent directory as a changed path.
		 */
		if (NULL != file && NULL != file->parent && !watch_dir_params->only_list_files) {
			mark_dir_path_changed(file->parent->topdir, file->parent);
		}
		if (NULL != file)
			ds_file_remove(file);
		break;
	}
}


/*
 * Process incoming inotify events.
 *
 * If the top directory is removed, it is not freed, but all its values are
 * nullified.
 */
static void process_inotify_events(ds_topdir_t topdir)
{
	unsigned char readbuf[8192];	 /* flawfinder: ignore */
	ssize_t got, pos;

	/*
	 * flawfinder rationale: readbuf is populated by read() which is
	 * passed readbuf's size.
	 */

	if (NULL == topdir)
		return;
	if (topdir->fd_inotify < 0)
		return;

	memset(readbuf, 0, sizeof(readbuf));

	/*
	 * Read as many events as we can.
	 */
	got = read(topdir->fd_inotify, readbuf, sizeof(readbuf));	/* flawfinder: ignore */
	/* flawfinder - read() called only once, bounded to buffer size. */
	if (got <= 0) {
		error("%s: %s: (%d): %s", topdir->dir.absolute_path, "inotify read event", got, strerror(errno));
		(void) close(topdir->fd_inotify);
		topdir->fd_inotify = -1;
		return;
	}
	/* This shouldn't be possible, but check anyway. */
	if (got > (ssize_t) sizeof(readbuf))
		got = (ssize_t) sizeof(readbuf);

	/*
	 * Process each event that we've read.
	 */
	for (pos = 0; pos < got;) {
		struct inotify_event *event;
		/*@null@ */ ds_dir_t dir = NULL;

		event = (struct inotify_event *) &(readbuf[pos]);
		dir = ds_watch_index_lookup(topdir, event->wd, NULL);

#if ENABLE_DEBUGGING
		/*@-branchstate@ */
		/* splint doesn't like "p" being conditional. */
		if (debugging_enabled) {
			char flags[1024];	/* flawfinder: ignore */
			char *end;
			char *p;

			/*
			 * flawfinder - flags[] is large enough, initialised
			 * with a null terminator, and used with stpecpy()
			 * which also guards against overflow and ensures
			 * null termination.
			 */

			end = flags + sizeof(flags);
			flags[0] = '\0';
			p = flags;

			if (0 != (event->mask & IN_ACCESS))
				p = stpecpy(p, end, " IN_ACCESS");
			if (0 != (event->mask & IN_ATTRIB))
				p = stpecpy(p, end, " IN_ATTRIB");
			if (0 != (event->mask & IN_CLOSE_WRITE))
				p = stpecpy(p, end, " IN_CLOSE_WRITE");
			if (0 != (event->mask & IN_CLOSE_NOWRITE))
				p = stpecpy(p, end, " IN_CLOSE_NOWRITE");
			if (0 != (event->mask & IN_CREATE))
				p = stpecpy(p, end, " IN_CREATE");
			if (0 != (event->mask & IN_DELETE))
				p = stpecpy(p, end, " IN_DELETE");
			if (0 != (event->mask & IN_DELETE_SELF))
				p = stpecpy(p, end, " IN_DELETE_SELF");
			if (0 != (event->mask & IN_MODIFY))
				p = stpecpy(p, end, " IN_MODIFY");
			if (0 != (event->mask & IN_MOVE_SELF))
				p = stpecpy(p, end, " IN_MOVE_SELF");
			if (0 != (event->mask & IN_MOVED_FROM))
				p = stpecpy(p, end, " IN_MOVED_FROM");
			if (0 != (event->mask & IN_MOVED_TO))
				p = stpecpy(p, end, " IN_MOVED_TO");
			if (0 != (event->mask & IN_OPEN))
				p = stpecpy(p, end, " IN_OPEN");
			if (0 != (event->mask & IN_IGNORED))
				p = stpecpy(p, end, " IN_IGNORED");
			if (0 != (event->mask & IN_ISDIR))
				p = stpecpy(p, end, " IN_ISDIR");
			if (0 != (event->mask & IN_Q_OVERFLOW))
				p = stpecpy(p, end, " IN_Q_OVERFLOW");
			if (0 != (event->mask & IN_UNMOUNT))
				p = stpecpy(p, end, " IN_UNMOUNT");
			if (event->len > 0) {
				debug("%s: %d: %s: %.*s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : dir->absolute_path, event->len, event->name, flags);
			} else {
				debug("%s: %d: %s: %s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : dir->absolute_path, "(none)", flags);
			}
		}
		/*@+branchstate@ */
#endif				/* ENABLE_DEBUGGING */

		pos += sizeof(*event) + event->len;

		/*
		 * There's nothing we can do if we don't know which
		 * directory it was.
		 */
		if (NULL == dir)
			continue;

		/*
		 * If the directory is itself being deleted, remove it.
		 */
		if (0 != (event->mask & IN_DELETE_SELF)) {
			/*
			 * Before removing it, mark its parent directory as
			 * a changed path.
			 */
			if ((NULL != dir->parent) && (NULL != dir->parent->path)
			    && (!watch_dir_params->only_list_files)) {
				mark_dir_path_changed(dir->topdir, dir->parent);
			}
			ds_dir_remove(dir);
			continue;
		}

		/*
		 * If this isn't an event about a named thing in this
		 * directory, we can't do anything.
		 */
		if (event->len < 1)
			continue;
		if ('\0' == event->name[0])
			continue;

		if (0 != (event->mask & IN_ISDIR)) {
			process_dir_change(event, dir);
		} else {
			process_file_change(event, dir);
		}
	}
}


/*
 * Add a path to the list of changed paths.
 */
static void _mark_path_changed( /*@null@ */ ds_topdir_t topdir, const char *path, bool isdir)
{
	char *savepath = NULL;

	if (NULL == topdir)
		return;

	if (NULL == path)
		return;

	if (asprintf(&savepath, "%s%s", path, isdir ? "/" : "") < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}
	assert(NULL != savepath);
	if (NULL == savepath)
		return;

	/*
	 * Check the path isn't already listed - don't list it twice.
	 */
	/*@-mustfreefresh@ */
	if (NULL != topdir->tree_changed_paths) {
		nullable_string result;
		result = tfind(savepath, &(topdir->tree_changed_paths), compare_with_strcmp);
		if (NULL != result) {
			debug("%s: %s", savepath, "path is already in the changed paths list");
			free(savepath);
			return;
		}
	}
	/*@+mustfreefresh@ */
	/* splint thinks tfind() allocates "result", but it doesn't. */

	/*
	 * Extend the array if necessary.
	 */
	if (topdir->changed_paths_length >= topdir->changed_paths_alloced) {
		if (!extend_array
		    ((void **) (&(topdir->changed_paths)), &(topdir->changed_paths_alloced),
		     sizeof(topdir->changed_paths[0]), topdir->changed_paths_alloced + CHANGEDPATH_ALLOC_CHUNK))
			return;
	}
	assert(NULL != topdir->changed_paths);
	if (NULL == topdir->changed_paths)
		return;

	debug("%s: %s", "adding to changed paths", savepath);

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	topdir->changed_paths[topdir->changed_paths_length] = savepath;
	topdir->changed_paths_length++;

	/* Add the changed path to the binary tree. */
	/*@-nullstate@ */
	(void) tsearch(savepath, &(topdir->tree_changed_paths), compare_with_strcmp);
	/*@+nullstate@ *//* topdir->tree_changed_paths is OK to be NULL here. */
}


/*
 * Add a file's path to the list of changed paths, using the absolute_paths
 * parameter to determine whether to add its absolute path or its path
 * relative to the topdir.
 */
static void mark_file_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_file_t file)
{
	const char *path_to_add;

	if (NULL == topdir)
		return;
	if (NULL == file)
		return;

	path_to_add = watch_dir_params->absolute_paths ? file->absolute_path : file->path;
	_mark_path_changed(topdir, path_to_add, false);
}


/*
 * Add a directory's path to the list of changed paths, using the
 * absolute_paths parameter to determine whether to add its absolute path or
 * its path relative to the topdir.
 */
static void mark_dir_path_changed( /*@null@ */ ds_topdir_t topdir, /*@dependent@ */ ds_dir_t dir)
{
	const char *path_to_add;

	if (NULL == topdir)
		return;
	if (NULL == dir)
		return;

	path_to_add = watch_dir_params->absolute_paths ? dir->absolute_path : dir->path;
	_mark_path_changed(topdir, path_to_add, true);
}


/*
 * Write out a new file containing the current changed paths list, and clear
 * the list.
 */
static void dump_changed_paths(ds_topdir_t topdir, const char *savedir)
{
	char *savefile = NULL;
	char *temp_filename = NULL;
	struct tm *tm;
	time_t t;
	int tmpfd;
	FILE *fptr;
	size_t idx;
	char delimiter;

	if (NULL == topdir->changed_paths)
		return;
	if (topdir->changed_paths_length < 1)
		return;

	delimiter = '\n';
	if (watch_dir_params->null_delimiter)
		delimiter = '\0';

	t = time(NULL);
	tm = localtime(&t);

	if (asprintf
	    (&savefile, "%s/%04d%02d%02d-%02d%02d%02d.%d%s", savedir,
	     tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec, getpid(),
	     NULL == watch_dir_params->changefile_suffix ? "" : watch_dir_params->changefile_suffix) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}
	if (NULL == savefile) {
		die("%s: %s", "asprintf", "null");
		return;
	}

	tmpfd = ds_tmpfile(savefile, &temp_filename);
	if ((tmpfd < 0) || (NULL == temp_filename)) {
		free(savefile);
		return;
	}

	fptr = fdopen(tmpfd, "w");
	if (NULL == fptr) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		fprintf(fptr, "%s%c", topdir->changed_paths[idx], delimiter);
	}

	if (0 != fclose(fptr)) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	if (rename(temp_filename, savefile) != 0) {
		error("%s: %s", savefile, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	free(temp_filename);
	free(savefile);

	if (NULL != topdir->tree_changed_paths) {
		tdestroy(topdir->tree_changed_paths, _tdestroy_noop);
		topdir->tree_changed_paths = NULL;
	}

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		free(topdir->changed_paths[idx]);
	}

	free(topdir->changed_paths);
	topdir->changed_paths = NULL;
	topdir->changed_paths_length = 0;
	topdir->changed_paths_alloced = 0;
}


/*
 * Handler for an exit signal such as SIGTERM - set a flag to trigger an
 * exit.
 */
static void watch_dir_exitsignal( /*@unused@ */  __attribute__((unused))
				 int signum)
{
	watch_dir_exit_now = true;
}


/*
 * Main entry point.  Set everything up and enter the main loop, which does
 * the following:
 *
 *   - A periodic rescan from the top level directory down.
 *   - Processing of inotify events from all known directories.
 *   - Processing of the change queue generated from the above two.
 *   - Periodic output of a file listing updated paths.
 *
 * Scanned directories are watched using inotify, so that changes to files
 * within it can be noticed immediately.
 *
 * A change queue is maintained, comprising a list of files and directories
 * to re-check, and the time at which to do so.  This is so that when
 * multiple files are changed, or the same file is changed several times, it
 * can be dealt with intelligently - they are pushed on to the change queue,
 * with duplicates being ignored, and then the change queue is processed in
 * chunks to avoid starvation caused by inotify events from one file
 * changing rapidly.
 */
int watch_dir(struct watch_dir_params_s *params)
{
	int *inotify_fds;		 /* fds to watch for inotify on */
	time_t next_full_scan;		 /* when to run next full scan */
	/*@null@ */ ds_topdir_t *topdirs;
	/* top-level directory contents */
	time_t next_change_queue_run;	 /* when to next run changes */
	time_t next_changedpath_dump;	 /* when to next dump changed paths */
	size_t dir_idx;

	watch_dir_params = params;

	/*
	 * Set up the signal handlers.
	 */
	/*
	 * splint sees sa.sa_mask as leaking memory, so put this code in its
	 * own scope so that compdestroy can be turned off for as short a
	 * time as possible.
	 */
	/*@-compdestroy@ */
	{
		struct sigaction sa;

		memset(&sa, 0, sizeof(sa));

		sa.sa_handler = watch_dir_exitsignal;
		(void) sigemptyset(&(sa.sa_mask));
		sa.sa_flags = 0;
		(void) sigaction(SIGTERM, &sa, NULL);

		sa.sa_handler = watch_dir_exitsignal;
		(void) sigemptyset(&(sa.sa_mask));
		sa.sa_flags = 0;
		(void) sigaction(SIGINT, &sa, NULL);
	}
	/*@+compdestroy@ */

	/*
	 * Create the inotify event queues.
	 *
	 * We use a separate event queue for each top-level directory so
	 * that when an event comes in, we can associate it with the
	 * appropriate top-level.
	 */
	inotify_fds = calloc(params->toplevel_path_count, sizeof(int));
	if (NULL == inotify_fds) {
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		inotify_fds[dir_idx] = inotify_init();
		if (inotify_fds[dir_idx] < 0) {
			error("%s: %s", "inotify", strerror(errno));
			free(inotify_fds);
			return EXIT_FAILURE;
		}
	}

	/*
	 * Create the top-level directory memory structures.
	 */
	topdirs = calloc(params->toplevel_path_count, sizeof(*topdirs));
	if (NULL == topdirs) {
		free(inotify_fds);
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		topdirs[dir_idx] = ds_dir_toplevel(inotify_fds[dir_idx], params->toplevel_paths[dir_idx]);
		if (NULL == topdirs[dir_idx]) {
			free(topdirs);
			free(inotify_fds);
			return EXIT_FAILURE;
		}
	}

	/*
	 * Enter the main loop.
	 */

	next_change_queue_run = 0;
	next_full_scan = 0;
	next_changedpath_dump = 0;

	while (!watch_dir_exit_now) {
		time_t now;
		fd_set readfds;
		struct timeval timeout;
		int ready, fd_max;

		/*
		 * Wait for an inotify event on any of the queues.
		 */
		memset(&readfds, 0, sizeof(readfds));
#if SPLINT
		/* splint has trouble with FD_ZERO. */
#else
		FD_ZERO(&readfds);
#endif
		fd_max = 0;
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			/*@-shiftnegative@ */
			FD_SET(fd_inotify, &readfds);
			/*@+shiftnegative@ *//* splint issues with FD_SET. */
			if (fd_inotify > fd_max)
				fd_max = fd_inotify;
		}

		timeout.tv_sec = 0;
		timeout.tv_usec = 100000;
		/*@-nullpass@ */
		ready = select(1 + fd_max, &readfds, NULL, NULL, &timeout);
		/*@+nullpass@ *//* passing NULL is OK here. */

		if (ready < 0) {
			if (errno != EINTR)
				error("%s: %s", "select", strerror(errno));
			watch_dir_exit_now = true;
			break;
		}

		/*
		 * Process new inotify events.
		 */
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && ready > 0 && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			/*@-shiftnegative@ */
			if (!FD_ISSET(fd_inotify, &readfds))
				continue;
			/*@+shiftnegative@ *//* splint issues with FD_ISSET. */

			process_inotify_events(topdirs[dir_idx]);
			if (NULL == topdirs[dir_idx]->dir.absolute_path) {
				error("%s: %s", params->toplevel_paths[dir_idx], "top directory removed");
				watch_dir_exit_now = true;
				break;
			}
		}

		if (watch_dir_exit_now)
			break;

		if (0 == ready) {
			timeout.tv_sec = 0;
			timeout.tv_usec = 100000;
			/*@-nullpass@ */
			(void) select(0, NULL, NULL, NULL, &timeout);
			/*@+nullpass@ *//* passing NULL is OK here. */
		}

		(void) time(&now);

		/*
		 * Do a full scan periodically.
		 */
		if (now >= next_full_scan) {
			next_full_scan = now + params->full_scan_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_dir_add(&(topdirs[dir_idx]->dir), 0);
			}
		}

		/*
		 * Run our change queue.
		 */
		if (now >= next_change_queue_run) {
			next_change_queue_run = now + params->queue_run_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_process(topdirs[dir_idx], now + params->queue_run_max_seconds);
			}
		}

		/*
		 * Dump our list of changed paths.
		 */
		if (now >= next_changedpath_dump) {
			next_changedpath_dump = now + params->changedpath_dump_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				dump_changed_paths(topdirs[dir_idx], params->changedpath_dir);
			}
		}
	}

	/*@-compdestroy@ */
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		ds_topdir_free_contents(topdirs[dir_idx]);
		(void) free(topdirs[dir_idx]);
		if (inotify_fds[dir_idx] >= 0) {
			if (0 != close(inotify_fds[dir_idx]))
				error("%s: %s", "close", strerror(errno));
		}
	}
	/*@+compdestroy@ */
	/*
	 * splint doesn't know that ds_topdir_free_contents() frees the
	 * internals of a topdir.
	 */
	(void) free(topdirs);
	(void) free(inotify_fds);

	return EXIT_SUCCESS;
}
