/*
 * Watch the given directory, maintaining an output file containing a list
 * of files changed.
 *
 * Copyright 2014, 2021, 2023, 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

/* File and subdirectory array allocation chunk size */
#define DIRCONTENTS_ALLOC_CHUNK	128

/* Directory index array allocation chunk size */
#define DIR_INDEX_ALLOC_CHUNK 1024

/* Change queue allocation chunk size */
#define CHANGE_QUEUE_ALLOC_CHUNK 1024

/* Changed paths list allocation chunk size */
#define CHANGEDPATH_ALLOC_CHUNK 1024

/* Macro to choose absolute or relative paths in the change queue */
#define CHANGED_PATH(x) watch_dir_params->absolute_paths ? x->absolute_path : x->path

#define _ATFILE_SOURCE

#include "config.h"
#include "common.h"
#include "watch.h"

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdint.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#ifndef SPLINT
/* splint 3.1.2 chokes on syslog.h */
#include <syslog.h>
#endif
#include <limits.h>
#include <sys/inotify.h>
#include <sys/select.h>
#include <poll.h>
#include <fnmatch.h>


/*
 * Actions to take on inotify events.
 */
typedef enum {
	IN_ACTION_CREATE,
	IN_ACTION_UPDATE,
	IN_ACTION_DELETE,
	IN_ACTION_NONE
} inotify_action_t;

struct ds_file_s;
typedef struct ds_file_s *ds_file_t;
struct ds_dir_s;
typedef struct ds_dir_s *ds_dir_t;
struct ds_watch_index_s;
typedef struct ds_watch_index_s *ds_watch_index_t;
struct ds_change_queue_s;
typedef struct ds_change_queue_s *ds_change_queue_t;


/*
 * Structure holding information about a file.  An example path would be
 * "0/12/12345/foo.txt"; the leaf would be "foo.txt"; the absolute_path
 * would be "/top/dir/0/12/12345/foo.txt" if the top level directory was
 * "/top/dir".
 *
 * The "absolute_path" is malloc()ed; "path" and "leaf" point within it, so
 * aren't separately allocated.
 */
struct ds_file_s {
	char *absolute_path;		 /* absolute path to file */
	/*@null@ *//*@dependent@ */ char *path;
	/* path relative to top level */
	/*@null@ *//*@dependent@ */ char *leaf;
	/* leafname of this file */
	time_t mtime;			 /* file last-modification time */
	off_t size;			 /* file size */
	/*@null@ *//*@dependent@ */ ds_dir_t parent;
	/* containing directory */
	bool seen_in_rescan;		 /* set during dir rescan */
};


/*
 * Structure holding information about a directory.  An example path would
 * be "0/12/12345" with a leaf of "12345".
 *
 * The "absolute_path" is malloc()ed; "path" and "leaf" point within it, so
 * aren't separately allocated.
 */
struct ds_dir_s {
	char *absolute_path;		 /* absolute path to directory */
	/*@null@ *//*@dependent@ */ char *path;
	/* path relative to top level */
	/*@null@ *//*@dependent@ */ char *leaf;
	/* leafname of this directory */
	int wd;				 /* inotify watch fd (if dir) */
	int depth;			 /* subdirs deep from top level */
	size_t file_count;		 /* number of files in directory */
	size_t subdir_count;		 /* number of immediate subdirs */
	ds_file_t *files;		 /* array of files */
	ds_dir_t *subdirs;		 /* array of subdirectories */
	size_t file_array_alloced;	 /* file entries allocated */
	size_t subdir_array_alloced;	 /* subdir entries allocated */
	/*@null@ *//*@dependent@ */ ds_dir_t parent;
	/* pointer to parent directory */
	/*@null@ *//*@dependent@ */ ds_dir_t topdir;
	/* pointer to top directory */
	bool seen_in_rescan;		 /* set during dir rescan */
	bool files_unsorted;		 /* set when files need re-sorting */
	bool subdirs_unsorted;		 /* set when dirs need re-sorting */
	/*
	 * Items used only in the top level directory:
	 */
	int fd_inotify;			 /* directory watch file descriptor */
	ds_watch_index_t watch_index;	 /* array of watch descriptors */
	size_t watch_index_length;	 /* number of entries in array */
	size_t watch_index_alloced;	 /* number of entries allocated */
	bool watch_index_unsorted;	 /* set if array needs sorting */
	ds_change_queue_t change_queue;	 /* array of changes needed */
	size_t change_queue_length;	 /* number of changes in queue */
	size_t change_queue_alloced;	 /* array size allocated */
	char **changed_paths;		 /* array of changed paths */
	size_t changed_paths_length;	 /* number of paths in array */
	size_t changed_paths_alloced;	 /* array size allocated */
};


/*
 * Structure for indexing directory structures by watch identifier.
 */
struct ds_watch_index_s {
	int wd;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


/*
 * Structure for a file check or directory scan queue entry.
 */
struct ds_change_queue_s {
	time_t when;
	/*@null@ *//*@dependent@ */ ds_file_t file;
	/*@null@ *//*@dependent@ */ ds_dir_t dir;
};


static int ds_filename_valid(const char *name);

static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add(ds_dir_t dir, const char *name);
static void ds_file_remove(ds_file_t file);
static int ds_file_checkchanged(ds_file_t file);

static /*@null@ */ ds_dir_t ds_dir_toplevel(int fd_inotify, const char *top_path);
static /*@null@ */ ds_dir_t ds_dir_add(ds_dir_t dir, const char *name);
static void ds_dir_remove(ds_dir_t dir);
static int ds_dir_scan(ds_dir_t dir, bool no_recurse);

static void ds_watch_index_add(ds_dir_t dir, int wd);
static void ds_watch_index_remove(ds_dir_t topdir, int wd);
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_dir_t topdir, int wd);

static void ds_change_queue_file_add(ds_file_t file, time_t when);
static void ds_change_queue_file_remove(ds_file_t file);
static void ds_change_queue_dir_add(ds_dir_t dir, time_t when);
static void ds_change_queue_dir_remove(ds_dir_t dir);

static void ds_change_queue_process(ds_dir_t topdir, time_t work_until);

static void mark_path_changed( /*@null@ */ ds_dir_t topdir, const char *path, bool isdir);
static void dump_changed_paths(ds_dir_t topdir, const char *changedpath_dir);


static struct watch_dir_params_s *watch_dir_params;
static bool watch_dir_exit_now = false;


/*@-compdestroy @ */
/*@-compmempass @ */
/*@-usereleased @ */
/*@-branchstate @ */
/*@-temptrans @ */
/*@-compdef @ */

/*
 * splint: there doesn't seem to be a way to extend the watch_index array
 * with realloc() without causing warnings.
 */


/*
 * Add the given watch descriptor to the directory index.
 */
static void ds_watch_index_add(ds_dir_t dir, int wd)
{
	if (NULL == dir)
		return;
	if (wd < 0)
		return;
	if (NULL == dir->topdir)
		return;

	/*
	 * Extend the array if we need to.
	 */
	if (dir->topdir->watch_index_length >= dir->topdir->watch_index_alloced) {
		size_t new_size;
		ds_watch_index_t newptr;

		new_size = dir->topdir->watch_index_alloced + DIR_INDEX_ALLOC_CHUNK;
		newptr = realloc(dir->topdir->watch_index, new_size * sizeof(dir->topdir->watch_index[0]));
		if (NULL == newptr) {
			die("%s: %s", "realloc", strerror(errno));
			return;
		}
		dir->topdir->watch_index = newptr;
		dir->topdir->watch_index_alloced = new_size;
	}

	memset(&(dir->topdir->watch_index[dir->topdir->watch_index_length]), 0, sizeof(dir->topdir->watch_index[0]));

	dir->topdir->watch_index[dir->topdir->watch_index_length].wd = wd;
	dir->topdir->watch_index[dir->topdir->watch_index_length].dir = dir;

	dir->topdir->watch_index_length++;
	dir->topdir->watch_index_unsorted = true;
}

/*@+compdestroy @ */
/*@+compmempass @ */
/*@+usereleased @ */
/*@+branchstate @ */
/*@+temptrans @ */
/*@+compdef @ */


/*
 * Remove the given watch descriptor from the directory index.
 */
static void ds_watch_index_remove(ds_dir_t topdir, int wd)
{
	size_t readidx, writeidx;

	if (NULL == topdir)
		return;

	for (readidx = 0, writeidx = 0; readidx < topdir->watch_index_length; readidx++) {
		if (topdir->watch_index[readidx].wd == wd) {
			continue;
		}
		if (readidx != writeidx) {
			topdir->watch_index[writeidx] = topdir->watch_index[readidx];
		}
		writeidx++;
	}
	topdir->watch_index_length = writeidx;
	topdir->watch_index_unsorted = true;
}


/*
 * Comparison function for the directory index.
 */
static int ds_watch_index_compare(const void *a, const void *b)
{
	if (((ds_watch_index_t) a)->wd < ((ds_watch_index_t) b)->wd)
		return -1;
	if (((ds_watch_index_t) a)->wd > ((ds_watch_index_t) b)->wd)
		return 1;
	return 0;
}


/*
 * Return the directory structure associated with the given watch
 * descriptor, or NULL if none.
 */
static
/*@null@ */
/*@dependent@ */
ds_dir_t ds_watch_index_lookup(ds_dir_t topdir, int wd)
{
	struct ds_watch_index_s key;
	/*@null@ */ ds_watch_index_t result;

	if (NULL == topdir)
		return NULL;
	if (NULL == topdir->watch_index)
		return NULL;

	if ((topdir->watch_index_unsorted)
	    && (topdir->watch_index_length > 0)) {
		qsort(topdir->watch_index, topdir->watch_index_length,
		      sizeof(topdir->watch_index[0]), ds_watch_index_compare);
		topdir->watch_index_unsorted = false;
	}

	key.wd = wd;
	result =
	    bsearch(&key, topdir->watch_index, topdir->watch_index_length,
		    sizeof(topdir->watch_index[0]), ds_watch_index_compare);

	if (NULL == result)
		return NULL;

	return result->dir;
}

/*@-usereleased@ */
/*@-branchstate@ */
/*@-temptrans@ */
/*@-compmempass@ */
/*@-unqualifiedtrans@ */
/*@-compdef@ */

/* splint: realloc() again, as above. */

/*
 * Add an entry to the change queue.
 */
static void _ds_change_queue_add( /*@null@ */ ds_dir_t topdir, time_t when, /*@null@ */ ds_file_t file,
				 /*@null@ */ ds_dir_t dir)
{
	size_t idx;

	if (NULL == topdir)
		return;

	if ((NULL == file) && (NULL == dir))
		return;

	/*
	 * Check the change isn't already queued - don't queue it twice.
	 */
	for (idx = 0; idx < topdir->change_queue_length; idx++) {
		if (NULL != file) {
			if (topdir->change_queue[idx].file == file)
				return;
		}
		if (NULL != dir) {
			if (topdir->change_queue[idx].dir == dir)
				return;
		}
	}

	/*
	 * Extend the array if necessary.
	 */
	if (topdir->change_queue_length >= topdir->change_queue_alloced) {
		size_t new_size;
		ds_change_queue_t newptr;
		new_size = topdir->change_queue_alloced + CHANGE_QUEUE_ALLOC_CHUNK;
		newptr = realloc(topdir->change_queue, new_size * sizeof(topdir->change_queue[0]));
		if (NULL == newptr) {
			die("%s: %s", "realloc", strerror(errno));
			return;
		}
		topdir->change_queue = newptr;
		topdir->change_queue_alloced = new_size;
	}

	if (NULL != file) {
		debug("%s: %s: %s", "adding to change queue", "check file", file->path);
	} else if (NULL != dir) {
		debug("%s: %s: %s", "adding to change queue", "scan directory", dir->path);
	}

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	topdir->change_queue[topdir->change_queue_length].when = when;
	topdir->change_queue[topdir->change_queue_length].file = file;
	topdir->change_queue[topdir->change_queue_length].dir = dir;

	topdir->change_queue_length++;
}

/*@+usereleased@ */
/*@+branchstate@ */
/*@+temptrans@ */
/*@+compmempass@ */
/*@+unqualifiedtrans@ */
/*@+compdef@ */


/*
 * Queue a file check.
 */
static void ds_change_queue_file_add(ds_file_t file, time_t when)
{
	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;

	if (0 == when) {
		/*
		 * If no specific time was given, choose a time that's a
		 * couple of seconds in the future so it's more likely it's
		 * finished being changed before we take any action.
		 *
		 * The larger the file is, the longer we give it.
		 */
		if (file->size < 102400) {  /* < 100KiB */
			when = time(NULL) + 2;
		} else if (file->size < 1048576) {	/* < 1MiB */
			when = time(NULL) + 3;
		} else if (file->size < 10485760) {	/* < 10MiB */
			when = time(NULL) + 4;
		} else if (file->size < 104857600) {	/* < 100MiB */
			when = time(NULL) + 5;
		} else {		    /* >= 100 MiB */
			when = time(NULL) + 6;
		}
	}

	_ds_change_queue_add(file->parent->topdir, when, file, NULL);
}


/*
 * Remove a file from the change queue.
 */
static void ds_change_queue_file_remove(ds_file_t file)
{
	ds_dir_t topdir;
	size_t idx;

	if (NULL == file)
		return;
	if (NULL == file->parent)
		return;
	if (NULL == file->parent->topdir)
		return;

	topdir = file->parent->topdir;

	for (idx = 0; idx < topdir->change_queue_length; idx++) {
		if (topdir->change_queue[idx].file == file) {
			topdir->change_queue[idx].file = NULL;
		}
	}
}


/*
 * Queue a directory scan.
 */
static void ds_change_queue_dir_add(ds_dir_t dir, time_t when)
{
	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;
	if (0 == when)
		when = time(NULL);
	_ds_change_queue_add(dir->topdir, when, NULL, dir);
}


/*
 * Remove a directory from the change queue.
 */
static void ds_change_queue_dir_remove(ds_dir_t dir)
{
	size_t idx;

	if (NULL == dir)
		return;
	if (NULL == dir->topdir)
		return;

	for (idx = 0; idx < dir->topdir->change_queue_length; idx++) {
		if (dir->topdir->change_queue[idx].dir == dir) {
			dir->topdir->change_queue[idx].dir = NULL;
		}
	}
}


/*@-usereleased@ */
/*@-branchstate@ */
/*@-temptrans@ */
/*@-compmempass@ */
/*@-unqualifiedtrans@ */
/*@-compdef@ */
/*@-compdestroy@ */

/* splint: realloc() again, as above. */

/*
 * Add a file to the list of files in the given directory; if the file is
 * already in the list, return the existing file.
 *
 * The "name" string should contain the name of the file relative to the
 * directory (not the full path), e.g. "somefile".
 *
 * Returns the file, or NULL on error.
 */
static
/*@null@ */
/*@dependent@ */
ds_file_t ds_file_add(ds_dir_t dir, const char *name)
{
	ds_file_t file;
	size_t idx;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;
	if (NULL == dir->absolute_path)
		return NULL;


	/*
	 * Check we don't already have this file in this directory - if we
	 * do, return the existing file.
	 */
	for (idx = 0; idx < dir->file_count; idx++) {
		if ((NULL != dir->files[idx]->leaf) && (strcmp(dir->files[idx]->leaf, name) == 0)) {
			return dir->files[idx];
		}
	}

	/*
	 * Extend the file array in the directory structure if we need to.
	 */
	if (dir->file_count >= dir->file_array_alloced) {
		size_t target_array_alloced;
		void *newptr;

		target_array_alloced = dir->file_array_alloced;
		while (target_array_alloced <= dir->file_count) {
			target_array_alloced += DIRCONTENTS_ALLOC_CHUNK;
		}
		newptr = realloc((void *) (dir->files), target_array_alloced * sizeof(dir->files[0]));
		if (NULL == newptr) {
			die("%s: %s", "realloc", strerror(errno));
			return NULL;
		}
		dir->files = newptr;
		dir->file_array_alloced = target_array_alloced;
	}

	/*
	 * Allocate a new file structure.
	 */
	file = calloc(1, sizeof(*file));
	if (NULL == file) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the file structure.
	 */
	/*@-unrecog@ */
	/* splint doesn't know about asprintf() */
	if (asprintf(&(file->absolute_path), "%s/%s", dir->absolute_path, name) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		free(file);
		return NULL;
	}
	/*@+unrecog@ */
	if (NULL != dir->topdir) {
		file->path = &(file->absolute_path[strlen(dir->topdir->absolute_path) + 1]);
	} else {
		file->path = &(file->absolute_path[strlen(dir->absolute_path) + 1]);
	}
	file->leaf = ds_leafname(file->absolute_path);
	file->parent = dir;
	file->seen_in_rescan = false;

	/*
	 * Add the file to the directory structure, and mark the list as
	 * unsorted.
	 */
	dir->files[dir->file_count] = file;
	dir->file_count++;
	dir->files_unsorted = true;

	return file;
}

/*@+usereleased@ */
/*@+branchstate@ */
/*@+temptrans@ */
/*@+compmempass@ */
/*@+unqualifiedtrans@ */
/*@+compdef@ */
/*@+compdestroy@ */


/*
 * Free a file information structure and its contents.
 *
 * The parent directory's "files" list is updated to remove this file from
 * it.
 */
static void ds_file_remove(ds_file_t file)
{
	if (NULL == file)
		return;
	if (NULL == file->path)
		return;

	/*
	 * Remove this file from our parent directory's file listing, if we
	 * have a parent.
	 */
	if ((NULL != file->parent) && (NULL != file->leaf)) {
		size_t readidx, writeidx;
		for (readidx = 0, writeidx = 0; readidx < file->parent->file_count; readidx++) {
			if ((NULL != file->parent->files[readidx]->leaf)
			    && (strcmp(file->parent->files[readidx]->leaf, file->leaf) == 0)
			    ) {
				continue;
			}
			if (readidx != writeidx) {
				file->parent->files[writeidx] = file->parent->files[readidx];
			}
			writeidx++;
		}
		file->parent->file_count = writeidx;
		file->parent->files_unsorted = true;
	}

	/* Remove the file from the change queue. */
	ds_change_queue_file_remove(file);

	/*
	 * Free the memory used by the pathname.
	 */
	debug("%s: %s", file->path, "removing from file list");
	free(file->absolute_path);
	file->absolute_path = NULL;
	file->path = NULL;
	file->leaf = NULL;

	/* Free the file structure itself. */
	free(file);
}


/*
 * Check the given file's mtime and size; if either have changed, return 1.
 *
 * Returns 0 if nothing has changed, 1 if it has, or -1 if the file does not
 * exist or is not a regular file.
 *
 * If the file is successfully opened but there is an error while reading
 * it, 0 is returned as if it had not changed.
 */
static int ds_file_checkchanged(ds_file_t file)
{
	struct stat sb;

	if (NULL == file)
		return -1;

	if (NULL == file->absolute_path)
		return -1;

	memset(&sb, 0, sizeof(sb));

	if (lstat(file->absolute_path, &sb) != 0)
		return -1;

	if (!S_ISREG((mode_t) (sb.st_mode)))
		return -1;

	if (((time_t) (sb.st_mtime) == file->mtime) && ((off_t) (sb.st_size) == file->size))
		return 0;

	debug("%s: %s", NULL == file->path ? "(null)" : file->path, "file changed");

	file->mtime = (time_t) (sb.st_mtime);
	file->size = (off_t) (sb.st_size);

	return 1;
}


/*
 * Allocate and return a new top-level directory absolutely rooted at
 * "top_path".  All reported paths within the structure will be relative to
 * "top_path".
 *
 * The "fd_inotify" parameter should be the file descriptor to add directory
 * watches to for inotify, or -1 if inotify is not being used.
 */
static /*@null@ */ ds_dir_t ds_dir_toplevel(int fd_inotify, const char *top_path)
{
	/*@null@ */ ds_dir_t dir;

	dir = calloc(1, sizeof(*dir));
	if (NULL == dir) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*@-unrecog@ */
	/* splint doesn't know about realpath */
	dir->absolute_path = realpath(top_path, NULL);
	/*@+unrecog@ */
	if (NULL == dir->absolute_path) {
		die("%s: %s", "realpath", strerror(errno));
		free(dir);
		return NULL;
	}

	dir->path = &(dir->absolute_path[strlen(dir->absolute_path)]);
	dir->leaf = dir->path;
	dir->wd = -1;
	dir->depth = 0;
	dir->parent = NULL;
	dir->topdir = dir;
	dir->seen_in_rescan = false;

	dir->fd_inotify = fd_inotify;

	return dir;
}


/*
 * Add a subdirectory to the list of subdirectories in the given parent
 * directory; if the subdirectory is already in the list, just return the
 * existing subdirectory.
 *
 * The "name" string should contain the name of the directory relative to
 * the parent directory (not the full path), e.g. "somedir".
 *
 * Returns the subdirectory, or NULL on error.
 */
static /*@null@ */ ds_dir_t ds_dir_add(ds_dir_t dir, const char *name)
{
	/*@null@ */ ds_dir_t subdir;
	size_t idx;

	if (NULL == dir)
		return NULL;
	if (NULL == name)
		return NULL;

	/*
	 * Check that this subdirectory wouldn't be too deep.
	 */
	if (dir->depth >= watch_dir_params->max_dir_depth) {
		debug("%s/%s: %s", NULL == dir->path ? "(null)" : dir->path, name, "too deep - not adding");
		return NULL;
	}

	/*
	 * Check we don't already have this subdirectory in the directory
	 * structure - if we do, return the existing structure.
	 */
	for (idx = 0; idx < dir->subdir_count; idx++) {
		if ((NULL != dir->subdirs[idx]->leaf) && (strcmp(dir->subdirs[idx]->leaf, name) == 0)) {
			return dir->subdirs[idx];
		}
	}

	/*
	 * Extend the subdirectory array in the directory structure if we
	 * need to.
	 */
	if (dir->subdir_count >= dir->subdir_array_alloced) {
		size_t target_array_alloced;
		void *newptr;

		target_array_alloced = dir->subdir_array_alloced;
		while (target_array_alloced <= dir->subdir_count) {
			target_array_alloced += DIRCONTENTS_ALLOC_CHUNK;
		}
		newptr = realloc((void *) (dir->subdirs), target_array_alloced * sizeof(dir->subdirs[0]));
		if (NULL == newptr) {
			die("%s: %s", "realloc", strerror(errno));
			return NULL;
		}
		dir->subdirs = newptr;
		dir->subdir_array_alloced = target_array_alloced;
	}

	/*
	 * Allocate a new directory structure for the subdirectory.
	 */
	subdir = calloc(1, sizeof(*subdir));
	if (NULL == subdir) {
		die("%s: %s", "calloc", strerror(errno));
		return NULL;
	}

	/*
	 * Fill in the new subdirectory structure.
	 */
	if (asprintf(&(subdir->absolute_path), "%s/%s", dir->absolute_path, name) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		free(subdir);
		return NULL;
	}
	if (NULL != dir->topdir) {
		subdir->path = &(subdir->absolute_path[strlen(dir->topdir->absolute_path) + 1]);
	} else {
		subdir->path = &(subdir->absolute_path[strlen(dir->absolute_path) + 1]);
	}
	subdir->leaf = ds_leafname(subdir->absolute_path);

	subdir->wd = -1;
	subdir->depth = dir->depth + 1;
	subdir->parent = dir;
	subdir->topdir = dir->topdir;
	subdir->seen_in_rescan = false;

	/*
	 * Add the subdirectory to the directory structure, and mark the
	 * list as unsorted.
	 */
	dir->subdirs[dir->subdir_count] = subdir;
	dir->subdir_count++;
	dir->subdirs_unsorted = true;

	return subdir;
}


/*
 * Free a directory information structure.  This includes all files and
 * directories it contains.
 *
 * The parent directory's "subdirs" list is updated to remove this
 * subdirectory from it.
 *
 * If this the top directory, it is not freed, but all its values are set to
 * NULL.
 */
static void ds_dir_remove(ds_dir_t dir)
{
	bool is_top_dir;
	size_t item;

	if (NULL == dir)
		return;

	is_top_dir = (dir == dir->topdir) ? true : false;

	/*
	 * Remove the watch on this directory.
	 */
	if ((dir->wd >= 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s", NULL == dir->path ? "(null)" : dir->path, "removing watch");
		if (inotify_rm_watch(dir->topdir->fd_inotify, dir->wd) != 0) {
			/*
			 * We can get EINVAL if the directory was deleted,
			 * so just ignore that.
			 */
			if (errno != EINVAL) {
				error("%s: %s", "inotify_rm_watch", strerror(errno));
			}
		}
		ds_watch_index_remove(dir->topdir, dir->wd);
		dir->wd = -1;
	}

	/*
	 * Remove all files from this directory.
	 */
	if (NULL != dir->files) {
		for (item = 0; item < dir->file_count; item++) {
			/* wipe parent field to avoid wasted work */
			dir->files[item]->parent = NULL;
			ds_file_remove(dir->files[item]);
		}
		free(dir->files);
		dir->files = NULL;
		dir->file_count = 0;
		dir->file_array_alloced = 0;
	}

	/*
	 * Remove all subdirectories from this directory (recursive).
	 */
	if (NULL != dir->subdirs) {
		for (item = 0; item < dir->subdir_count; item++) {
			dir->subdirs[item]->parent = NULL;
			ds_dir_remove(dir->subdirs[item]);
		}
		free(dir->subdirs);
		dir->subdirs = NULL;
		dir->subdir_count = 0;
		dir->subdir_array_alloced = 0;
	}

	/*
	 * Remove this subdirectory from our parent's directory listing, if
	 * we have a parent.  Note that when we call ourselves, above, we've
	 * wiped the parent in the subdirectory, to avoid wasted work.
	 */
	if ((NULL != dir->parent) && (NULL != dir->leaf)) {
		size_t readidx, writeidx;
		for (readidx = 0, writeidx = 0; readidx < dir->parent->subdir_count; readidx++) {
			if ((NULL != dir->parent->subdirs[readidx]->leaf)
			    && (strcmp(dir->parent->subdirs[readidx]->leaf, dir->leaf) == 0)
			    ) {
				continue;
			}
			if (readidx != writeidx) {
				dir->parent->subdirs[writeidx] = dir->parent->subdirs[readidx];
			}
			writeidx++;
		}
		dir->parent->subdir_count = writeidx;
		dir->parent->subdirs_unsorted = true;
	}

	/* Remove the directory from the change queue. */
	ds_change_queue_dir_remove(dir);

	/*
	 * Free the memory used by the pathname.
	 */
	if (NULL != dir->absolute_path) {
		debug("%s: %s", NULL == dir->path ? "(null)" : dir->path, "removing from directory list");
		free(dir->absolute_path);
		dir->absolute_path = NULL;
		dir->path = NULL;
		dir->leaf = NULL;
	}

	/*
	 * Free the watch index.
	 */
	if (NULL != dir->watch_index) {
		free(dir->watch_index);
		dir->watch_index = NULL;
	}

	/*
	 * Free the change queue.
	 */
	if (NULL != dir->change_queue) {
		free(dir->change_queue);
		dir->change_queue = NULL;
	}

	/*
	 * Free the changed paths list.
	 */
	if (NULL != dir->changed_paths) {
		size_t idx;
		for (idx = 0; idx < dir->changed_paths_length; idx++) {
			free(dir->changed_paths[idx]);
		}
		free(dir->changed_paths);
		dir->changed_paths = NULL;
		dir->changed_paths_length = 0;
		dir->changed_paths_alloced = 0;
	}

	/* Free the directory structure itself, if it's not the top dir. */
	if (!is_top_dir)
		free(dir);
}


/*
 * Filter for any filename.
 *
 * Ignore anything ending in .tmp or ~ by default, or if exclude_count is
 * >0, ignore anything matching any of the patterns in excludes[].
 *
 * Returns 1 if the file should be included, 0 if it should be ignored.
 */
static int ds_filename_valid(const char *leafname)
{
	if (leafname[0] == '\0')
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '\0'))
		return 0;
	if ((leafname[0] == '.') && (leafname[1] == '.')
	    && (leafname[2] == '\0'))
		return 0;

	if (watch_dir_params->exclude_count > 0) {
		/*
		 * If given an exclusion list, use it.
		 */
		size_t eidx;
		for (eidx = 0; eidx < watch_dir_params->exclude_count; eidx++) {
			if (NULL == watch_dir_params->excludes[eidx])
				continue;
			if (fnmatch(watch_dir_params->excludes[eidx], leafname, 0) == 0)
				return 0;
		}
	} else {
		/*
		 * Default is to exclude *~ and *.tmp
		 */
		size_t namelen;
		namelen = strlen(leafname);
		if ((1 <= namelen) && (leafname[namelen - 1] == '~'))
			return 0;
		if ((namelen > 4)
		    && (strcmp(&(leafname[namelen - 4]), ".tmp") == 0))
			return 0;
	}

	return 1;
}


/*
 * Filter for scanning directories - only include filenames we should be
 * processing.
 */
static int scan_directory_filter(const struct dirent *d)
{
	return ds_filename_valid(d->d_name);
}


/*
 * Recursively scan the given directory.  Also checks files for changes. 
 * Returns nonzero if the scan failed, in which case the directory will have
 * been deleted from the lists.
 *
 * If no_recurse is true, then no subdirectories are scanned, though
 * subdirectories are still added and removed as necessary.
 */
static int ds_dir_scan(ds_dir_t dir, bool no_recurse)
{
	struct dirent **namelist;
	int namelist_length, itemidx, diridx, fileidx;
	struct stat dirsb;

	if (NULL == dir)
		return 1;
	if (NULL == dir->absolute_path)
		return 1;

	if (dir->depth > watch_dir_params->max_dir_depth) {
		debug("%s: %s: %s", NULL == dir->path ? "(null)" : dir->path, "too deep - removing");
		ds_dir_remove(dir);
		return 1;
	}

	memset(&dirsb, 0, sizeof(dirsb));

	if (lstat(dir->absolute_path, &dirsb) != 0) {
		error("%s: %s: %s", NULL == dir->path ? "(null)" : dir->path, "lstat", strerror(errno));
		ds_dir_remove(dir);
		return 1;
	}

	namelist_length = scandir(dir->absolute_path, &namelist, scan_directory_filter, alphasort);
	if (namelist_length < 0) {
		error("%s: %s: %s", dir->absolute_path, "scandir", strerror(errno));
		ds_dir_remove(dir);
		return 1;
	}

	/*
	 * Mark all subdirectories and files as having not been seen, so we
	 * can spot which ones have been removed after we've scanned the
	 * directory.
	 */
	for (diridx = 0; diridx < (int) (dir->subdir_count); diridx++) {
		dir->subdirs[diridx]->seen_in_rescan = false;
	}
	for (fileidx = 0; fileidx < (int) (dir->file_count); fileidx++) {
		dir->files[fileidx]->seen_in_rescan = false;
	}

	/*
	 * Go through the directory scan results, adding new items to the
	 * arrays.
	 */
	for (itemidx = 0; itemidx < namelist_length; itemidx++) {
		/*@null@ */ char *item_full_path;
		char *item_leaf;
		struct stat sb;

		item_full_path = NULL;
		if (asprintf(&item_full_path, "%s/%s", dir->absolute_path, namelist[itemidx]->d_name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return 1;
		}
		if (NULL == item_full_path) {
			die("%s: %s", "asprintf", "NULL");
			return 1;
		}
		item_leaf = ds_leafname(item_full_path);

		free(namelist[itemidx]);

		if ((NULL == item_leaf) || (ds_filename_valid(item_leaf) == 0)) {
			free(item_full_path);
			continue;
		}

		memset(&sb, 0, sizeof(sb));

		if (lstat(item_full_path, &sb) != 0) {
			free(item_full_path);
			continue;
		}

		if (S_ISREG((mode_t) (sb.st_mode))) {
			ds_file_t file;
			file = ds_file_add(dir, item_leaf);
			if (NULL != file)
				file->seen_in_rescan = true;
		} else if (S_ISDIR((mode_t) (sb.st_mode))) {
			ds_dir_t subdir;
			if (sb.st_dev == dirsb.st_dev) {
				subdir = ds_dir_add(dir, item_leaf);
				if (NULL != subdir)
					subdir->seen_in_rescan = true;
			} else {
				debug("%s/%s: %s", NULL == dir->path ? "(null)" : dir->path, item_leaf,
				      "skipping - different filesystem");
			}
		}

		free(item_full_path);
	}

	free(namelist);

	/*
	 * Delete any subdirectories that we did not see on rescan, and
	 * recursively scan those that we did.
	 */
	for (diridx = 0; diridx < (int) (dir->subdir_count); diridx++) {
		if (dir->subdirs[diridx]->seen_in_rescan) {
			if (no_recurse)
				continue;
			if (ds_dir_scan(dir->subdirs[diridx], false) != 0) {
				/* Go back one, as this diridx has now gone */
				diridx--;
			}
		} else {
			ds_dir_remove(dir->subdirs[diridx]);
			/* Go back one, as this diridx has now gone */
			diridx--;
		}
	}

	/*
	 * Delete any files that we did not see on rescan.
	 */
	for (fileidx = 0; fileidx < (int) (dir->file_count); fileidx++) {
		if (dir->files[fileidx]->seen_in_rescan)
			continue;
		ds_file_remove(dir->files[fileidx]);
		/* Go back one, as this fileidx has now gone */
		fileidx--;
	}

	/*
	 * Check all files for changes.
	 */
	for (fileidx = 0; fileidx < (int) (dir->file_count); fileidx++) {
		int changed;
		changed = ds_file_checkchanged(dir->files[fileidx]);
		if (changed < 0) {
			ds_file_remove(dir->files[fileidx]);
			/* Go back one, as this fileidx has now gone */
			fileidx--;
		}
	}

	/*
	 * Add an inotify watch to this directory if there isn't one
	 * already.
	 */
	if ((dir->wd < 0) && (NULL != dir->topdir)
	    && (dir->topdir->fd_inotify >= 0)) {
		debug("%s: %s", NULL == dir->path ? "(null)" : dir->path, "adding watch");
		dir->wd =
		    inotify_add_watch(dir->topdir->fd_inotify,
				      dir->absolute_path,
				      (uint32_t) (IN_CREATE | IN_DELETE | IN_MODIFY | IN_DELETE_SELF | IN_MOVED_FROM |
						  IN_MOVED_TO));
		if (dir->wd < 0) {
			error("%s: %s: %s", NULL == dir->path ? "(null)" : dir->path, "inotify_add_watch",
			      strerror(errno));
		} else {
			/*
			 * Add this watch descriptor to the directory index,
			 * so that we can find this directory structure when
			 * an inotify event arrives.
			 */
			ds_watch_index_add(dir, dir->wd);
		}
	}

	return 0;
}


/*
 * Process queued changes until the given time or until all queue entries
 * we're ready to process have been done.
 */
static void ds_change_queue_process(ds_dir_t topdir, time_t work_until)
{
	size_t readidx, writeidx;

	if (NULL == topdir)
		return;

	if (topdir->change_queue_length < 1)
		return;

	debug("%s: %d", "change queue: starting run, queue length", topdir->change_queue_length);

	for (readidx = 0, writeidx = 0; readidx < topdir->change_queue_length; readidx++) {
		ds_change_queue_t entry;
		time_t now;

		entry = &(topdir->change_queue[readidx]);
		if ((entry->file == NULL) && (entry->dir == NULL))
			continue;

		(void) time(&now);

		/*
		 * Skip if it's not yet time for this item, or if we've
		 * reached our work_until time.
		 */
		if ((entry->when > now) || (now >= work_until)) {
			if (readidx != writeidx) {
				memcpy(&(topdir->change_queue[writeidx]),
				       &(topdir->change_queue[readidx]), sizeof(*entry));
				writeidx++;
			}
			writeidx++;
			continue;
		}

		if (NULL != entry->file) {
			int changed;
			ds_file_t file;

			file = entry->file;
			entry->file = NULL;

			debug("%s: %s", NULL == file->path ? "(null)" : file->path, "checking for changes");
			changed = ds_file_checkchanged(file);

			if (changed < 0) {
				if ((NULL != file->parent->path) && (!watch_dir_params->only_list_files)) {
					mark_path_changed(file->parent->topdir, CHANGED_PATH(file->parent), true);
				}
				ds_file_remove(file);
			} else if (changed > 0) {
				if ((NULL != file->path) && (!watch_dir_params->only_list_directories)) {
					mark_path_changed(file->parent->topdir, CHANGED_PATH(file), false);
				}
			}
		} else if (NULL != entry->dir) {
			ds_dir_t dir;

			dir = entry->dir;
			entry->dir = NULL;

			debug("%s: %s", NULL == dir->path ? "(null)" : dir->path, "triggering scan");
			(void) ds_dir_scan(dir, false);
		}
	}

	topdir->change_queue_length = writeidx;

	debug("%s: %d", "change queue: run ended, queue length", topdir->change_queue_length);
}


/*
 * Process a change to a directory inside a watched directory.
 */
static void process_dir_change(struct inotify_event *event, ds_dir_t dir)
{
	ds_dir_t subdir;
	inotify_action_t action;
	size_t idx;
	char *fullpath = NULL;
	struct stat sb;
	ds_dir_t newdir;

	/*
	 * Find the directory structure to which this event refers, if
	 * known.
	 */
	subdir = NULL;
	for (idx = 0; idx < dir->subdir_count; idx++) {
		if (NULL == dir->subdirs[idx]->leaf)
			continue;
		if (strcmp(event->name, dir->subdirs[idx]->leaf) != 0)
			continue;
		subdir = dir->subdirs[idx];
		break;
	}

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != subdir)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != subdir)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new directory we've not seen before.
		 */

		/*
		 * Ignore the directory if it doesn't pass the filename
		 * filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		if (asprintf(&fullpath, "%s/%s", dir->absolute_path, event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the directory if it doesn't exist.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		}

		/*
		 * Ignore it if it's not a directory.
		 */
		if (!S_ISDIR((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Add the new directory and queue a scan for it.
		 */
		debug("%s: %s", fullpath, "adding new subdirectory");
		newdir = ds_dir_add(dir, event->name);
		free(fullpath);
		ds_change_queue_dir_add(newdir, 0);

		/*
		 * Mark this as a changed path.
		 */
		if (!watch_dir_params->only_list_files) {
			mark_path_changed(dir->topdir, CHANGED_PATH(newdir), true);
		}

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a directory we've seen before, so queue a rescan for
		 * it.
		 */
		if (NULL != subdir) {
			debug("%s: %s", NULL == subdir->path ? "(null)" : subdir->path, "queueing rescan");
			ds_change_queue_dir_add(subdir, 0);
		}
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this directory before, delete its
		 * structure.
		 */
		if (NULL != subdir) {
			debug("%s: %s", NULL == subdir->path ? "(null)" : subdir->path, "triggering removal");
			ds_dir_remove(subdir);
		}
		/*
		 * Mark the parent directory as a changed path.
		 */
		if ((NULL != dir->path) && (!watch_dir_params->only_list_files)) {
			mark_path_changed(dir->topdir, CHANGED_PATH(dir), true);
		}
		break;
	}
}


/*
 * Process a change to a file inside a watched directory.
 */
static void process_file_change(struct inotify_event *event, ds_dir_t dir)
{
	ds_file_t file;
	inotify_action_t action;
	size_t idx;
	char *fullpath;
	struct stat sb;
	ds_file_t newfile;

	/*
	 * Find the file structure to which this event refers, if known.
	 */
	file = NULL;
	for (idx = 0; idx < dir->file_count; idx++) {
		if (NULL == dir->files[idx]->leaf)
			continue;
		if (strcmp(event->name, dir->files[idx]->leaf) != 0)
			continue;
		file = dir->files[idx];
		break;
	}

	/*
	 * Decide what to do: is this a newly created item, an existing item
	 * that has been modified, or an existing item that has been
	 * deleted?
	 */
	action = IN_ACTION_NONE;
	if (0 != (event->mask & (IN_ATTRIB | IN_CREATE | IN_MODIFY | IN_MOVED_TO))) {
		action = IN_ACTION_CREATE;
		if (NULL != file)
			action = IN_ACTION_UPDATE;
	} else if ((0 != (event->mask & (IN_DELETE | IN_MOVED_FROM)))
		   && (NULL != file)) {
		action = IN_ACTION_DELETE;
	}

	switch (action) {
	case IN_ACTION_NONE:
		break;
	case IN_ACTION_CREATE:
		/*
		 * This a new file we've not seen before.
		 */

		/*
		 * Ignore the file if it doesn't pass the filename filter.
		 */
		if (ds_filename_valid(event->name) == 0) {
			break;
		}

		/*
		 * Make a copy of the full pathname.
		 */
		fullpath = NULL;
		if (asprintf(&fullpath, "%s/%s", dir->absolute_path, event->name) < 0) {
			die("%s: %s", "asprintf", strerror(errno));
			return;
		}
		if (NULL == fullpath) {
			die("%s: %s", "asprintf", "NULL");
			return;
		}

		memset(&sb, 0, sizeof(sb));

		/*
		 * Ignore the file if it doesn't exist or it isn't a regular
		 * file.
		 */
		if (lstat(fullpath, &sb) != 0) {
			free(fullpath);
			break;
		} else if (!S_ISREG((mode_t) (sb.st_mode))) {
			free(fullpath);
			break;
		}

		/*
		 * Add the new file and queue it to be checked.
		 */
		debug("%s: %s", fullpath, "adding new file");
		newfile = ds_file_add(dir, event->name);
		ds_change_queue_file_add(newfile, 0);

		free(fullpath);

		break;
	case IN_ACTION_UPDATE:
		/*
		 * This a file we've seen before, so queue a check for it.
		 */
		ds_change_queue_file_add(file, 0);
		break;
	case IN_ACTION_DELETE:
		/*
		 * If we've seen this file before, delete its structure.
		 */
		debug("%s: %s", NULL == file->path ? "(null)" : file->path, "triggering removal");
		/*
		 * Mark the parent directory as a changed path.
		 */
		if (!watch_dir_params->only_list_files) {
			mark_path_changed(file->parent->topdir, CHANGED_PATH(file->parent), true);
		}
		ds_file_remove(file);
		break;
	}
}


/*
 * Process incoming inotify events.
 *
 * If the top directory is removed, it is not freed, but all its values are
 * nullified.
 */
static void process_inotify_events(ds_dir_t topdir)
{
	unsigned char readbuf[8192];
	ssize_t got, pos;

	if (NULL == topdir)
		return;
	if (topdir->fd_inotify < 0)
		return;

	memset(readbuf, 0, sizeof(readbuf));

	/*
	 * Read as many events as we can.
	 */
	got = read(topdir->fd_inotify, readbuf, sizeof(readbuf));
	if (got <= 0) {
		error("%s: (%d): %s", "inotify read event", got, strerror(errno));
		(void) close(topdir->fd_inotify);
		topdir->fd_inotify = -1;
		return;
	}

	/*
	 * Process each event that we've read.
	 */
	for (pos = 0; pos < got;) {
		struct inotify_event *event;
		/*@null@ */ ds_dir_t dir = NULL;

		event = (struct inotify_event *) &(readbuf[pos]);
		dir = ds_watch_index_lookup(topdir, event->wd);

#if ENABLE_DEBUGGING
		if (debugging_enabled) {
			char flags[1024];
			flags[0] = '\0';
			if (0 != (event->mask & IN_ACCESS))
				strcat(flags, " IN_ACCESS");
			if (0 != (event->mask & IN_ATTRIB))
				strcat(flags, " IN_ATTRIB");
			if (0 != (event->mask & IN_CLOSE_WRITE))
				strcat(flags, " IN_CLOSE_WRITE");
			if (0 != (event->mask & IN_CLOSE_NOWRITE))
				strcat(flags, " IN_CLOSE_NOWRITE");
			if (0 != (event->mask & IN_CREATE))
				strcat(flags, " IN_CREATE");
			if (0 != (event->mask & IN_DELETE))
				strcat(flags, " IN_DELETE");
			if (0 != (event->mask & IN_DELETE_SELF))
				strcat(flags, " IN_DELETE_SELF");
			if (0 != (event->mask & IN_MODIFY))
				strcat(flags, " IN_MODIFY");
			if (0 != (event->mask & IN_MOVE_SELF))
				strcat(flags, " IN_MOVE_SELF");
			if (0 != (event->mask & IN_MOVED_FROM))
				strcat(flags, " IN_MOVED_FROM");
			if (0 != (event->mask & IN_MOVED_TO))
				strcat(flags, " IN_MOVED_TO");
			if (0 != (event->mask & IN_OPEN))
				strcat(flags, " IN_OPEN");
			if (0 != (event->mask & IN_IGNORED))
				strcat(flags, " IN_IGNORED");
			if (0 != (event->mask & IN_ISDIR))
				strcat(flags, " IN_ISDIR");
			if (0 != (event->mask & IN_Q_OVERFLOW))
				strcat(flags, " IN_Q_OVERFLOW");
			if (0 != (event->mask & IN_UNMOUNT))
				strcat(flags, " IN_UNMOUNT");
			if (event->len > 0) {
				debug("%s: %d: %s: %.*s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : dir->path, event->len, event->name, flags);
			} else {
				debug("%s: %d: %s: %s:%s", "inotify", event->wd,
				      NULL == dir ? "(unknown)" : dir->path, "(none)", flags);
			}
		}
#endif				/* ENABLE_DEBUGGING */

		pos += sizeof(*event) + event->len;

		/*
		 * There's nothing we can do if we don't know which
		 * directory it was.
		 */
		if (NULL == dir)
			continue;

		if (0 != (event->mask & IN_DELETE_SELF)) {
			ds_dir_remove(dir);
			continue;
		}

		/*
		 * If this isn't an event about a named thing in this
		 * directory, we can't do anything.
		 */
		if (event->len <= 0)
			continue;
		if ('\0' == event->name[0])
			continue;

		if (0 != (event->mask & IN_ISDIR)) {
			process_dir_change(event, dir);
		} else {
			process_file_change(event, dir);
		}
	}
}


/*
 * Add a path to the list of changed paths.
 */
static void mark_path_changed( /*@null@ */ ds_dir_t topdir, const char *path, bool isdir)
{
	char *savepath;
	size_t idx;

	if (NULL == topdir)
		return;

	if (NULL == path)
		return;

	if (asprintf(&savepath, "%s%s", path, isdir ? "/" : "") < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}

	/*
	 * Check the path isn't already listed - don't list it twice.
	 */
	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		if (strcmp(topdir->changed_paths[idx], savepath) == 0) {
			free(savepath);
			return;
		}
	}

	/*
	 * Extend the array if necessary.
	 */
	if (topdir->changed_paths_length >= topdir->changed_paths_alloced) {
		size_t new_size;
		char **newptr;
		new_size = topdir->changed_paths_alloced + CHANGEDPATH_ALLOC_CHUNK;
		newptr = realloc(topdir->changed_paths, new_size * sizeof(topdir->changed_paths[0]));
		if (NULL == newptr) {
			die("%s: %s", "realloc", strerror(errno));
			return;
		}
		topdir->changed_paths = newptr;
		topdir->changed_paths_alloced = new_size;
	}

	debug("%s: %s", "adding to changed paths", savepath);

	/*
	 * Add the new entry, and extend the length of the array.
	 */
	topdir->changed_paths[topdir->changed_paths_length] = savepath;
	topdir->changed_paths_length++;
}


/*
 * Write out a new file containing the current changed paths list, and clear
 * the list.
 */
static void dump_changed_paths(ds_dir_t topdir, const char *savedir)
{
	char *savefile = NULL;
	char *temp_filename;
	struct tm *tm;
	time_t t;
	int tmpfd;
	FILE *fptr;
	size_t idx;
	char delimiter;

	if (NULL == topdir->changed_paths)
		return;
	if (topdir->changed_paths_length <= 0)
		return;

	delimiter = '\n';
	if (watch_dir_params->null_delimiter)
		delimiter = '\0';

	t = time(NULL);
	tm = localtime(&t);

	if (asprintf
	    (&savefile, "%s/%04d%02d%02d-%02d%02d%02d.%d%s", savedir,
	     tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec, getpid(),
	     NULL == watch_dir_params->changefile_suffix ? "" : watch_dir_params->changefile_suffix) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return;
	}
	if (NULL == savefile) {
		die("%s: %s", "asprintf", "null");
		return;
	}

	tmpfd = ds_tmpfile(savefile, &temp_filename);
	if (tmpfd < 0) {
		free(savefile);
		return;
	}

	fptr = fdopen(tmpfd, "w");
	if (NULL == fptr) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) close(tmpfd);
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		fprintf(fptr, "%s%c", topdir->changed_paths[idx], delimiter);
	}

	if (0 != fclose(fptr)) {
		error("%s: %s", temp_filename, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	if (rename(temp_filename, savefile) != 0) {
		error("%s: %s", savefile, strerror(errno));
		(void) remove(temp_filename);
		free(temp_filename);
		free(savefile);
		return;
	}

	free(temp_filename);
	free(savefile);

	for (idx = 0; idx < topdir->changed_paths_length; idx++) {
		free(topdir->changed_paths[idx]);
	}

	free(topdir->changed_paths);
	topdir->changed_paths = NULL;
	topdir->changed_paths_length = 0;
	topdir->changed_paths_alloced = 0;
}


/*
 * Handler for an exit signal such as SIGTERM - set a flag to trigger an
 * exit.
 */
static void watch_dir_exitsignal( /*@unused@ */  __attribute__((unused))
				 int signum)
{
	watch_dir_exit_now = true;
}


/*
 * Main entry point.  Set everything up and enter the main loop, which does
 * the following:
 *
 *   - A periodic rescan from the top level directory down.
 *   - Processing of inotify events from all known directories.
 *   - Processing of the change queue generated from the above two.
 *   - Periodic output of a file listing updated paths.
 *
 * Scanned directories are watched using inotify, so that changes to files
 * within it can be noticed immediately.
 *
 * A change queue is maintained, comprising a list of files and directories
 * to re-check, and the time at which to do so.  This is so that when
 * multiple files are changed, or the same file is changed several times, it
 * can be dealt with intelligently - they are pushed on to the change queue,
 * with duplicates being ignored, and then the change queue is processed in
 * chunks to avoid starvation caused by inotify events from one file
 * changing rapidly.
 */
int watch_dir(struct watch_dir_params_s *params)
{
	int *inotify_fds;		 /* fds to watch for inotify on */
	time_t next_full_scan;		 /* when to run next full scan */
	/*@null@ */ ds_dir_t *topdirs;
	/* top-level directory contents */
	time_t next_change_queue_run;	 /* when to next run changes */
	time_t next_changedpath_dump;	 /* when to next dump changed paths */
	struct sigaction sa;
	size_t dir_idx;

	watch_dir_params = params;

	memset(&sa, 0, sizeof(sa));

	/*
	 * Set up the signal handlers.
	 */
	sa.sa_handler = watch_dir_exitsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGTERM, &sa, NULL);

	sa.sa_handler = watch_dir_exitsignal;
	(void) sigemptyset(&(sa.sa_mask));
	sa.sa_flags = 0;
	(void) sigaction(SIGINT, &sa, NULL);

	/*
	 * Create the inotify event queues.
	 *
	 * We use a separate event queue for each top-level directory so
	 * that when an event comes in, we can associate it with the
	 * appropriate top-level.
	 */
	inotify_fds = calloc(params->toplevel_path_count, sizeof(int));
	if (NULL == inotify_fds) {
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		inotify_fds[dir_idx] = inotify_init();
		if (inotify_fds[dir_idx] < 0) {
			error("%s: %s", "inotify", strerror(errno));
			return EXIT_FAILURE;
		}
	}

	/*
	 * Create the top-level directory memory structures.
	 */
	topdirs = calloc(params->toplevel_path_count, sizeof(*topdirs));
	if (NULL == topdirs) {
		return EXIT_FAILURE;
	}
	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		topdirs[dir_idx] = ds_dir_toplevel(inotify_fds[dir_idx], params->toplevel_paths[dir_idx]);
		if (NULL == topdirs[dir_idx]) {
			return EXIT_FAILURE;
		}
	}

	/*
	 * Enter the main loop.
	 */

	next_change_queue_run = 0;
	next_full_scan = 0;
	next_changedpath_dump = 0;

	while (!watch_dir_exit_now) {
		time_t now;
		fd_set readfds;
		struct timeval timeout;
		int ready, fd_max;

		/*
		 * Wait for an inotify event on any of the queues.
		 */
		FD_ZERO(&readfds);
		fd_max = 0;
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			FD_SET(fd_inotify, &readfds);
			if (fd_inotify > fd_max)
				fd_max = fd_inotify;
		}

		timeout.tv_sec = 0;
		timeout.tv_usec = 100000;
		ready = select(1 + fd_max, &readfds, NULL, NULL, &timeout);

		if (ready < 0) {
			if (errno != EINTR)
				error("%s: %s", "select", strerror(errno));
			watch_dir_exit_now = true;
			break;
		}

		/*
		 * Process new inotify events.
		 */
		for (dir_idx = 0; dir_idx < params->toplevel_path_count && ready > 0 && !watch_dir_exit_now; dir_idx++) {
			int fd_inotify;

			fd_inotify = inotify_fds[dir_idx];
			if (fd_inotify < 0)
				continue;

			if (!FD_ISSET(fd_inotify, &readfds))
				continue;

			process_inotify_events(topdirs[dir_idx]);
			if (NULL == topdirs[dir_idx]->absolute_path) {
				error("%s: %s", params->toplevel_paths[dir_idx], "top directory removed");
				watch_dir_exit_now = true;
				break;
			}
		}

		if (watch_dir_exit_now)
			break;

		if (0 == ready) {
			timeout.tv_sec = 0;
			timeout.tv_usec = 100000;
			(void) select(0, NULL, NULL, NULL, &timeout);
		}

		(void) time(&now);

		/*
		 * Do a full scan periodically.
		 */
		if (now >= next_full_scan) {
			next_full_scan = now + params->full_scan_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_dir_add(topdirs[dir_idx], 0);
			}
		}

		/*
		 * Run our change queue.
		 */
		if (now >= next_change_queue_run) {
			next_change_queue_run = now + params->queue_run_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				ds_change_queue_process(topdirs[dir_idx], now + params->queue_run_max_seconds);
			}
		}

		/*
		 * Dump our list of changed paths.
		 */
		if (now >= next_changedpath_dump) {
			next_changedpath_dump = now + params->changedpath_dump_interval;
			for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
				dump_changed_paths(topdirs[dir_idx], params->changedpath_dir);
			}
		}
	}

	for (dir_idx = 0; dir_idx < params->toplevel_path_count; dir_idx++) {
		ds_dir_remove(topdirs[dir_idx]);
		(void) free(topdirs[dir_idx]);
		if (inotify_fds[dir_idx] >= 0) {
			if (0 != close(inotify_fds[dir_idx]))
				error("%s: %s", "close", strerror(errno));
		}
	}
	(void) free(topdirs);
	(void) free(inotify_fds);

	return EXIT_SUCCESS;
}
