/*
 * Functions and variables common to all programs in the project.
 *
 * Copyright 2014, 2021, 2023, 2025 Andrew Wood
 *
 * License GPLv3+: GNU GPL version 3 or later <https://gnu.org/licenses/gpl.html>.
 */

#include "config.h"
#include "common.h"

#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#ifndef SPLINT
/* splint 3.1.2 chokes on syslog.h */
#include <syslog.h>
#endif
#include <sys/stat.h>

#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
#include <wchar.h>
#if defined(HAVE_WCTYPE_H)
#include <wctype.h>
#endif
#endif

#include <unistd.h>
#if defined(HAVE_SYS_IOCTL_H)
#include <sys/ioctl.h>
#endif

bool debugging_enabled = false;		 /* global flag to enable debugging */
bool using_syslog = false;		 /* global flag to report to syslog */
char *common_program_name = "";		 /* set this to program leafname */
int error_count = 0;			 /* global error counter from error() */

#if ENABLE_DEBUGGING
/*
 * In debug mode, output the given formatted string to stderr.
 */
void debug(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	if (debugging_enabled) {
		time_t t;
		struct tm *tm;
		char tbuf[128];		 /* flawfinder: ignore */

		/*
		 * flawfinder note: tbuf is only written to by strftime()
		 * which takes its size, and we enforce string termination.
		 */

		(void) time(&t);
		tm = localtime(&t);
		tbuf[0] = '\0';
		if (0 == strftime(tbuf, sizeof(tbuf), "%Y-%m-%d %H:%M:%S", tm)) {
			tbuf[0] = '\0';
		}
		tbuf[sizeof(tbuf) - 1] = '\0';	/* enforce termination */

		(void) fprintf(stderr, "[%s] ", tbuf);
		(void) vfprintf(stderr, format, ap);	/* flawfinder: ignore */
		(void) fprintf(stderr, "\n");

		/*
		 * flawfinder note: vfprintf format is explicitly controlled
		 * by the caller of this function - no mitigation possible
		 * or desirable.
		 */
	}
	va_end(ap);
}
#endif				/* ENABLE_DEBUGGING */


/*
 * Output an error, and increment the error counter.
 */
void error(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	(void) fprintf(stderr, "%s: ", common_program_name);
	(void) vfprintf(stderr, format, ap);	/* flawfinder: ignore */
	(void) fprintf(stderr, "\n");
	va_end(ap);

	if (using_syslog) {
		va_start(ap, format);
		/*@-unrecog@ *//* splint can't use syslog.h */
		(void) vsyslog(LOG_ERR, format, ap);	/* flawfinder: ignore */
		/*@+unrecog@ */
		va_end(ap);
	}

	/*
	 * flawfinder note: vfprintf / vsyslog format strings are explicitly
	 * controlled by the caller of this function - no mitigation
	 * possible or desirable.
	 */

	error_count++;
}


/*
 * Output an error and exit the program.
 */
void die(const char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	(void) fprintf(stderr, "%s: ", common_program_name);
	(void) vfprintf(stderr, format, ap);	/* flawfinder: ignore */
	(void) fprintf(stderr, "\n");
	va_end(ap);

	if (using_syslog) {
		va_start(ap, format);
		/*@-unrecog@ *//* splint can't use syslog.h */
		(void) vsyslog(LOG_ERR, format, ap);	/* flawfinder: ignore */
		/*@+unrecog@ */
		va_end(ap);
	}

	/* flawfinder note: same reasons as error(). */

	exit(EXIT_FAILURE);
}


/*
 * Return the string position of the character just after the last '/' in
 * the given path.
 */
size_t ds_leafname_pos( /*@null@ */ char *pathname)
{
	size_t leafpos;

	if (NULL == pathname)
		return 0;

	leafpos = strlen(pathname);	    /* flawfinder: ignore */
	/*
	 * flawfinder note: we have to rely on the caller to pass a proper
	 * null-terminated string.
	 */

	while ((leafpos > 0) && (pathname[leafpos] != '/')) {
		leafpos--;
	}

	if (pathname[leafpos] == '/')
		leafpos++;

	return leafpos;
}


/*
 * Utility function to return the leafname of the given path.
 */
/*@null@*//*@dependent@ */ char *ds_leafname( /*@null@ */ char *pathname)
{
	if (NULL == pathname)
		return NULL;
	return &(pathname[ds_leafname_pos(pathname)]);
}


/*
 * Open a securely named hidden temporary file based on the given absolute
 * path, and return the file descriptor, filling in the malloced name into
 * tmpnameptr (*tmpnameptr will need to be freed by the caller).
 *
 * Returns -1 on error, in which case nothing is put into tmpnameptr.
 */
int ds_tmpfile( /*@null@ */ char *pathname, /*@null@ */ char **tmpnameptr)
{
	size_t leafpos;
	int fd;
	char *temporary_name = NULL;
	mode_t prev_umask;

	if (NULL == pathname)
		return -1;
	if (NULL == tmpnameptr)
		return -1;

	leafpos = ds_leafname_pos(pathname);
	if (leafpos < 1) {
		error("%s: %s", pathname, _("path contains no directory components"));
		return -1;
	}

	/*@-unrecog@ */
	/* splint doesn't know about asprintf() */
	if (asprintf(&temporary_name, "%.*s/.%sXXXXXX", (int) (leafpos - 1), pathname, &(pathname[leafpos])) < 0) {
		die("%s: %s", "asprintf", strerror(errno));
		return -1;
	}
	/*@+unrecog@ */

	if (NULL == temporary_name) {
		die("%s: %s", "asprintf", strerror(errno));
		return -1;
	}

	/*@-type@ */
	/* splint doesn't like mode_t */
	prev_umask = umask(0000);	    /* flawfinder: ignore */
	(void) umask(prev_umask | 0133);    /* flawfinder: ignore */

	/*@-unrecog@ */
	/* splint doesn't know about mkstemp() */
	fd = mkstemp(temporary_name);	    /* flawfinder: ignore */
	/*@+unrecog@ */
	if (fd < 0) {
		error("%s: %s: %s", temporary_name, "mkstemp", strerror(errno));
		(void) umask(prev_umask);   /* flawfinder: ignore */
		free(temporary_name);
		return -1;
	}

	(void) umask(prev_umask);	    /* flawfinder: ignore */

	/*
	 * flawfinder rationale (umask, mkstemp) - flawfinder recommends
	 * setting the most restrictive umask possible when calling
	 * mkstemp(), so this is what we have done.
	 *
	 * We get the original umask and OR it with 0133 to make sure new
	 * files will be at least chmod 644.  Then we put the umask back to
	 * what it was, after creating the temporary file.
	 */

	/*@+type@ */

	*tmpnameptr = temporary_name;
	return fd;
}


#if ENABLE_SETPROCTITLE
extern char **environ;

/*@null@*/ static char **base_argv = NULL;
static size_t space_available = 0;


/*
 * Assuming the environment comes after the command line arguments, we make
 * a duplicate of the environment array and all its values, so we can use
 * the space the environment used to occupy for the process title.
 *
 * The logic for this was derived from util-linux-ng.
 */
/*@-mustfreefresh@ */
/*@-nullstate@ */
void initproctitle(int argc, char **argv)
{
	char **original_environment = environ;
	char **new_environment;
	size_t env_array_size, env_index;

	/* Find the number of entries in the environment array. */
	for (env_array_size = 0; environ[env_array_size] != NULL; env_array_size++)
		continue;

	/* Allocate a new environment array. */
	new_environment = (char **) malloc(sizeof(char *) * (env_array_size + 1));
	if (NULL == new_environment)
		return;

	/* Duplicate the strings in the environment. */
	for (env_index = 0; original_environment[env_index] != NULL; env_index++) {
		new_environment[env_index] = xstrdup(original_environment[env_index]);
		if (NULL == new_environment[env_index])
			return;
	}
	new_environment[env_index] = NULL;

	base_argv = argv;

	/* Work out how much room we have. */
	if (env_index > 0) {
		/* From argv[0] to the end of the last environment value. */
		space_available = (size_t)
		    (original_environment[env_index - 1] + strlen(original_environment[env_index - 1]) - argv[0]);	/* flawfinder: ignore */
	} else if (argc > 0) {
		/* No environment; from argv[0] to the end of the last argument. */
		space_available = (size_t) (argv[argc - 1] + strlen(argv[argc - 1]) - argv[0]);	/* flawfinder: ignore */
	}

	/*
	 * flawfinder - we have to trust that the environment and argument
	 * strings are null-terminated, since that's what the OS is supposed
	 * to guarantee.
	 */

	environ = new_environment;
}

/*@+nullstate@ */
/*@+mustfreefresh@ */


/*
 * Set the process title visible to ps(1), by overwriting argv[0] with the
 * given printf() format string.
 */
void setproctitle(const char *format, ...)
{
	char title[1024];		 /* flawfinder: ignore */
	size_t length;
	va_list ap;

	/* flawfinder - buffer is zeroed and users of it are bounded. */

	if (NULL == base_argv)
		return;

	memset(title, 0, sizeof(title));

	va_start(ap, format);
	(void) vsnprintf(title, sizeof(title) - 1, format, ap);	/* flawfinder: ignore */
	va_end(ap);

	/*
	 * flawfinder - the format is explicitly caller-supplied.  Callers
	 * of this function are all passing fixed format strings and not
	 * user-supplied formats.
	 */

	length = strlen(title);		    /* flawfinder: ignore */
	/* flawfinder - we left a byte at the end to force null-termination. */

	if (length > space_available - 2)
		length = space_available - 2;

	(void) snprintf(base_argv[0], space_available, "%s -- %s", PACKAGE_NAME, title);
	if (space_available > length)
		memset(base_argv[0] + length, 0, space_available - length);
}
#endif				/* ENABLE_SETPROCTITLE */


/*
 * Like strdup but die() on failure.
 */
/*@null@*/ char *xstrdup(const char *str)
{
	char *ptr;
	/*@-unrecog@ *//* splint doesn't know about strdup() */
	ptr = strdup(str);
	/*@+unrecog@ */
	if (NULL == ptr)
		die("%s", strerror(errno));
	return ptr;
}


/*
 * Create the parent directories of a file, if they don't already exist. 
 * The parent directories are created with mode 755.
 */
void create_parent_dirs(const char *file)
{
	char *dir;
	size_t filename_length, read_pos;

	dir = xstrdup(file);
	if (NULL == dir) {
		return;
	}

	filename_length = strlen(dir);	    /* flawfinder: ignore */
	/*
	 * flawfinder note: we have to rely on the caller to pass a proper
	 * null-terminated string.
	 */

	for (read_pos = 1; read_pos < filename_length; read_pos++) {
		struct stat sb;

		if (dir[read_pos] != '/')
			continue;
		dir[read_pos] = '\0';

		if (0 != stat(dir, &sb)) {
			debug("%s: %s", "creating directory", dir);
			if (0 != mkdir(dir, 0755)) {
				debug("%s: %s: %s", dir, "directory creation failed", strerror(errno));
				break;
			}
		}

		dir[read_pos] = '/';
	}

	free(dir);
}


/*
 * The following functions were taken from SCW 0.4.1.
 */

/*
 * Return the number of display columns needed to show the given string.
 *
 * To do this, we convert it to a wide character string, and use the wide
 * character display width function "wcswidth()" on it.
 *
 * If NLS is disabled, or the string cannot be converted, this is just the
 * same as "strlen()".
 */
size_t calculateDisplayedWidth(const char *string)
{
	size_t width;

#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
	size_t countWideChars;
	size_t bufferSize;
	wchar_t *wideString;

	if (NULL == string)
		return 0;

	/*@-nullpass@ */
	/*
	 * splint note: mbstowcs() manual page on Linux explicitly says it
	 * takes NULL.
	 */
	countWideChars = mbstowcs(NULL, string, 0);
	/*@+nullpass@ */
	if (countWideChars == (size_t) -1) {
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		return strlen(string);	    /* flawfinder: ignore */
		/*
		 * flawfinder rationale: we have already checked for NULL,
		 * and we don't know the size of the originating buffer so
		 * can't use strnlen(); it is up to the caller to provide a
		 * terminated string.
		 */
	}

	bufferSize = sizeof(*wideString) * (1 + countWideChars);
	wideString = malloc(bufferSize);
	if (NULL == wideString) {
		perror("malloc");
		return strlen(string);	    /* flawfinder: ignore */
		/* flawfinder rationale: see above. */
	}
	memset(wideString, 0, bufferSize);

	if (mbstowcs(wideString, string, 1 + countWideChars) == (size_t) -1) {
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		width = strlen(string);	    /* flawfinder: ignore */
		/* flawfinder rationale: see above. */
	} else if (NULL != wideString) {
		/*@-unrecog @ */
		/* splint seems unable to see the prototype. */
		width = wcswidth(wideString, countWideChars);
		/*@+unrecog @ */
	} else {
		width = 0;
	}

	free(wideString);

#else				/* ! defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
	if (NULL == string)
		return 0;

	width = strlen(string);		    /* flawfinder: ignore */
	/* flawfinder rationale: see above. */
#endif				/* defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */

	return width;
}


/*
 * Output a string to "stream", word wrapping to "wrapWidth" characters, and
 * left-padding any new lines after the first one with "leftMargin" spaces.
 *
 * This is a 7-bit ASCII version of outputWordWrap() below - it does not
 * understand multi-byte characters, so it would count those incorrectly.
 */
static void nonWideOutputWordWrap(FILE * stream, const char *string, size_t wrapWidth, size_t leftMargin)
{
	const char *start;
	const char *end;

	if (NULL == string)
		return;

	start = string;

	while (strlen(start) > wrapWidth) { /* flawfinder: ignore */
		/* flawfinder rationale: see above. */

		/*
		 * Find the end of the last word that will fit on this line.
		 */
		end = start + wrapWidth;
		while ((end > start) && (end[0] != ' '))
			end--;
		if (end == start) {
			/*
			 * No appropriate space found, so just display all
			 * the characters up to the wrap width.
			 */
			end = start + wrapWidth;
		} else {
			/*
			 * We've moved backwards to before the space, so
			 * move forwards again into the space.
			 */
			end++;
		}

		/* Output the selected part of the string. */
		fprintf(stream, "%.*s", (int) (end - start), start);

		/* Ensure we never get stuck. */
		if (end == start)
			end++;

		/*
		 * Move the start position forward to where we just
		 * displayed up to.
		 */
		start = end;

		/*
		 * If there's anything left to display, output a new line,
		 * plus the spaces for the left margin.
		 */
		if (start[0] != '\0')
			fprintf(stream, "\n%*s", (int) leftMargin, "");
	}

	/* Output whatever remains. */
	fprintf(stream, "%s", start);
}


/*
 * Output a string to "stream", word wrapping to "wrapWidth" display
 * character positions, and left-padding any new lines after the first one
 * with "leftMargin" spaces.
 *
 * Wide characters are handled if NLS is enabled, but if they can't be, this
 * falls back to the version above, which just counts bytes as characters.
 */
void outputWordWrap(FILE * stream, const char *string, size_t wrapWidth, size_t leftMargin)
{
#if defined(ENABLE_NLS) && defined(HAVE_WCHAR_H)
	size_t stringWidth;
	size_t bufferSize;
	wchar_t *wideString;
	size_t startPosition, endPosition, charsRemaining;

	if (NULL == string)
		return;

	/*@-nullpass@ */
	/* splint note: see earlier mbstowcs() call. */
	stringWidth = mbstowcs(NULL, string, 0);
	/*@+nullpass@ */
	if (stringWidth == (size_t) -1) {
		/*
		 * Unable to determine how many wide characters the
		 * multibyte string will conver to - fall back to the non
		 * multibyte version of this function.
		 */
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}

	bufferSize = sizeof(*wideString) * (1 + stringWidth);
	wideString = malloc(bufferSize);
	if (NULL == wideString) {
		/*
		 * Unable to allocate a buffer to hold the wide-character
		 * version of the supplied multibyte string - fall back to
		 * the non multibyte version of this function.
		 */
		perror("malloc");
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}
	memset(wideString, 0, bufferSize);

	if (mbstowcs(wideString, string, 1 + stringWidth) == (size_t) -1) {
		/*
		 * Failed to perform the conversion - fall back, as above.
		 */
		debug("%s: %s: %s", "mbstowcs", string, strerror(errno));
		free(wideString);
		nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
		return;
	}

	startPosition = 0;
	charsRemaining = stringWidth;

	while (charsRemaining > 0 && wcswidth(&(wideString[startPosition]), charsRemaining) > wrapWidth) {
		size_t nextLineStartPosition;

		/*
		 * Find the end of the last word that will fit on this line.
		 */
		endPosition = startPosition + wrapWidth;
		while ((endPosition > startPosition) && (!iswspace(wideString[endPosition])))
			endPosition--;
		if (endPosition == startPosition) {
			/*
			 * No appropriate space found, so just display all
			 * the characters up to the wrap width.
			 */
			endPosition = startPosition + wrapWidth;
		} else {
			/*
			 * We've moved backwards to before the space, so
			 * move forwards again into the space.
			 */
			endPosition++;
		}

		/*
		 * Record where the next line starts - if it looks like it
		 * would start where we already started this line, move on
		 * one, to avoid getting stuck.
		 */
		nextLineStartPosition = endPosition;
		if (endPosition == startPosition)
			nextLineStartPosition++;

		/*
		 * Convert each wide character to a multibyte string, so it
		 * can be written to the output.
		 */
		while (startPosition < endPosition && startPosition < stringWidth) {
			char multiByteString[MB_CUR_MAX + 1];	/* flawfinder: ignore */
			/*
			 * flawfinder rationale: array is explicitly
			 * cleared, large enough according to the wctomb()
			 * manual, and we explicitly terminate the string.
			 */
			memset(multiByteString, 0, MB_CUR_MAX + 1);
			if (wctomb(multiByteString, wideString[startPosition]) >= 0) {
				multiByteString[MB_CUR_MAX] = '\0';
				fprintf(stream, "%s", multiByteString);
			}
			startPosition++;
		}

		startPosition = nextLineStartPosition;

		/*
		 * If there's anything left to display, output a new line,
		 * plus the spaces for the left margin.
		 */
		if (startPosition < stringWidth)
			fprintf(stream, "\n%*s", (int) leftMargin, "");

		charsRemaining = stringWidth - startPosition;
	}

	/*
	 * Output whatever remains.  As in the loop above, convert each wide
	 * character in turn to a multibyte string to output it.
	 */
	while (startPosition < stringWidth) {
		char multiByteString[MB_CUR_MAX + 1];	/* flawfinder: ignore */
		/* flawfinder rationale as above. */
		memset(multiByteString, 0, MB_CUR_MAX + 1);
		if (wctomb(multiByteString, wideString[startPosition]) >= 0) {
			multiByteString[MB_CUR_MAX] = '\0';
			fprintf(stream, "%s", multiByteString);
		}
		startPosition++;
	}

	free(wideString);

#else				/* ! defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
	nonWideOutputWordWrap(stream, string, wrapWidth, leftMargin);
#endif				/* defined(ENABLE_NLS) && defined(HAVE_WCHAR_H) */
}


/*
 * Fill in *columns and *rows with the size of the terminal associated with
 * "stream", if possible.  Returns true if values were read.
 */
bool readTerminalSize(FILE * stream, /*@null@ */ size_t *columns, /*@null@ */ size_t *rows)
{
#ifdef TIOCGWINSZ
	struct winsize wsz;
#endif

	if (NULL == stream)
		return false;

#ifdef TIOCGWINSZ
	memset(&wsz, 0, sizeof(wsz));

	if (0 != isatty(fileno(stream))) {
		if (0 == ioctl(fileno(stream), TIOCGWINSZ, &wsz)) {
			if (NULL != columns)
				*columns = (size_t) (wsz.ws_col);
			if (NULL != rows)
				*rows = (size_t) (wsz.ws_row);
			return true;
		}
	}
#endif
	return false;
}


/*
 * Display parameter definitions, word wrapping them as appropriate.
 */
void showParameterDefinitions(FILE * stream, size_t columns, struct parameterDefinition *parameterDefinitions)
{
	unsigned int parameterIndex;
	size_t widestParameterWidth = 0;
	size_t descriptionLeftMargin = 0;
	size_t maxDescriptionWidth = 40;
	size_t rightMargin = 77;

	if (columns > 5) {
		rightMargin = columns - 3;
	}

	/*
	 * Translate the help text, and calculate the displayed width of
	 * each part of each parameter definition.  The total display width
	 * of the short option / action name, long option, and parameter
	 * argument together form the "parameterWidth" - we look for the
	 * widest one to calculate the left margin for all of the
	 * descriptions to start at.
	 */
	for (parameterIndex = 0; NULL != parameterDefinitions[parameterIndex].shortOption; parameterIndex++) {
		struct parameterDefinition *definition;
		size_t parameterWidth;

		definition = &(parameterDefinitions[parameterIndex]);
		parameterWidth = 0;

		definition->width.shortOption = calculateDisplayedWidth(definition->shortOption);
		definition->width.longOption = 0;
		definition->width.parameterArgument = 0;
		definition->width.description = 0;

		if (NULL != definition->longOption) {
			definition->width.longOption = calculateDisplayedWidth(definition->longOption);
		}

		if (NULL != definition->parameterArgument) {
			/*@observer@ */ const char *translated;
			translated = _(definition->parameterArgument);
			if (NULL != translated) {
				definition->parameterArgument = translated;
			}
			definition->width.parameterArgument = calculateDisplayedWidth(definition->parameterArgument);
		}

		if (NULL != definition->description) {
			/*@observer@ */ const char *translated;
			translated = _(definition->description);
			if (NULL != translated) {
				definition->description = translated;
			}
			definition->width.description = calculateDisplayedWidth(definition->description);
		}

		/*
		 * The parameterWidth is padded with a left margin of 2
		 * spaces, a ", " between the short and long options, a
		 * space between the long option and the argument, and two
		 * spaces after the argument:
		 *
		 * "  <short>, <long> <arg>  <description>"
		 *
		 * If we don't have getopt_long() then ", <long>" is omitted.
		 */
		parameterWidth += 2 + definition->width.shortOption;	/* "  short" */
#ifdef HAVE_GETOPT_LONG
		if (definition->width.longOption > 0)
			parameterWidth += 2 + definition->width.longOption;	/* ", <long>" */
#endif
		parameterWidth += 1 + definition->width.parameterArgument;	/* " ARG" */
		parameterWidth += 2;	    /* final 2 spaces */

		if (parameterWidth > widestParameterWidth) {
			widestParameterWidth = parameterWidth;
		}
	}

	/*
	 * Set the left margin for the parameter descriptions, based on the
	 * widest parameter width, or (rightMargin - maxDescriptionWidth),
	 * whichever is less, so that there is always room for
	 * "maxDescriptionWidth" characters of description.
	 */
	descriptionLeftMargin = rightMargin - maxDescriptionWidth;
	if (widestParameterWidth < descriptionLeftMargin) {
		descriptionLeftMargin = widestParameterWidth;
	}

	debug("%s: descriptionLeftMargin=%d, widestParameterWidth=%d, rightMargin=%d", "help display",
	      (int) descriptionLeftMargin, (int) widestParameterWidth, (int) rightMargin);

	/*
	 * Display each of the parameter definitions, word wrapping the
	 * descriptions.
	 */
	for (parameterIndex = 0; NULL != parameterDefinitions[parameterIndex].shortOption; parameterIndex++) {
		struct parameterDefinition *definition;
		size_t parameterWidth;

		definition = &(parameterDefinitions[parameterIndex]);
		parameterWidth = 0;

		if (definition->width.shortOption > 0 && NULL != definition->shortOption) {
			fprintf(stream, "  %s", definition->shortOption);
			parameterWidth += 2 + definition->width.shortOption;
		}
#ifdef HAVE_GETOPT_LONG
		if (definition->width.longOption > 0 && NULL != definition->longOption) {
			fprintf(stream, ", %s", definition->longOption);
			parameterWidth += 2 + definition->width.longOption;
		}
#endif
		if (definition->width.parameterArgument > 0 && NULL != definition->parameterArgument) {
			fprintf(stream, " %s", definition->parameterArgument);
			parameterWidth += 1 + definition->width.parameterArgument;
		}

		/* Just start a new line if there's no description. */
		if ((0 == definition->width.description) || (NULL == definition->description)) {
			fprintf(stream, "\n");
			continue;
		}

		/*
		 * If the parameter is too wide, start a new line for the
		 * description.  In both cases, pad with spaces up to the
		 * description left margin.
		 */
		if (parameterWidth >= descriptionLeftMargin) {
			fprintf(stream, "\n%*s", (int) descriptionLeftMargin, "");
		} else if (parameterWidth < descriptionLeftMargin) {
			fprintf(stream, "%*s", (int) (descriptionLeftMargin - parameterWidth), "");
		}

		/* Output the description, word wrapped. */
		outputWordWrap(stream, definition->description, rightMargin - descriptionLeftMargin,
			       descriptionLeftMargin);

		fprintf(stream, "\n");
	}
}
