#!/bin/sh
#
# Functions for generating the package information file.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Generate package information for the package file $1, with original
# filename $2, writing the information to file $3.  Uses ${workDir} to pass
# files into and out of the container.
#
updatePackageInfo () {

	if ! test "${action}" = 'inside-container'; then
		cp "$1" "${workDir}/package" || return "${RC_LOCAL_FAULT}"
	fi

	# In container mode, do the rest inside a container instead.
	if ! ${nativeOnly}; then
		if ! test "${action}" = 'inside-container'; then
			case "$2" in
			*.rpm) setTargetOs 'almalinux9' || return $? ;;
			*.deb) setTargetOs 'debian12' || return $? ;;
			*)
				reportError "$2: unknown package type"
				return "${RC_NOT_SUPPORTED}"
				;;
			esac
			loadComponent 'containers'
			runInContainer 'containers update-info' 'update-info' '/mnt/package' "$2" "/mnt/information" || return $?
			# Copy the generated information to the destination.
			cp "${workDir}/information" "$3" || return "${RC_LOCAL_FAULT}"
			return 0
		fi
	fi

	case "$2" in
	*.rpm)
		if ! command -v rpm >/dev/null 2>&1; then
			reportError "$2: rpm: command not available"
			return "${RC_NOT_SUPPORTED}"
		fi
		{
		printf '%s: %s\n' 'OS' "${targetOs}"
		# shellcheck disable=SC2016
		rpm --query \
		  --queryformat='Name: %{NAME}\nVersion: %{VERSION}-%{RELEASE}\nArchitecture: %{ARCH}\nSummary: %{SUMMARY}\n %{DESCRIPTION}\n' \
		  --package "$1"
		} \
		| sed '/^ /,$s/^/ /;s/^  / /' \
		> "$3" \
		|| return "${RC_LOCAL_FAULT}"
		;;
	*.deb)
		if ! command -v dpkg-deb >/dev/null 2>&1; then
			reportError "$2: dpkg-deb: command not available"
			return "${RC_NOT_SUPPORTED}"
		fi
		{
		printf '%s: %s\n' 'OS' "${targetOs}"
		# shellcheck disable=SC2016
		dpkg-deb --show \
		  --showformat='Name: ${Package}\nVersion: ${Version}\nArchitecture: ${Architecture}\nSummary: ${Description}\n' \
		  "$1"
		} > "$3" \
		|| return "${RC_LOCAL_FAULT}"
		;;
	*)
		reportError "$2: unknown package type"
		return "${RC_NOT_SUPPORTED}"
		;;
	esac

	return 0
}
