#!/bin/sh
#
# Functions for selecting packages in one or more repositories.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Record activity in a repository to the log file in directory $1, writing
# the message $2 preceded by a timestamp and the current username.
#
logRepoActivity () {
	printf '[%s] (%s) %s\n' "$(date '+%Y-%m-%d %H:%M:%S')" "${logUser}" "$2" >> "$1/activity.log"
}


# Regenerate the "contents.txt" index file for the target operating system
# $2, repository $1, under the repository collection directory
# ${destinationPath}.
#
regenerateRepoContentsFile () {
	oldConstraints="${constraints}"
	constraints="$(printf '%s %s\n%s %s\n%s %s\n' 'target' "$2" 'repository' "$1" 'included' 'true')"
	oldSourcePath="${sourcePath}"
	sourcePath=''

	listRepositoryCollectionContents > "${destinationPath}/$1/$2/contents.txt.new"
	mv -f "${destinationPath}/$1/$2/contents.txt.new" "${destinationPath}/$1/$2/contents.txt"

	constraints="${oldConstraints}"
	sourcePath="${oldSourcePath}"
}


# Select version $2 of package $1 in the applicable repositories under
# ${destinationPath}, taking each one from the relevant target operating
# system subdirectory of the package archive ${sourcePath}.
#
# If the version, $2, is "none" or "exclude", remove the package from the
# repositories.
#
# If $3, the pin mode, is "true", override any pins with the new pin note
# $4, which should be empty or "-" to remove the pin.
#
selectPackage () {
	usePackage="$1"
	useVersion="$2"
	pinMode="$3"
	newPinNote="$4"

	test "${useVersion}" = 'none' && useVersion=''
	test "${useVersion}" = 'exclude' && useVersion=''
	test "${useVersion}" = '-' && useVersion=''
	test "${newPinNote}" = '-' && newPinNote=''

	# We need listArchiveContents().
	loadComponent 'archive-contents' || return $?
	# We also need listRepositoryCollectionContents().
	loadComponent 'repository-contents' || return $?

	# Write lists to the temporary working directory, so that when we
	# iterate over directories below we can do it in the same process
	# rather than reading from a pipe in a subprocess.

	printf '%s\n' "${constraints}" | awk '$1=="repository" {print $2}' > "${workDir}/constrainToRepositories"
	printf '%s\n' "${constraints}" | awk '$1=="target" {print $2}' > "${workDir}/constrainToTargets"

	find "${destinationPath}" -mindepth 1 -maxdepth 1 -type d -printf '%f\n' | {
		if test -s "${workDir}/constrainToRepositories"; then
			grep -Fxf "${workDir}/constrainToRepositories"
		else
			cat
		fi
	} > "${workDir}/selectRepositories"

	returnCode='0'

	# Note that we use variable names "selectRepository" and
	# "selectRepoOs" even though they are a bit cumbersome, because the
	# repository collection index functions use "repository" and
	# "repoOs", and Bourne shell doesn't have local variables, so bad
	# things happen if we call those functions when using the same
	# variable names.

	{
	while read -r selectRepository; do
		test -n "${selectRepository}" || continue
		find "${destinationPath}/${selectRepository}" -mindepth 1 -maxdepth 1 -type d -printf '%f\n' | {
			if test -s "${workDir}/constrainToTargets"; then
				grep -Fxf "${workDir}/constrainToTargets"
			else
				cat
			fi
		} > "${workDir}/selectTargets"
		{
		while read -r selectRepoOs; do
			test -n "${selectRepoOs}" || continue
			repoPath="${destinationPath}/${selectRepository}/${selectRepoOs}"

			# Acquire a concurrency lock.
			exec 8>>"${repoPath}/.lock"
			if ! flock -x 8; then
				reportError "${action}: ${usePackage}: ${selectRepository}/${selectRepoOs}: failed to acquire concurrency lock"
				returnCode="${RC_LOCAL_FAULT}"
				continue
			fi

			currentlySelectedFile=''
			currentlySelectedVersion=''
			currentPinNote=''

			if test -s "${repoPath}/contents.txt"; then
				currentlySelectedFile="$(
				  awk -F "\t" -v "package=${usePackage}" '$2==package { print $1 }' < "${repoPath}/contents.txt" \
				  | sed -n 1p
				)"
				currentlySelectedVersion="$(
				  awk -F "\t" -v "package=${usePackage}" '$2==package { print $3 }' < "${repoPath}/contents.txt" \
				  | sed -n 1p
				)"
				test -s "${repoPath}/${currentlySelectedFile}.pinned" \
				&& currentPinNote="$(sed -n 1p < "${repoPath}/${currentlySelectedFile}.pinned")"
			fi

			# We have to try to find the archive file for the
			# selected version, so that if the version was
			# inexactly specified (without the release part
			# "-*"), we can determine the real full version.

			useVersionForThisTarget="${useVersion}"

			if test -n "${useVersion}"; then
				# List the contents of the package archive
				# for this target operating system, if we
				# haven't already done so.
				if ! test -e "${workDir}/${selectRepoOs}-archivePackageInfo"; then
					( constraints="target ${selectRepoOs}"; listArchiveContents ) <&7 7<&- 8<&- \
					> "${workDir}/${selectRepoOs}-archivePackageInfo"
				fi

				# Find the filename of the selected version.
				newPackageFile="$(
				  awk -F "\t" \
				    -v "package=${usePackage}" \
				    -v "version=${useVersion}" \
				    '$2==package && $3==version { print $1 }' \
				    < "${workDir}/${selectRepoOs}-archivePackageInfo" \
				  | sed -n 1p
				)"

				# If the exact version isn't found, and the
				# specified version has no release part
				# ("-*" suffix), ignore the release part of
				# the archive package versions and use the
				# most recent according to "sort -V".
				if test -z "${newPackageFile}"; then
					newPackageInfo="$(
					  awk -F "\t" \
					    -v "package=${usePackage}" \
					    -v "version=${useVersion}" \
					    '$2==package { x=$3; sub(/-.*$/,"",x); if (x==version) print }' \
					    < "${workDir}/${selectRepoOs}-archivePackageInfo" \
					  | sort -t "$(printf "\t")" -k 2rV \
					  | sed -n 1p
					)"
					if test -n "${newPackageInfo}"; then
						newPackageFile="$(
						  printf '%s\n' "${newPackageInfo}" \
						  | awk -F "\t" '{ print $1 }'
						)"
						useVersionForThisTarget="$(
						  printf '%s\n' "${newPackageInfo}" \
						  | awk -F "\t" '{ print $3 }'
						)"
					fi
				fi
			fi

			# From here on, we nust use $useVersionForThisTarget
			# instead of $useVersion, since the former is the
			# real full version number for this target, based on
			# the requested version in $useVersion.

			# If there is no change to the version, adjust the
			# pin note if applicable, and skip further
			# processing.
			if test "${currentlySelectedVersion}" = "${useVersionForThisTarget}"; then
				if test "${pinMode}" = 'true' && ! test "${currentPinNote}" = "${newPinNote}"; then
					if test -n "${newPinNote}" && test -n "${useVersionForThisTarget}"; then
						printf '%s\n%s\n' "${newPinNote}" "${logUser}" > "${repoPath}/${currentlySelectedFile}.pinned.new"
						mv -f "${repoPath}/${currentlySelectedFile}.pinned.new" "${repoPath}/${currentlySelectedFile}.pinned"
						logRepoActivity "${repoPath}" "${usePackage}: pin added (version ${currentlySelectedVersion})"
					else
						rm "${repoPath}/${currentlySelectedFile}.pinned"
						logRepoActivity "${repoPath}" "${usePackage}: pin removed"
					fi
				fi

				regenerateRepoContentsFile "${selectRepository}" "${selectRepoOs}" <&7 7<&- 8<&-

				exec 8<&-
				continue
			fi

			# If not in pin-update mode, and there is a pin in
			# place, refuse to do anything.
			if test -n "${currentPinNote}" && ! test "${pinMode}" = 'true'; then
				reportError "${action}: ${usePackage}: ${selectRepository}/${selectRepoOs}: change prevented by pin: ${currentPinNote}"
				test "${returnCode}" -eq 0 && returnCode="${RC_BAD_ARGS}"
				exec 8<&-
				continue
			fi

			if test -z "${useVersionForThisTarget}"; then
				# No version selected: remove the previously
				# selected package.
				rm -f "${repoPath}/${currentlySelectedFile}" \
				  "${repoPath}/${currentlySelectedFile}.pinned" \
				  "${repoPath}/${currentlySelectedFile}.txt"
				logRepoActivity "${repoPath}" "${usePackage}: package removed (version ${currentlySelectedVersion})"
			else
				# New version selected: find it, copy it in,
				# and if there was a version selected
				# previously, remove that.

				if test -z "${newPackageFile}"; then
					# Report an error if the requested
					# version was not found.
					reportError "${action}: ${usePackage}: ${selectRepository}/${selectRepoOs}: ${useVersionForThisTarget}: version not found in archive"
					test "${returnCode}" -eq 0 && returnCode="${RC_BAD_ARGS}"
					exec 8<&-
					continue
				elif ! cp "${sourcePath}/${selectRepoOs}/${newPackageFile}" "${repoPath}/${newPackageFile}"; then
					# Check the package copy worked.
					reportError "${action}: ${usePackage}: ${selectRepository}/${selectRepoOs}: ${useVersionForThisTarget}: failed to copy ${newPackageFile}"
					test "${returnCode}" -eq 0 && returnCode="${RC_LOCAL_FAULT}"
					exec 8<&-
					continue
				elif ! cp "${sourcePath}/${selectRepoOs}/${newPackageFile}.txt" "${repoPath}/${newPackageFile}.txt"; then
					# Check the package info copy worked.
					reportError "${action}: ${usePackage}: ${selectRepository}/${selectRepoOs}: ${useVersionForThisTarget}: failed to copy ${newPackageFile}.txt"
					rm -f "${sourcePath}/${selectRepoOs}/${newPackageFile}"
					test "${returnCode}" -eq 0 && returnCode="${RC_LOCAL_FAULT}"
					exec 8<&-
					continue
				fi

				# Record the change.
				logRepoActivity "${repoPath}" "${usePackage}: selected version ${useVersionForThisTarget}"

				# Remove the previously selected version
				# from the repository.
				if test -n "${currentlySelectedFile}"; then
					rm -f "${repoPath}/${currentlySelectedFile}" \
					  "${repoPath}/${currentlySelectedFile}.pinned" \
					  "${repoPath}/${currentlySelectedFile}.txt"
				fi

				# Add a pin note if appropriate.
				if test "${pinMode}" = 'true' && test -n "${newPinNote}"; then
					printf '%s\n%s\n' "${newPinNote}" "${logUser}" > "${repoPath}/${newPackageFile}.pinned.new"
					mv -f "${repoPath}/${newPackageFile}.pinned.new" "${repoPath}/${newPackageFile}.pinned"
					logRepoActivity "${repoPath}" "${usePackage}: pin added (version ${useVersionForThisTarget})"
				fi
			fi

			# Re-index the repository.

			canIndex='1'
			if "${nativeOnly}"; then
				# Check that the prerequisite commands are available.
				if ! possiblySilenceOutput indexRepository "${selectRepoOs}" "${repoPath}" "${repoPath}" check-prerequisites <&7 7<&- 8<&-; then
					test "${returnCode}" -eq 0 && returnCode="${RC_LOCAL_FAULT}"
					canIndex='0'
				fi
			fi
			if test "${canIndex}" -eq 1; then
				possiblySilenceOutput indexRepository "${selectRepoOs}" "${repoPath}" "${repoPath}" <&7 7<&- 8<&- || returnCode="$?"
			fi

			regenerateRepoContentsFile "${selectRepository}" "${selectRepoOs}" <&7 7<&- 8<&-

			exec 8<&-
		done
		} < "${workDir}/selectTargets"
	done
	} 7<&0 < "${workDir}/selectRepositories"

	return "${returnCode}"
}
