#!/bin/sh
#
# Functions for listing the contents of a repository collection.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Scan the packages under the repository collection directory
# ${destinationPath}.
#
# If constraints are supplied in ${constraints}, use those to filter the
# list of packages, using the package archive directory ${sourcePath} to
# check "included" or "outdated" constraints.
#
# Write one line of text per package to stdout, containing tab-separated
# fields: filename, package name, version, architecture, repository, target
# operating system, pin note, and summary.
#
# If a package for a target operating system is not included in a repository
# but is available in the archive, it will be shown with a version of "-",
# or it will be omitted if the constraint "included=true" is active.
#
listRepositoryCollectionContents () {
	constrainToPinned="$(printf '%s\n' "${constraints}" | awk '$1=="pinned" {print $2}' | sed -n '$p')"
	constrainToIncluded="$(printf '%s\n' "${constraints}" | awk '$1=="included" {print $2}' | sed -n '$p')"
	constrainToOutdated="$(printf '%s\n' "${constraints}" | awk '$1=="outdated" {print $2}' | sed -n '$p')"

	# Package constraints are applied by listArchiveContents() and
	# packageDirectoryContents(), so we don't need to know them here.
	# constrainToPackages="$(printf '%s\n' "${constraints}" | awk '$1=="package" {print $2}')"

	# We need listRepositoryCollectionContents().
	if ! type 'listRepositoryCollectionContents' >/dev/null 2>&1; then
		loadComponent 'archive-contents' || return $?
	fi

	# Write lists to the temporary working directory, so that when we
	# iterate over directories below we can do it in the same process
	# rather than reading from a pipe in a subprocess.

	printf '%s\n' "${constraints}" | awk '$1=="repository" {print $2}' > "${workDir}/constrainToRepositories"
	printf '%s\n' "${constraints}" | awk '$1=="target" {print $2}' > "${workDir}/constrainToTargets"

	find "${destinationPath}" -mindepth 1 -maxdepth 1 -type d -printf '%f\n' | {
		if test -s "${workDir}/constrainToRepositories"; then
			grep -Fxf "${workDir}/constrainToRepositories"
		else
			cat
		fi
	} > "${workDir}/repositories"

	{
	while read -r repository; do
		test -n "${repository}" || continue
		find "${destinationPath}/${repository}" -mindepth 1 -maxdepth 1 -type d -printf '%f\n' | {
			if test -s "${workDir}/constrainToTargets"; then
				grep -Fxf "${workDir}/constrainToTargets"
			else
				cat
			fi
		} > "${workDir}/targets"
		{
		while read -r repoOs; do
			test -n "${repoOs}" || continue
			repoPath="${destinationPath}/${repository}/${repoOs}"

			# List the information about the available packages
			# in the archive for this target operating system. 
			# Package constraints are applied.  The lookup is
			# cached, so we look up the archive info for a
			# target OS only once, not once for every
			# repository.
			archivePackageInfoDir="${workDir}/${repoOs}-archivePackageInfoFiles"
			if ! test -e "${workDir}/${repoOs}-archivePackageInfo"; then
				{
				oldConstraints="${constraints}"
				constraints="$(
				printf '%s\n' "${constraints}" \
				| awk '$1!="target"{print}'; \
				printf '%s %s\n' 'target' "${repoOs}"
				)"

				# NB we remove the first line because
				# listArchiveContents() produces a header,
				# which we don't want.
				test -n "${sourcePath}" && listArchiveContents | sed 1d

				constraints="${oldConstraints}"
				} > "${workDir}/${repoOs}-archivePackageInfo"

				# Split the information into one file per package.
				rm -rf "${archivePackageInfoDir}"
				mkdir -p "${archivePackageInfoDir}"
				awk -F "\t" \
				  -v "archivePackageInfoDir=${archivePackageInfoDir}" \
				  '
				  $2~/^[0-9A-Za-z_-]+$/ { packageInfoFile=sprintf("%s/%s",archivePackageInfoDir,$2); print >> packageInfoFile }
				  ' \
				"${workDir}/${repoOs}-archivePackageInfo"

				# Also list just the package names.
				awk -F "\t" '{print $2}' "${workDir}/${repoOs}-archivePackageInfo" \
				| sort -u \
				| grep -E . \
				> "${workDir}/${repoOs}-archivePackageList"
			fi

			# List the information about the packages for this
			# target operating system, in this repository
			# collection.  Package constraints are applied.
			packageDirectoryContents "${repository}" "${repoOs}" "${repoPath}" \
			> "${workDir}/${repository}-${repoOs}-repoPackageInfo"

			# Split the information into one file per package.
			repoPackageInfoDir="${workDir}/${repository}-${repoOs}-repoPackageInfoFiles"
			rm -rf "${repoPackageInfoDir}"
			mkdir -p "${repoPackageInfoDir}"
			awk -F "\t" \
			  -v "repoPackageInfoDir=${repoPackageInfoDir}" \
			  '
			  $2~/^[0-9A-Za-z_-]+$/ { packageInfoFile=sprintf("%s/%s",repoPackageInfoDir,$2); print >> packageInfoFile }
			  ' \
			"${workDir}/${repository}-${repoOs}-repoPackageInfo"

			# Also list just the package names.
			awk -F "\t" '{print $2}' "${workDir}/${repository}-${repoOs}-repoPackageInfo" \
			| sort -u \
			| grep -E . \
			> "${workDir}/${repository}-${repoOs}-repoPackageList"

			# Determine which packages we are going to consider
			# for inclusion in the output.
			if test "${constrainToIncluded}" = "true"; then
				cp "${workDir}/${repository}-${repoOs}-repoPackageList" "${workDir}/${repository}-${repoOs}-toConsider"
			elif test "${constrainToIncluded}" = "false"; then
				comm -23 \
				  "${workDir}/${repoOs}-archivePackageList" \
				  "${workDir}/${repository}-${repoOs}-repoPackageList" \
				  > "${workDir}/${repository}-${repoOs}-toConsider"
			else
				cat \
				  "${workDir}/${repoOs}-archivePackageList" \
				  "${workDir}/${repository}-${repoOs}-repoPackageList" \
				  | sort -u \
				  | grep -E . \
				  > "${workDir}/${repository}-${repoOs}-toConsider"
			fi

			# Output the relevant information for each package
			# being considered, applying the "pinned" and
			# "outdated" constraints.
			awk -F "\t" \
			  -v "archivePackageInfoDir=${archivePackageInfoDir}" \
			  -v "repoPackageInfoDir=${repoPackageInfoDir}" \
			  -v "repoPath=${repoPath}" \
			  -v "repository=${repository}" \
			  -v "constrainToPinned=${constrainToPinned}" \
			  -v "constrainToOutdated=${constrainToOutdated}" \
			  '
			  /^[0-9A-Za-z_-]+$/ {
			    package = $1
			    repoInfoFile = sprintf("%s/%s", repoPackageInfoDir, package)
			    repoInfoLine = ""
			    getline repoInfoLine < repoInfoFile
			    close(repoInfoFile)
			    split(repoInfoLine, repoInfoArray)

			    pinFile = ""
			    pinNote = ""
			    if (repoInfoLine != "" && repoInfoArray[1] ~ /^[0-9A-Za-z_][0-9A-Za-z_.-]+$/) {
			      pinFile = sprintf("%s/%s.pinned", repoPath, repoInfoArray[1])
			      getline pinNote < pinFile
			      close(pinFile)
			    }

			    # Apply the "pinned" constraint.
			    if (constrainToPinned == "true" && pinNote == "")
			      next
			    if (constrainToPinned == "false" && pinNote != "")
			      next

			    gsub(/\t/, " ", pinNote)
			    if (pinNote == "")
			      pinNote = "-"

			    archiveInfoFile = sprintf("%s/%s", archivePackageInfoDir, package)

			    # Apply the "outdated" constraint.
			    if (constrainToOutdated != "") {
			      # Can only check versions if package is in the repo.
			      if (repoInfoLine == "")
			        next

			      componentCount = split(repoInfoArray[3], versionComponents, /\./)
			      repoCompareVersion = ""
			      for (componentIndex = 1; componentIndex <= componentCount; componentIndex++) {
			        repoCompareVersion = sprintf("%s%20s", repoCompareVersion, versionComponents[componentIndex])
			      }

			      isOutdated = 0

			      archiveInfoLine = ""
			      while (getline archiveInfoLine < archiveInfoFile == 1) {
			        split(archiveInfoLine, archiveInfoArray)
			        componentCount = split(archiveInfoArray[3], versionComponents, /\./)
			        archiveCompareVersion = ""
			        for (componentIndex = 1; componentIndex <= componentCount; componentIndex++) {
			          archiveCompareVersion = sprintf("%s%20s", archiveCompareVersion, versionComponents[componentIndex])
			        }
			        if (archiveCompareVersion > repoCompareVersion) {
			          isOutdated = 1
			        }
			      }
			      close(archiveInfoFile)

			      if (constrainToOutdated == "true" && isOutdated == 0)
			        next
			      if (constrainToOutdated == "false" && isOutdated == 1)
			        next
			    }

			    # Output the information.
			    if (repoInfoLine != "") {
			      printf "%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
			        repoInfoArray[1],
			        repoInfoArray[2],
			        repoInfoArray[3],
			        repoInfoArray[4],
			        repoInfoArray[5],
			        repoInfoArray[6],
			        pinNote,
			        repoInfoArray[7]
			    } else {
			      archiveInfoLine = ""
			      getline archiveInfoLine < archiveInfoFile
			      close(archiveInfoFile)
			      split(archiveInfoLine, archiveInfoArray)
			      printf "%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
			        archiveInfoArray[1],
			        archiveInfoArray[2],
			        "-",
			        archiveInfoArray[4],
			        repository,
			        archiveInfoArray[5],
			        "-",
			        archiveInfoArray[6]
			    }
			  }
			  ' "${workDir}/${repository}-${repoOs}-toConsider"
		done
		} < "${workDir}/targets"
	done
	} < "${workDir}/repositories" > "${workDir}/packageList"

	if test -s "${workDir}/packageList"; then
		printf '%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n' 'Filename' 'Package' 'Version' 'Architecture' 'Repository' 'OS' 'Pin' 'Summary'
		cat "${workDir}/packageList"
	fi

	return 0
}
