#!/bin/sh
#
# Functions for handling containers.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

havePodman=''
haveDocker=''
#
# Run a docker / podman command.
#
# Use whichever of docker or podman is available.  If both are available,
# use docker, unless $PARMO_CONTAINER_COMMAND is 'podman'.
#
dockerCommand () {
	if test -z "${havePodman}"; then
		havePodman='false'
		command -v podman >/dev/null 2>&1 && havePodman='true'
	fi
	if test -z "${haveDocker}"; then
		haveDocker='false'
		command -v docker >/dev/null 2>&1 && haveDocker='true'
	fi
	if test "${havePodman}" = 'true' && test "${haveDocker}" = 'true'; then
		if test "${PARMO_CONTAINER_COMMAND}" = 'podman'; then
			podman "$@"
			return $?
		else
			docker "$@"
			return $?
		fi
	elif test "${havePodman}" = 'true'; then
		podman "$@"
		return $?
	elif test "${haveDocker}" = 'true'; then
		docker "$@"
		return $?
	else
		reportError "containers are not available"
		return "${RC_NOT_SUPPORTED}"
	fi
}

# Update the currently selected build image by running $* inside it.
#
updateBuildImage () {
	oldId="$(dockerCommand image inspect "${buildImageName}:latest" | awk '$1=="\"Id\":"{print $2}' | tr -dc '0-9a-f')"
	interimName="$(uuidgen 2>/dev/null || dbus-uuidgen 2>/dev/null || od -N 16 -An -t x1 < /dev/urandom | tr -d ' ')"

	rmiFlag=''
	dockerCommand run --help 2>&1 | grep -Fq -- --rmi && rmiFlag='--rmi=false'
	squashFlag=''
	dockerCommand commit --help 2>&1 | grep -Fq -- --squash && squashFlag='--squash=true'

	reportProgress 'updating the build image'

	# shellcheck disable=SC2086
	dockerCommand run --name="${interimName}" ${rmiFlag} "${buildImageName}" \
	  sh -c "$*" \
	  || return "${RC_LOCAL_FAULT}"

	reportProgress 'committing the updated image'

	# shellcheck disable=SC2086
	dockerCommand commit ${squashFlag} "${interimName}" "${buildImageName}:latest" \
	  || return "${RC_LOCAL_FAULT}"

	reportProgress 'removing interim containers and images'

	dockerCommand container rm "${interimName}"

	test -n "${oldId}" && dockerCommand image rm "${oldId}" >/dev/null 2>&1

	return 0
}

# Call the appropriate function to ensure the build image for ${targetOs}
# exists.
#
buildImage () {
	loadOsSpecificComponent 'build-image' || return $?
	"${osSpecificFunction}"
	return $?
}

# Call the appropriate function to apply updates to the build image for
# ${targetOs}.
#
updateImage () {
	buildImage || return $?
	updateImageFunction="update_${osSpecificFunction#build_}"
	"${updateImageFunction}"
	return $?
}

# Call the appropriate function to install packages into the build image for
# ${targetOs}.
#
installIntoImage () {
	buildImage || return $?
	installIntoImageFunction="install_into_${osSpecificFunction#build_}"
	"${installIntoImageFunction}" "$@"
	return $?
}

# Run this script inside a container based on ${buildImageName}, sharing
# ${workDir} as "/mnt" inside the container.  Copies this script, including
# the components listed in $1 (a space-separated list for locateComponent to
# process), to "/mnt/run" inside the container (${workDir}/run outside) and
# runs it with the action "inside-container" and the remaining arguments.
#
# If $1 starts with "volume:" it should be of the form
# "volume:OUTSIDE:INSIDE" and describes a --volume mount for the container;
# in that case the list of components is in $2.
#
runInContainer () {
	useComponents="$1"
	shift

	extraVolume=""
	if ! test "${useComponents#volume:}" = "${useComponents}"; then
		extraVolume="${useComponents#volume:}"
		useComponents="$1"
		shift
	fi

	buildImage || return $?

	{
	for componentName in ${useComponents}; do
		componentPath="$(locateComponent "${componentName}")" || return $?
		cat "${componentPath}" || return $?
	done
	scriptPath="$(locateSelf)" || return $?
	cat "${scriptPath}" || return $?
	} > "${workDir}/run"

	interactiveFlags=''
	test -t 0 && interactiveFlags='--interactive --tty'

	if test -n "${extraVolume}"; then
		# shellcheck disable=SC2086
		dockerCommand run \
		  ${interactiveFlags} \
		  --rm=true \
		  --volume="${workDir}:/mnt" \
		  --volume="${extraVolume}" \
		  "${buildImageName}" \
		  sh '/mnt/run' "--target=${targetOs}" 'inside-container' "$@" \
		|| return "${RC_LOCAL_FAULT}"
	else
		# shellcheck disable=SC2086
		dockerCommand run \
		  ${interactiveFlags} \
		  --rm=true \
		  --volume="${workDir}:/mnt" \
		  "${buildImageName}" \
		  sh '/mnt/run' "--target=${targetOs}" 'inside-container' "$@" \
		|| return "${RC_LOCAL_FAULT}"
	fi

	return 0
}

# Run the appropriate functions according to the parameters, from inside a
# container, as called by runInContainer.
#
runFromInsideContainer () {

	# Set a permissive umask so that the host side can remove files and
	# directories and so on.
	umask 000

	realAction="$1"
	shift

	workDir='/mnt'

	rc="${RC_NOT_SUPPORTED}"
	case "${realAction}" in
	'build-instructions') generateBuildInstructions "$@"; rc=$? ;;
	'build-package') runFunctionWithGenericFallback "build_package_${targetOsFunctionName}" "$@"; rc=$? ;;
	'update-info') updatePackageInfo "$@"; rc=$? ;;
	'index-repository') runFunctionWithGenericFallback "index_repository_${targetOsFunctionName}" "$@"; rc=$? ;;
	esac

	return "${rc}"
}
