#!/bin/sh
#
# Functions for maintaining a container image for building CentOS 7
# packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Ensure that the container image "${buildImageName}" exists.
#
build_image_centos7 () {
	# Do nothing if the image already exists.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	# Write the build instructions.
	{
	printf '%s\n' 'FROM centos:7'
	printf '%s\n' 'ADD vault.repo /root/vault.repo'
	printf '%s\n' 'ADD packageList /tmp/packageList'
	# shellcheck disable=SC2016
	printf '%s %s %s %s %s %s %s %s %s\n' \
	  'RUN rm -f /etc/yum.repos.d/*' \
	  '&& cp /root/vault.repo /etc/yum.repos.d/CentOS-Vault.repo' \
	  '&& yum -y update' \
	  '&& rm -f /etc/yum.repos.d/*' \
	  '&& cp /root/vault.repo /etc/yum.repos.d/CentOS-Vault.repo' \
	  '&& yum -y install $(cat /tmp/packageList)' \
	  '&& yum clean all' \
	  '&& rm -f /tmp/packageList /etc/yum.repos.d/*' \
	  '&& cp /root/vault.repo /etc/yum.repos.d/CentOS-Vault.repo'
	} > "${workDir}/Dockerfile"

	# Latest CentOS 7 vault repository.
	cat > "${workDir}/vault.repo" <<REPOFILE
[C7.9.2009-base]
name=CentOS-7.9.2009 - Base
baseurl=http://vault.centos.org/7.9.2009/os/\$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
enabled=1

[C7.9.2009-updates]
name=CentOS-7.9.2009 - Updates
baseurl=http://vault.centos.org/7.9.2009/updates/\$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
enabled=1

[C7.9.2009-extras]
name=CentOS-7.9.2009 - Extras
baseurl=http://vault.centos.org/7.9.2009/extras/\$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
enabled=1

[C7.9.2009-centosplus]
name=CentOS-7.9.2009 - CentOSPlus
baseurl=http://vault.centos.org/7.9.2009/centosplus/\$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
enabled=1

[C7.9.2009-fasttrack]
name=CentOS-7.9.2009 - Fasttrack
baseurl=http://vault.centos.org/7.9.2009/fasttrack/\$basearch/
gpgcheck=1
gpgkey=file:///etc/pki/rpm-gpg/RPM-GPG-KEY-CentOS-7
enabled=1
REPOFILE

	# Package list.
	{
	# Build essentials (includes automake, gcc, rpm-build, rpm-sign, etc):
	printf '%s\n' '@development'
	# Needed for maintaining repositories:
	printf '%s\n' 'createrepo'
	# Needed if signing required:
	printf '%s\n' 'pinentry'
	# Commonly needed packages:
	printf '%s\n' 'cmake pkgconfig java-openjdk'
	# Various libraries:
	cat <<LIST
bzip2-devel
cups-devel
desktop-file-utils
expat-devel
expect
freetype-devel
gd-devel
glib2-devel
gmp-devel
gnutls-devel
gtk2-devel
help2man
json-c-devel
libevent-devel
libgpg-error-devel
libjpeg-devel
libnet-devel
libpcap-devel
libpng-devel
lzo-devel
mysql-devel
ncurses-devel
net-snmp-devel
openldap-devel
openssl-devel
pam-devel
perl(Class::ISA)
perl(DateTime)
perl(DateTime::TimeZone)
perl-ExtUtils-Embed
perl(ExtUtils::MakeMaker)
perl(File::Find::Rule)
perl(GD)
perl(inc::Module::Install)
perl(IO::Stringy)
perl(LWP::UserAgent)
perl-MailTools
perl(Params::Validate)
perl(Test::Differences)
perl(Test::Manifest)
perl(Test::More)
perl(Test::Pod)
perl(Test::Pod::Coverage)
readline-devel
tcl
tcl-devel
tk
zlib-devel
LIST
	} > "${workDir}/packageList"

	reportProgress 'building the base image'

	# Build the image.
	(
	cd "${workDir}" || exit "${RC_LOCAL_FAULT}"
	dockerCommand build --tag "${buildImageName}" . || exit "${RC_LOCAL_FAULT}"
	exit 0
	)

	# Check that the image built.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	reportError "${buildImageName}: failed to create base image"
	return "${RC_LOCAL_FAULT}"
}


# Apply updates to the container image "${buildImageName}".
#
update_image_centos7 () {
	build_image_centos7 || return $?
	updateBuildImage 'yum -y update; rm -f /etc/yum.repos.d/*; cp /root/vault.repo /etc/yum.repos.d/CentOS-Vault.repo; yum clean all'
}


# Install packages into the container image "${buildImageName}".
#
install_into_image_centos7 () {
	build_image_centos7 || return $?
	updateBuildImage 'yum -y install '"$(printf '%s\n' "$*" | tr -dc ' \t0-9A-Za-z_.-')"' || exit $?; rm -f /etc/yum.repos.d/*; cp /root/vault.repo /etc/yum.repos.d/CentOS-Vault.repo; yum clean all'
}
