#!/bin/sh
#
# Functions for maintaining a container image for building AlmaLinux
# packages.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Ensure that the container image "${buildImageName}" exists.
#
build_image_almalinux () {
	# Do nothing if the image already exists.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	osVersion="${targetOs#almalinux}"

	# Write the build instructions.
	{
	printf '%s%d\n' 'FROM almalinux:' "${osVersion}"
	printf '%s\n' 'ADD packageList /tmp/packageList'
	# shellcheck disable=SC2016
	printf '%s %s %s %s\n' \
	  'RUN dnf -y update' \
	  '&& dnf -y install $(cat /tmp/packageList)' \
	  '&& dnf clean all' \
	  '&& rm -f /tmp/packageList'
	} > "${workDir}/Dockerfile"

	# Package list.
	{
	# Build essentials (includes automake, gcc, rpm-build, rpm-sign, etc):
	printf '%s\n' '@development'
	# Needed for maintaining repositories:
	printf '%s\n' 'createrepo'
	# Needed if signing required:
	printf '%s\n' 'pinentry'
	# Commonly needed packages:
	printf '%s\n' 'cmake pkgconfig java-openjdk'
	# Various libraries that are available in AlmaLinux 8 and 9:
	cat <<LIST
bzip2-devel
cups-devel
desktop-file-utils
expat-devel
expect
freetype-devel
gd-devel
glib2-devel
gmp-devel
gnutls-devel
gtk2-devel
libevent-devel
libgpg-error-devel
libjpeg-devel
libpng-devel
lzo-devel
ncurses-devel
net-snmp-devel
openldap-devel
openssl-devel
pam-devel
perl-ExtUtils-Embed
perl(ExtUtils::MakeMaker)
perl(LWP::UserAgent)
perl-MailTools
perl(Test::More)
readline-devel
tcl
tcl-devel
tk
zlib-devel
LIST

	# These are only available in AlmaLinux 8:
	if test "${osVersion}" = "8"; then
		printf '%s\n' 'json-c-devel mysql-devel perl(Class::ISA)'
	fi

	} > "${workDir}/packageList"

	reportProgress 'building the base image'

	# Build the image.
	(
	cd "${workDir}" || exit "${RC_LOCAL_FAULT}"
	dockerCommand build --tag "${buildImageName}" . || exit "${RC_LOCAL_FAULT}"
	exit 0
	)

	# Check that the image built.
	dockerCommand image inspect "${buildImageName}" >/dev/null 2>&1 && return 0

	reportError "${buildImageName}: failed to create base image"
	return "${RC_LOCAL_FAULT}"
}


# Apply updates to the container image "${buildImageName}".
#
update_image_almalinux () {
	build_image_almalinux || return $?
	updateBuildImage 'dnf -y update; dnf clean all'
}


# Install packages into the container image "${buildImageName}".
#
install_into_image_almalinux () {
	build_image_almalinux || return $?
	updateBuildImage 'dnf -y install '"$(printf '%s\n' "$*" | tr -dc ' \t0-9A-Za-z_.-')"' || exit $?; dnf clean all'
}
