#!/bin/sh
#
# Functions for listing the contents of a package archive.
#
# Copyright 2025 Andrew Wood
#
# License GPLv3+: GNU GPL version 3 or later; see `docs/COPYING'.
#

# Read a directory $3 for target operating system $2 in repository $1,
# containing package information text files that were generated by the
# "build-package" action, and output one tab-separated line per file:
# package filename, package name, version, architecture, repository, target
# operating system, and summary.
#
# If ${constraints} contains any "package" constraints, these are applied.
#
# TODO: where packages are equal, sort by version.
#
packageDirectoryContents () {
	packageRepo="$1"
	packageOs="$2"
	packageDir="$3"

	constrainDirToPackages="$(printf '%s\n' "${constraints}" | awk '$1=="package" {print $2}')"

	directoryContents="$(
	find "${packageDir}" \
	  -mindepth 1 \
	  -maxdepth 1 \
	  -type f \
	  -name '*.*.txt' \
	  -exec awk \
	    -v "repo=${packageRepo}" \
	    -v "os=${packageOs}" '
	    /^[A-Z]/ {
	      n=$1
	      sub(/:/,"",n)
	      sub(/^[^:]*: /,"")
	      gsub(/\t/," ")
	      v[n]=$0
	    }
	    END {
	      n=FILENAME
	      sub(/^.*\//,"",n)
	      sub(/\.txt$/,"",n)
	      printf "%s\t%s\t%s\t%s\t%s\t%s\t%s\n",n,v["Name"],v["Version"],v["Architecture"],repo,os,v["Summary"]
	    }' \
	    '{}' ';'
	)"

	# Apply package constraints, if any were supplied.
	if test -n "${constrainDirToPackages}" && test -n "${directoryContents}"; then
		directoryContents="$(
		for packageName in ${constrainDirToPackages}; do
			printf '%s\n' "${directoryContents}" | awk -v "package=${packageName}" -F "\t" '$2==package {print}'
		done
		)"
	fi

	test -n "${directoryContents}" && printf '%s\n' "${directoryContents}"
}


# Scan the packages under the archive directory ${sourcePath}, which are
# expected to either be directly under it, or in one subdirectory per target
# operating system.  If constraints are supplied in ${constraints}, use
# those to filter the list of packages.
#
# Write one line of text per package to stdout, containing tab-separated
# fields: filename, package name, version, architecture, target operating
# system, and summary.
#
listArchiveContents () {
	constrainToTargets="$(printf '%s\n' "${constraints}" | awk '$1=="target" {print $2}')"

	find "${sourcePath}" -mindepth 1 -maxdepth 2 -type f -name '.target' | {
	packageList=""
	while read -r targetFile; do
		dirOs="$(cat "${targetFile}")" || continue
		test -n "${dirOs}" || continue

		# If constraining the target, check the constraint is met.
		if test -n "${constrainToTargets}"; then
			printf '%s\n' "${constrainToTargets}" | grep -Fqx "${dirOs}" || continue
		fi

		dirPackages="$(
		packageDirectoryContents '-' "${dirOs}" "${targetFile%/*}" \
		| cut -d "$(printf '\t')" -f 1,2,3,4,6,7
		)"

		if test -n "${dirPackages}"; then
			if test -n "${packageList}"; then
				packageList="$(printf '%s\n%s\n' "${packageList}" "${dirPackages}")"
			else
				packageList="${dirPackages}"
			fi
		fi
	done

	if test -n "${packageList}"; then
		printf '%s\t%s\t%s\t%s\t%s\t%s\n%s\n' 'Filename' 'Package' 'Version' 'Architecture' 'OS' 'Summary' "${packageList}"
	fi
	}

	return 0
}
